function WHTrackHighRes(varargin)
%
% This function takes a spotlist file and does the tracking automatically.
% varargin can contain defaults and spotlist
% Example: WHTrackHighRes('defaults.mat','T77_585_705_4uMATP_100ms_1_spotlist')
%
% spotlist: is the name of the spotlist file that contains a list of spots
% to be tracked as well as header information. This value is required (but
% can be either first or second in the list.

% defaults: is the name of a file that is generated by WHTrackDefaults.
% If not specified, WHTrackHighRes will search in current directory for
% defaults.mat (and die gracefully if it is not there).
% File contains constants whose values depend on a particular experiment these include:%
%
%    'OutputDirectory': location of traces.
%
%    'Method','Gauss2DNLSQ', 'Method','Gauss2DJILA', or 'Method','GaussMask' for full
%    non-linear least squares fitting, linear regression based fitting, and for Gaussian weighted centroid for
%    the latter (default is 'GaussMask').
%
%    'PixelPadding': added to calculated psf to give the area of an image cropped out for tracking.
%
%    'Waitbar',1 or 'Waitbar',0 tells the function to use a waitbar or not.
%    The default is no waitbar and the status output is to the command line.
%
%    'Output','pixels', or 'Output','nm' default is set to nm.
%
%% Store input variables
disp(' ')
disp(' ')
input=varargin;  %varargin gets reset to zero if a try/catch comand fails.  So transfer values to variable

%% Work out if defaults is defined and if it is load it.

clear defaults
try %try loading first input (only .mat files should load properly
    load(input{1})
    defaults;
catch
    try  %check out the second input
        load(input{2})
        defaults;
    catch
        try % loading defaults.mat from current directory
            load('defaults.mat')
        catch
            disp('Failed to load output from WHTrackDefaults: please specify correct ')
            disp('file or change directory to where a ''defaults.mat'' is present')
            return
        end
    end
end

dirpath=defaults.Basic.Directory;
constants.pixelsize=defaults.Basic.PixelSize;
constants.dt=defaults.Basic.TimeInterval;

%Settings for WHTrackHiRes
constants.OutputDirectory=defaults.HiRes.OutputDirectory ;
constants.OutputOverwrite=defaults.HiRes.OutputOverwrite;
constants.Method=defaults.HiRes.Method;
constants.pixelpadding=defaults.HiRes.PixelPadding;
constants.Output=defaults.HiRes.Output;

%Settings for Error model (& PSF calculation) in WHTrackHiRes
constants.wavelength=defaults.Error.Wavelength;
constants.NA=defaults.Error.NA;
constants.photonconversion=defaults.Error.Photonconversion;
constants.EMgain=defaults.Error.EMgain;
constants.Background=defaults.Error.Background;
constants.Bgstd=defaults.Error.Bgstd;

%% Spot files
% If a file is defined that ends with _spotlist.txt, then load it
% otherwise try and load all files in the directory specified in defaults.
% If no such files exist, exit gracefully.

spotlistdefined=0;
A=[];
B=[];
if ~isempty(input);  %if there are inputs, try loading both to look for spotlist
    try
        [A B] = read_spotlist(fullfile(dirpath,input{1}));
        ind = find_fields(A,'Sx','Sy','Start','End','Peak');
        if ~isempty(ind)
            spotlistdefined=1;
        end
    catch
        [A B] = read_spotlist(fullfile(dirpath,input{2}));
        ind = find_fields(A,'Sx','Sy','Start','End','Peak');
        if ~isempty(ind)
            spotlistdefined=2;
        end

    end
end
% Depending on where the spotlist file is, put it (or its) names into files
% variable.
if spotlistdefined==1;
    files.name=input{1};
elseif  spotlistdefined==2;
    files.name=input{2};
else  % try and load all spotlist files in directory specified in defaults.
    files=dir(fullfile(dirpath,'*spotlist.txt'));
    
    if isempty(files)
        disp(strcat('Could not find any files ending in ''spotlist.txt'' in :',dirpath))
        return
    end
end
disp(strcat('Processing following files in :',dirpath))
for i=1:length(files)
    disp([' >' files(i).name])
end
disp(' ')


%Loop through all entries in files
for ct=1:length(files)
    %try
    do_tracking(files(ct),dirpath,constants)
    %catch
    disp([ files(ct).name ' failed, possibly due to an empty spotlist.txt file'])
    disp(' ')
    %end
end


end
%% Read and output
function do_tracking(files,dirpath,constants)
%Put in function command here, loop through it

handles.sx = [];
handles.sy = [];
handles.ex = [];
handles.ey = [];
handles.start = [];
handles.ending = [];
handles.peak = [];
handles.file = [];
handles.flag =[];
c = 1; % counter


[A B] = read_spotlist(fullfile(dirpath,files.name));

% The following are the fields we are interested in:
% Sx, Sy, Start, Peak;
ind = find_fields(A,'Sx','Sy','Start','End','Peak','Flag');
% Check that A contains correct stuff:
if length(ind)<5 || min(ind)==0
   disp('Spotlist file must contain  a header with fields: ')
   disp('''Sx'',''Sy'',''Start'',''End'',''Peak'' and optionally ''Flag''')
   disp('separated by tabs')
   return
end
if length(ind)~=6
   disp('WARNING: Only using fields:')
   disp(A(ind))
   disp('You may want to check your spotlist file')
end

% Get the filename  and image type info from the spotlist header
try
    filename = read_header(fullfile(dirpath,files.name),'FileName');
catch
    disp(strcat(files.name, '   :Check that spotlist file header contains the flag ''FileName'''))
end
% Get the filetype: otherwise warn user:
try
    filetype = read_header(fullfile(dirpath,files.name),'ImageType');
catch
    disp(strcat(files.name, '    :Header does not specify image type, assuming tifstack'))
    filetype='tifstack';
end





%% Now check to see if the spot has been tracked
if strcmp(constants.OutputOverwrite,'No')==1
    for j=1:length(B{1})
        xyname = fullfile(constants.OutputDirectory,[filename '_spot' num2str(B{ind(5)}(j)) '*_xy.txt']);  %added * because now possible to have several files same spot no.
        tempfiles=[];
        tempfiles=dir(xyname);
        a = ~isempty(tempfiles);
        if a == 0
            % then store all the variables in a separate structure
            % for evaluation later.
            handles.sx(c) = B{ind(1)}(j);
            handles.sy(c) = B{ind(2)}(j);
            handles.start(c) = B{ind(3)}(j);
            handles.ending(c) = B{ind(4)}(j);
            handles.peak(c) = B{ind(5)}(j);
            if length(ind)==6  %only pick up the flags if they have been defined.
                handles.flag(c) = B{ind(6)}(j);
            end
            c = c+1;
        else
            disp(['Skipping: ' xyname ' already exists)']);
        end
    end
else
    for j=1:length(B{1})
            % then store all the variables in a separate structure
            % for evaluation later.
            handles.sx(c) = B{ind(1)}(j);
            handles.sy(c) = B{ind(2)}(j);
            handles.start(c) = B{ind(3)}(j);
            handles.ending(c) = B{ind(4)}(j);
            handles.peak(c) = B{ind(5)}(j);
            if length(ind)==6  %only pick up the flags if they have been defined.
                handles.flag(c) = B{ind(6)}(j);
            end
            c = c+1;
    end
            disp (['Will overwrite xy.txt files for ' filename])
end
%Flag must not contain value 0

if min(handles.flag)==0
   disp('Spotlist ''Flag'' field must contain integers greater than 1')
   return
end
%kill function here if there is nothing to trace
if isempty(handles.sx)==1
    disp(' ')
   return 
end
% handles contains all the information we need to track everything.


%% Work out flag names

if length(ind)==5  %if flag heading wasn't found:
    disp('Not using flags in output name')
    flagset=0;
elseif length(ind)==6   %if flags were specified:
    flagnos=unique(handles.flag);
    flagset=1;
    for m=flagnos
        temp=['Flag' num2str(m)];
        try
            
            flagname{m}=read_header(fullfile(dirpath,files.name),temp);
        catch
            flagname{m}=num2str(m);
            
            disp(['Can''t find ' temp ' in spotlist header: using number ' num2str(m) ' in output name'])
            disp(' ')
        end

    end
else
    disp('Error: check line 425')
end





%% figure out the width of the PSF
w = constants.wavelength/(2*constants.NA);
% convert it into pixels
w = round(w/constants.pixelsize);
PSFpix = w;
boxwidth = w + constants.pixelpadding;
a = constants.pixelsize;
% check to see if the box is even or odd
remainder = rem(boxwidth,2);
if remainder == 0
    boxwidth = boxwidth+1;
end

% preload the variables pos_t, sigma_t, photons into NaNs
pos_t = zeros(max(handles.ending),2,length(handles.sx))+NaN;
sigma_t = zeros(max(handles.ending),2,length(handles.sx))+NaN;
photons = zeros(max(handles.ending),length(handles.sx))+NaN;

% display will show every image if there are a lot of spots, less if there
% are fewer.
if length(handles.sx)<10  %less than 10spots and we can skip every 10 frames
    imskip=10;
    disp([filename ' image ' num2str(1) '/' num2str(max(handles.ending)) ' For:' num2str(length(handles.sx)) ' spots'])

elseif length(handles.sx)<100
    imskip=5;
    disp([filename ' image ' num2str(1) '/' num2str(max(handles.ending)) ' For:' num2str(length(handles.sx)) ' spots'])
else
    imskip=1;
end

% loop through images.
for i = min(handles.start):max(handles.ending)

    % display where we are at.
    if mod(i,imskip)== 0
        disp([filename ' image ' num2str(i) '/' num2str(max(handles.ending)) ' For:' num2str(length(handles.sx)) ' spots']);
    end

    % load image
    if  strcmp(filetype,'directory')==1
        input=fullfile(dirpath,filename);
        im=loaddirectoryimage(i,input);
        outname=filename;
    elseif strcmp(filetype,'tifstack')==1
        input=fullfile(dirpath,filename);
        im=loadtiffstack(i,input);
        outname=filename(1:end-4);
    end
%Work out background automatically unless it is defined.
       if i==min(handles.start)  % for first image
           getbgimg = false;
           if isnan(constants.Background)==1  %default is to set constants.Background to NaN
              bgd=mean(min(im));
              
           elseif isnumeric(constants.Background) == 1
             bgd=constants.Background;
           else
             bgdfull = imread(constants.Background);
             getbgimg = true;
           end
       end
    
    % loop through coordinates
    for j = 1:length(handles.sx)
        % display where we are at.

        %
        %         if mod(j,20)== 0
        %             disp(['_spot' num2str(handles.peak(j)) '/' num2str(length(handles.sx))]);
        %         end

        % get the coordinates to look around if for the current j, we are
        % in the range of the spotlist start and end
        if i >= handles.start(j) && i <= handles.ending(j)  %if image number falls between start and finish of track.
            if i == handles.start(j)  
                c = handles.sx(j);  %changed r to c to conform with ImageJ notation
                r = handles.sy(j);
            else
                if ~isnan(pos_t(i-1,1,j)) && ~isnan(pos_t(i-1,2,j))  % if spot was found in previous image.
                    r = floor(pos_t(i-1,1,j));  
                    c = floor(pos_t(i-1,2,j));
                else  %if previous image contains NaN 
                    c = handles.sx(j);  %changed r to c to conform with ImageJ notation
                    r = handles.sy(j);
                    for l = i-2:-1:1  %go backwards from the image before until you hit no NaN (or 1)
                        if ~isnan(pos_t(l,1,j)) || ~isnan(pos_t(l,2,j)) 
                            r = floor(pos_t(l,1,j));
                            c = floor(pos_t(l,2,j));
                            break
                        else
                            %disp(['Image ' num2str(i-1)  ' missing intensity for spot ' num2str(j)])
                        end
                    end
                end
            end

            % cut out the portion that we want
            if (r-((boxwidth-1)/2)) < 1
                tr = 1;
            else
                tr = r-((boxwidth-1)/2);
            end
            if (c-((boxwidth-1)/2)) < 1
                tc = 1;
            else
                tc = c-((boxwidth-1)/2);
            end
            if (r+((boxwidth-1)/2)) > size(im,1)
                lr = size(im,1);
            else
                lr = r+((boxwidth-1)/2);
            end
            if (c+((boxwidth-1)/2)) > size(im,2)
                lc = size(im,2);
            else
                lc = c+((boxwidth-1)/2);
            end
            temp = im(tr:lr,tc:lc);
            
            if getbgimg == true
                bgd = double(bgdfull(tr:lr, tc:lc));
            end

            % crop the image based on width of PSF and padding
            temp = double(temp);
            

            % subtract off the background
               
            temp = temp-bgd;
                        
            b = constants.Bgstd;
            % make all values greater than zero.
            temp = temp.*(temp>0);

            if strcmp(constants.Method,'GaussMask') == 1
                % find the centroid
                [pos N] = webbfit(temp,PSFpix);
                % assume standard deviation is just the PSF already in units of
                % nm
                sz = size(temp);
                cy = pos(1);
                cx = pos(2);
                %disp(pos);
%                 if (1 < cx) && (cx < sz(1)) && (1 < cy) && (cy < sz(2))
%                     pos = [cy, cx];
%                 else
%                     pos = [NaN, NaN];
%                 end
                s = [PSFpix PSFpix];
            elseif strcmp(constants.Method,'Gauss2DJILA') == 1
                % JILA 2D Gaussian fits
                [cx,cy,sx,sy] = Gaussian2DJILA(temp,1e-4);

                sz = size(temp);
                if (1 < cx) && (cx < sz(1)) && (1 < cy) && (cy < sz(2))
                    pos = [cy, cx];
                else
                    pos = [NaN, NaN];
                end

                s = [sy sx];
                s = s.*constants.pixelsize;
                N = sum(sum(temp));
                

                
            elseif strcmp(constants.Method,'Gauss2DNLSQ') == 1
                % 2D nonlinsq Gaussian fit
                a_fit = Gaussian2DNLSQ(temp,3000,3000);
                pos = [a_fit(2) a_fit(4)];
                % spot size width, which we get directly from the fit.
                % Convert it to nm units.
                s = [a_fit(3) a_fit(5)];
                s = s.*constants.pixelsize;
                N = sum(sum(temp));
            else
                
                disp('Exiting:  Please define ''Method'' as ''Gauss2DJILA'', or ''GaussMask''')
                return
            end
            % calculate the number of photons
            N = N*constants.photonconversion/constants.EMgain;
            % give coordinates absolute position
            if isnan(pos(1)) || isnan(pos(2))
                pos = [NaN NaN];
            else
                pos = [pos(1)+r-((boxwidth-1)/2)-1 pos(2)+c-((boxwidth-1)/2)-1];
                %                 pos = [r-pos(1)+1 c-pos(2)+1];
            end
            % error in nm units. (FIONA)
            sigma = sqrt(((s.^2)./N)+(((a^2)/12)/N)+((8*pi.*s.^2 .* b^2)./(a^2 * N^2)));
            pos_t(i,:,j) = pos;
            sigma_t(i,:,j) = sigma;
            photons(i,j) = N;
        else
            pos_t(i,:,j) = [NaN NaN];
            sigma_t(i,:,j) = [NaN NaN];
            photons(i,j) = NaN;
        end
    end
end

% now save everything to files.
for i = 1:length(handles.sx)
    % figure out what are the fields in the constants.
    fields = fieldnames(constants);
    l = length(fields);
    % make the header
    header = 'header:';
    for j=1:l
        if ischar(getfield(constants,fields{j})) ~=1
            str = num2str(getfield(constants,fields{j}));
        else
            str = getfield(constants,fields{j});
        end
        header = [header, ',' char(fields(j)) '=' char(str)];
    end

    % convert pixels to nm
    if strcmp(constants.Output,'nm') == 1
        pos_t(:,:,i) = pos_t(:,:,i).*constants.pixelsize;
    end

    % get rid of the offsets.
    %     pos_t(:,1,i) = pos_t(:,1,i)-floor(pos_t(1,1,i));
    %     pos_t(:,2,i) = pos_t(:,2,i)-floor(pos_t(1,2,i));

    %create the output directory if it doesn't already exist
    if exist(constants.OutputDirectory)~=7
        mkdir(constants.OutputDirectory)
    end


%% Define name of the xy.txt file: will depend on whether flags are set:

if flagset==0  %No column containing header flag
    foutname = [fullfile(constants.OutputDirectory,outname) '_spot' num2str(handles.peak(i)) '_xy.txt'];
else
    foutname = [fullfile(constants.OutputDirectory,outname) '_spot' num2str(handles.peak(i)) '_' char(flagname(handles.flag(i))) '_xy.txt'];
end

    
 
    disp(['Saving ' foutname]);
    
     % Save tracking data: format will differ depending on pixel on nm   
    fid = fopen(foutname,'w');
    fprintf(fid,'%s\n',header);
    fprintf(fid,'%s\t %s\t %s\t %s\t %s\n','x','y','sx','sy','N');
    if strcmp(constants.Output,'nm') == 1
        fprintf(fid,'%.2f\t %.2f\t %.2f\t %.2f\t %.1f\n',...
            [squeeze(pos_t(:,2,i))' ;squeeze(pos_t(:,1,i))' ;...    % Outputs x values (pos_t(:,2,i)) then y (pos_t(:,1,i)
            squeeze(sigma_t(:,2,i))' ;squeeze(sigma_t(:,1,i))';...
            squeeze(photons(:,i))']);
    else
        fprintf(fid,'%.4f\t %.4f\t %.2f\t %.2f\t %.1f\n',...
            [squeeze(pos_t(:,2,i))' ;squeeze(pos_t(:,1,i))' ;...    % Outputs x values (pos_t(:,2,i)) then y (pos_t(:,1,i)
            squeeze(sigma_t(:,2,i))' ;squeeze(sigma_t(:,1,i))';...
            squeeze(photons(:,i))']);

    end
    fclose(fid);


%% Define name of the xy_only.txt file: will depend on whether flags are set:    
if flagset==0  %No column containing header flag
    foutname = [fullfile('.',outname) '_spot' num2str(handles.peak(i)) '_xy_only.txt'];
else
    foutname = [fullfile('.',outname) '_spot' num2str(handles.peak(i)) '_' char(flagname(handles.flag(i))) '_xy_only.txt'];
end

    
 
    disp(['Saving ' foutname]);    
    
    % Save only xy tracking data: format will differ depending on pixel on nm   
    fid = fopen(foutname,'w');
    if strcmp(constants.Output,'nm') == 1
        fprintf(fid,'%.2f\t %.2f\n',...
            [squeeze(pos_t(:,2,i))' ;squeeze(pos_t(:,1,i))']);    % Outputs x values (pos_t(:,2,i)) then y (pos_t(:,1,i)
    else
        fprintf(fid,'%.4f\t %.4f\n',...
            [squeeze(pos_t(:,2,i))' ;squeeze(pos_t(:,1,i))']);    % Outputs x values (pos_t(:,2,i)) then y (pos_t(:,1,i)

    end
    fclose(fid);
end




%% end of file
end

%% Functions:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
% These need to be outside the main function (or otherwise the load command
% does not work...


%% Load image i from stack

function im=loadtiffstack(i,input)
%tf = imformats('tif');
%im = double(feval(tf.read, input, i));
im = double(imread(input, i));

% For some reason, 16 bit images show up with an extra 32,268 added to
% them.  I have no idea why this happens. -PAC 2010-09-23
im = im - 2.0^(floor(log2(double(min(im(:))))));

end

%% Load image from a directory:

function im=loaddirectoryimage(i,input)
tf = imformats('tif');
imagelist=dir(fullfile(input,'*.tif'));
image=imagelist(i).name;
im = double(feval(tf.read, fullfile(input,image)));


end

%% Find fields


function ind = find_fields(f,varargin)
% returns all the location of the field in f. f is a cellarray of strings. 

% possible fields: Sx, Sy, Ex, Ey, Peak, Flag, Start, Stop, End, x, y, sx,
% sy, N

ind = [];
for i=1:length(varargin)
    for j=1:length(f)
        if strcmp(f{j},varargin{i}) == 1
            ind(i) = j;
            break
        end 
    end
end
end

%%  Read header

function argout = read_header(file,arg)
% read the header parameters. Assume that the header has been separated by
% semicolons, and parameters are values are announced by equal signs
    fid = fopen(file,'r');
    str = textscan(fid,'%s','delimiter',';');
    L = length(str{1});
    for i=1:L
        n = findstr(char(str{1}(i)),'=');
        if ~isempty(n)
            argc = char(str{1}(i));
            if strcmp(arg,argc(1:n-1)) == 1
                argout = argc(n+1:end);
%                 break
            end
        end
    end
    fclose(fid);
end


%% Read spotlist

function [A B] = read_spotlist(name)
    % reads the spotlist.txt file; for now, assume that the structure of the
    % file is 1st n lines are header, last column is name of file, all other
    % columns have numbers. There are 8 total columns.
    %
    % A = read_spots(name,n);
    % output A returns the fields
    
    % go through the file and check how many headerlines there are. Do this
    % until we hit a number, usually.
    fid =[];
    name
    fid = fopen(name,'r');
    c = 1;
	while 1
%        A = fgetl(fid);
        A = fgets(fid);
        B = textscan(A,'%s','delimiter','\t','multipledelimsasone', 1);
        if isnan(str2double(A(1))) ~= 1
            break
        end
        c = c+1;
    end        
    fclose(fid);
    
    % recover the fields and headings
    fid = fopen(name,'r');    
    for i=1:c-1
%        A = fgetl(fid);
        A = fgets(fid);
        A = textscan(A,'%s','delimiter','\t','multipledelimsasone', 1);        
    end

    % now convert A from a one cell variable to a multiple cell of chars
    % with each cell being a field to be probed by 
    A = A{1};
    for i=1:length(A)
        D{i} = strtrim(char(A(i)));
    end    
    A = D;
%     A = textscan(char(A),'%s');
    fclose(fid);
    
%     L = length(A{1});
    L = length(A);
    fid = fopen(name,'r');    
    str2 = [];    
    for i=1:L
        % note that we are saving everything as floats
        str2 = [str2 '%f '];
    end   
    B = textscan(fid,str2,'headerlines',c-1);    
    fclose(fid);

end

%% Gaussian function

function [cx,cy,sx,sy,PeakOD] = Gaussian2DJILA(m,tol)
% following is adapted from JILA
% http://jilawww.colorado.edu/bec/BEC_for_everyone/matlabfitting.htm
% a function to fit a thermal cloud 2-D
    try
        options = optimset('Display','off','TolFun',tol,'LargeScale','off');
    
        % find the center of mass coordinates and peak
        [sizey sizex] = size(m);
        [cx,cy,sx,sy] = centerofmass(m);
        pOD = max(max(m));

        % estimate the 1D "x" coordinate
        mx = m(round(cy),:);
        x1D = 1:sizex;
        ip1D = [cx,sx,pOD];
        fp1D = fminunc(@fitGaussian1D,ip1D,options,mx,x1D);
        cx = fp1D(1);
        sx = fp1D(2);
        PeakOD = fp1D(3);

        % estimate the 1D "y" coordinate    
        my = m(:,round(cx))';
        y1D = 1:sizey;
        ip1D = [cy,sy,pOD];
        fp1D = fminunc(@fitGaussian1D,ip1D,options,my,y1D);
        cy = fp1D(1);
        sy = fp1D(2);
        PeakOD = fp1D(3);
    
        % now perform 2D fit based on estimates of 1D fittings
        [X,Y] = meshgrid(1:sizex,1:sizey);
        initpar = [cx,cy,sx,sy,PeakOD];
        m = m - median(min(m));
        fp = fminunc(@fitGaussian2D,initpar,options,m,X,Y);
        cx = fp(1);
        cy = fp(2);
        sx = fp(3);
        sy = fp(4);
        PeakOD = fp(5);
    catch
%         lasterr        
        cx = NaN;
        cy = NaN;
        sx = NaN;
        sy = NaN;
        PeakOD = NaN;        
    end
end

function [cx,cy,sx,sy] = centerofmass(m)
% PURPOSE: find c of m of distribution

    [sizey sizex] = size(m);
    vx = sum(m);
    vy = sum(m');

    vx = vx.*(vx>0);
    vy = vy.*(vy>0);

    x = [1:sizex];
    y = [1:sizey];

    cx = sum(vx.*x)/sum(vx);
    cy = sum(vy.*y)/sum(vy);

    sx = sqrt(sum(vx.*(abs(x-cx).^2))/sum(vx));
    sy = sqrt(sum(vy.*(abs(y-cy).^2))/sum(vy));
end

function [z] = fitGaussian1D(p,v,x)

%cx = p(1);
%wx = p(2);
%amp = p(3);

    zx = p(3)*exp(-0.5*(x-p(1)).^2./(p(2)^2)) - v;
    z = sum(zx.^2);
end

function [z] = fitGaussian2D(p,m,X,Y)

%cx = p(1);
%cy = p(2);
%wx = p(3);
%wy = p(4);
%amp = p(5);

    ztmp = p(5)*(exp(-0.5*(X-p(1)).^2./(p(3)^2)-0.5*(Y-p(2)).^2./(p(4)^2))) - m;
    z = sum(sum(ztmp.^2));
end
% End JILA code

%% 

function a_fit = Gaussian2DNLSQ(temp,mfe,mi)
% mfe= maxfunevals
% mi = maxiterations

    % 2D nonlinsq Gaussian fit
    lb = [0 0 0 0 0];
    ub = [Inf Inf Inf Inf Inf];
    [dx dy] = ndgrid(1:size(temp,1), 1:size(temp,2));
    [x y] = find(temp == max(max(temp)));
    estimate = [max(max(temp)) mean(x) PSFpix mean(y) PSFpix];
    options =...
        optimset('MaxFunEvals',mfe,'MaxIter',mi,'Display','off');%,'TolFun',1e-20,'TolX',1e-20); 
    a_fit =...
        lsqnonlin(@gcost2d,estimate,lb,ub,options,dx,dy,temp,0);  

    % Cost function for 2D function fitting
    function err = gcost2d(estimate,dx,dy,fz,bg)

        % Calculate model function
        h = gauss2d(estimate,dx,dy,bg);

        % Calculate and flatten the difference vector
        err = h - fz;
    end

    function h = gauss2d(estimate,dx,dy,bg)
        h = estimate(1)*exp(-0.5*((dx-estimate(2)).^2)/estimate(3).^2 - 0.5*((dy-estimate(4)).^2)/estimate(5).^2)+bg;
    end
end

%%

function [fitpos N] = webbfit(im,PSFwidth)
% 
% A = size(im);
% HalfWidth = (A(1) - 1) ./ 2;
% x = -HalfWidth:HalfWidth;
% [X Y] = ndgrid(x, x);
[X Y] = ndgrid(1:size(im,1), 1:size(im,2));

% M = size(X);

fitpos = [0 0];
for i = 1:200
    oldpos = fitpos;
    % compute the mask
    gaussmask = exp(-0.5*((X-fitpos(1)).^2)/(PSFwidth) -0.5*((Y-fitpos(2)).^2)/(PSFwidth));
    denom = sum(sum(im.*gaussmask));

        fitpos(1) = sum(sum(X.*im.*gaussmask))/denom;
        fitpos(2) = sum(sum(Y.*im.*gaussmask))/denom; 

end

    N = sum(sum(denom))/sum(sum(gaussmask.*gaussmask));

% ChiSq = sum(sum((MolImg - NumSigPhot .*
% squeeze(MaskArray(MaskElements(1), MaskElements(2), :, :))).^2));

end







