; ********* start of localize.pro
;+
; NAME:
;   localize
; PURPOSE:
;   Finds the center of sub-resoultion fluorescent sources in an image.
; CATEGORY:
;   Image Processing
; CALLING SEQUENCE:
;   localize, image stack, first frame, last frame, [selection=selection]
; INPUTS:
;   image stack: the time series movie to be analyzed.  Format is the Bio-Rad pic format
;   first frame: frame to start analyzing
;   last frame: frame to finish analyzing
;          note: these frame numbers don't have to be inclusive of entire time series
; REFERENCES:
;   Thompson, Larson, Webb. "Precise Nanometer Localization of Individual Fluorescent Probes."
;   Biophysical Journal, 2002.
;   Larson.  Ph. D. Thesis.  Cornell, 2004

; MODIFICATION HISTORY:
; 9/2000 written by Daniel Larson, Cornell University
; 4/2004 v 1.0 distribution

PRO localize, parameters, Selection = selection, Mag=mag, Fixed_mask_image=fixed_mask_image, manyROI=manyROI, auxiliary=auxiliary, batch=batch, bootstrap=bootstrap, reverse=reverse
; LOCALIZE is the integrated tracking program.
; General approach:
;     1. Find a spot and cut that spot to the localisation routine
;     2. Find the center
;     3. Find the number of photons in the spot
;     4. Reject spots with less than a certain number of photons
;     5. Repeat
T = SYSTIME(1)

algorithm = parameters.algorithm
filename = parameters.filename
first_frame=parameters.first
last_frame=parameters.last
FCS_FRAP_tally = 0L
deltax=0.0
deltay=0.0
gmask_res=dblarr(3)     ; these arrays hold the results from each of the fitting routines
                    ; format: [x0, y0, photon number]
gmask_position_array=dblarr(3, 3)

;read the first image, display it, and check for selection keyword and magnification keyowrd
raw_image=read_tiff(filename, image_index=(first_frame-1+parameters.composite*parameters.channel GT 0) ? first_frame-1+parameters.composite*parameters.channel : 0)
IF keyword_set(fixed_mask_image) then raw_image=fixed_mask_image

b=size(raw_image, /dimensions)
if keyword_set(mag) and keyword_set(selection) then begin
    print, 'cannot set the selection and mag keywords at the same time'
    return
    endif
if NOT keyword_set(mag) then mag=1
x_dim = b[0]
y_dim = b[1]

erase
if parameters.composite EQ 1 then tvscl, rebin(raw_image, x_dim/mag, y_dim/mag), 0,0,parameters.channel+1 else $
tvscl, rebin(raw_image, x_dim/mag, y_dim/mag), 0,0,2
x0=0
y0=0
NX=x_dim
NY=y_dim

IF KEYWORD_SET(Selection) THEN BEGIN
    check1=n_elements(selection)   ; = 4
    check2=(selection[0] GE 0)     ;=1
    check3=(selection[1] GE 0)  ;=1
    check4=(selection[2]+selection[0] LT x_dim)  ;=1
    check5=(selection[3]+selection[1] LT y_dim)  ;=1
    check6= total(selection) GT 0          ;=1
    if (check1*check2*check3*check4*check5*check6 EQ 4) then begin
        x0=selection[0]
        y0=selection[1]
        Nx=selection[2]
        Ny=selection[3]
        ;selection = 0   ;why is this line here?  If things start going wrong, maybe put it back
        endif else begin
        res=dialog_message( 'ROI is undefined or outside the boundaries of the image.'+string(10B)+$
                      'Proceeding with full field calculation.', title='region of interest in LOCALIZE')
        endelse
    endif

;open output file
output_file=change_extension(filename, 'loc')

if keyword_set(manyROI) then openw, 2, output_file, error=err, /append else openw, 2, output_file, error=err
IF (err NE 0) then begin
    res=dialog_message(!ERROR_STATE.MSG, title='File error in LOCALIZE. WTF?')
    close, 2
    return
    endif
i=0l

;check to make sure the frame limits are correct
if (last_frame GT parameters.num_images) OR (first_frame LT 1) then begin
    res=dialog_message('Frame limits are incorrect', title='File error in LOCALIZE')
    close, 2
    return
    endif

;**************define a buffer which can be used in FCS_FRAP mode to fill in the blanks for the no spot scenarios
;**************The buffer is updated at the end of each frame.********************************************
buffer=dblarr(4)
buffer[3]=first_frame-1

;*********************bootstrap mode*****************************************************************
;In this mode, the spot location in the previous frame is used to define the center of the ROI
;in the present frame. Thus, x0 and y0 are variable, but NX and NY are not. Use the same buffer as
;definded above. However, the initial values have to be initialized with the user input center of the ROI
If keyword_set(bootstrap) then begin
  buffer[0]=Nx/2.+x0
  buffer[1]=Ny/2.+y0
endif

;********start the main loop that cycles through the frames***************************************
for i = first_frame-1, last_frame - 1 do begin

if keyword_set(reverse) then begin
  bookkeeper=i
  i=last_frame-1-i
  endif

    raw_image=read_tiff(filename, image_index=i*(1+parameters.composite)+parameters.composite*parameters.channel)    ;if parameters.composite is set, every other image frame is read
    
IF keyword_set(fixed_mask_image) then raw_image=fixed_mask_image

IF (raw_image[0] EQ -1L) then begin
       res=dialog_message('Cannot read pic file', title='File error in LOCALIZE')
       close, 2
       return
     endif

    widget_control, parameters.outputlocalizeID, set_value='frame:'+string(i+1)+ '...',$
     /append, SET_TEXT_TOP_LINE=(i-1) > 0
     erase_this_mask=dblarr(x_dim, y_dim)
     erase_this_mask[x0, y0:y0+NY-1]=1.0
     erase_this_mask[x0+NX-1, y0:y0+NY-1]=1.0
     erase_this_mask[x0:x0+NX-1, y0]=1.0
     erase_this_mask[x0:x0+NX-1, y0+NY-1]=1.0
     tvscl, erase_this_mask, 0, 0,3
    selected_spot_results=find_spot(raw_image[x0:x0+NX-1, y0:y0+NY-1], parameters)
    spot_number=selected_spot_results[0].spot_number

if (spot_number EQ 0) then begin
    gmask_position_array=[0,0,0]
    algorithm_name=' '
    endif

;With the auxiliary switch set, the program reads in an arbitrary loc file to use the positions to calculate intensity
;This module is most useful for using the position measurement in one color to determine the intensity in
;another color.

if keyword_set(auxiliary) then begin
	if strupcase(STRMID(parameters.auxfile, 2, /REVERSE_OFFSET)) EQ 'LOC' then auxdat=read_loc(parameters.auxfile)
	if strupcase(STRMID(parameters.auxfile, 2, /REVERSE_OFFSET)) EQ 'TRK' then auxdat=read_trk(parameters.auxfile)
	auxindex=where(auxdat[3, *] EQ i, spot_number)	;note: here we override the previous spot number
	auxdat_frame=auxdat[*, auxindex]

	;because we are essentiall circumventing FIND_SPOT, we have to make a structure which resembles the output
	;of FIND_SPOT
	cut_width=round(parameters.cutsize*parameters.psfwidth)
	; structure that constitutes the cut spot
	cut_spot={spot_number:spot_number,deltax:0.0,deltay:0.0,spot_image:dblarr(2.0*cut_width+1, 2.0*cut_width+1)}
    selected_spot_results=replicate(cut_spot, spot_number)
    x0=0.0
    y0=0.0
	for p=0L, spot_number-1 do begin
		x=round(auxdat_frame[0, p])		;these are the real spot locations in the FULL image for a particular FRAME (i.e. no cropping)
		y=round(auxdat_frame[1, p])
		left_x=x-cut_width GT 0 ? x-cut_width : 0
    right_x=x+cut_width LT x_dim-1 ? x+cut_width : x_dim-1
    bottom_y=y-cut_width GT 0 ? y-cut_width : 0
    top_y= y+cut_width LT y_dim-1 ? y+cut_width : y_dim-1
		
		selected_spot_results[p].spot_image=raw_image[left_x:right_x, bottom_y:top_y]
		selected_spot_results[p].deltax=x-cut_width
		selected_spot_results[p].deltay=y-cut_width
		endfor
	endif

if (spot_number GT 0) then begin
    gmask_position_array=dblarr(3, spot_number)

    if (algorithm eq 0) then begin  ;Gaussian Mask Algorithm
    algorithm_name='Gaussian Mask Algorithm'
    for m=0L, spot_number-1 do begin
        selected_spot=selected_spot_results[m].spot_image
        deltax=selected_spot_results[m].deltax
        deltay=selected_spot_results[m].deltay

        ;THE FOLLOWING LINE HANDLES ASSIGNMENT OF BLACK LEVEL.
        if parameters.auto_black_level_val eq 0 then parameters.blacklevel=median(raw_image[x0:x0+NX-1, y0:y0+NY-1])

		;This is the function call to the actual fitting routine
		if keyword_set(auxiliary) then gmask_res=gmask_fit(selected_spot, parameters, $
									xy_input=[auxdat_frame[0, m]-selected_spot_results[m].deltax, auxdat_frame[1, m]-selected_spot_results[m].deltay]) $
       							 else gmask_res=gmask_fit(selected_spot, parameters)

       ; offset correction
        gmask_res[0]=gmask_res[0]+deltax+x0
        gmask_res[1]=gmask_res[1]+deltay+y0

        if (gmask_res[2] LT parameters.photonthreshold) then begin
        gmask_res[0]=0
        gmask_res[1]=0
        gmask_res[2]=0
        endif
        gmask_position_array[*, m]=[gmask_res[0], gmask_res[1], gmask_res[2]]
        endfor
    endif

    if (algorithm eq 1) then begin  ;Nonlinear Least Squares Algorithm
    algorithm_name='Nonlinear Least Squares Algorithm'
    for m=0L, spot_number-1 do begin
        selected_spot=selected_spot_results[m].spot_image
        deltax=selected_spot_results[m].deltax
        deltay=selected_spot_results[m].deltay

        ;THE FOLLOWING LINE DOES *VARIABLE* ASSIGNMENT OF BLACK LEVEL.
        if parameters.auto_black_level_val eq 0 then parameters.blacklevel=median(raw_image[x0:x0+NX-1, y0:y0+NY-1])
        gmask_res=leastsqr_fit(selected_spot, parameters)

       ; offset correction
        gmask_res[0]=gmask_res[0]+deltax+x0
        gmask_res[1]=gmask_res[1]+deltay+y0

        if (gmask_res[2] LT parameters.photonthreshold) then begin
        gmask_res[0]=0
        gmask_res[1]=0
        gmask_res[2]=0
        endif
        gmask_position_array[*, m]=[gmask_res[0], gmask_res[1], gmask_res[2]]
        endfor
    endif

    ;this line outputs the value used in the black_level, to make this field more informative
    widget_control, parameters.blacklevelID, set_value=parameters.blacklevel
endif

    ;now cull the spot list for redundancy (i.e. spots closer than the tolerance)
    ;****************************************************************************
    gm_culled_spots = cull_spot(gmask_position_array, parameters)
    value = output_center(raw_image, gm_culled_spots, mag, parameters.peak_white, parameters.channel); /writeitff)
    a=N_elements(gm_culled_spots)
    output = dblarr(4, a/3)
    output[0:2, *]=gm_culled_spots
    output[3, *] = i
    ;**************************************************************************

    ;*************************************************************************
    ;The following lines are for running in FCS-FRAP mode.  This mode is very particular to the type
    ;of data generated in a transcription site experiment and is NOT to be used in general experiments.
    ;If no spot is found, the position(s) from
    ;the PRECEEDING frame are used to calculate a photon number for the PRESENT frame.
    ;There are a couple of different ways one can imagine using the buffer:
    ;1) if NO spots are found, use list of spots in the last frame
    ;2) ALWAYS use at LEAST one spot from the last frame (the brightest) as a potential spot in the next frame
    ;3) or, a newer method, whenever a spot goes above threshold, keep that "spot" for all time.  Thus, the list
    ;   of spots is always growing.


    ;Method 2
    ;-----------------------------------------------------------------------------------

	;check the number of spots in the buffer:

	if parameters.FCS_FRAP EQ 0 then begin
       FCS_FRAP_radius=sqrt((gm_culled_spots[0,0]-buffer[0,0])^2+(gm_culled_spots[1,0]-buffer[1,0])^2)
  
    if ((FCS_FRAP_radius GT 0.0) AND (FCS_FRAP_radius LT parameters.maxdisp) AND (keyword_set(bootstrap))) then begin  ; i.e. if there was any movement from the previous frame, then re-define the ROI
          x0=(gm_culled_spots[0,0]-Nx/2. GE 0.0) ? gm_culled_spots[0,0]-Nx/2. : 0.0 
          y0=(gm_culled_spots[1,0]-Ny/2. GE 0.0) ? gm_culled_spots[1,0]-Ny/2. : 0.0
          ;print, FCS_frap_radius, x0, y0
        endif

      if ((FCS_FRAP_radius GE parameters.maxdisp) AND (i GE first_frame) AND (i LE last_frame-1)) then begin
            output=buffer
           for j=0, n_elements(buffer)/4 - 1 do begin
           		xy_input=buffer[0:1, j]

              ;the following lines essentially circumvent FIND_SPOT.PRO.  The cut_width is defined the same
              ;as in FIND_SPOT, and the selected_spot is the same size as in FIND_SPOT.PRO.  Thus, a portion
              ;of the image is sent to GMASK_FIT without going first through FIND_SPOT.
              ;I'm not entirely happy with this solution, but it seems the most economical and straightforward.
                cut_width=round(parameters.cutsize*parameters.psfwidth)

               if (xy_input[0] LT cut_width) OR (xy_input[1] LT cut_width) then begin
                    res=dialog_message('The spot in the preceeding frame is the trivial spot.', title='Error in FCS_FRAP module of LOCALIZE')
                    printf, 2, FORMAT = '(4(f11.3, "    "))',  output
                    close, 2
                   return
                   endif

                selected_spot=raw_image[xy_input[0]-cut_width:xy_input[0]+cut_width, xy_input[1]-cut_width:xy_input[1]+cut_width]
                gmask_res=gmask_fit(selected_spot, parameters, xy_input=[cut_width-0.5,cut_width+0.5])
                output[2,j]=gmask_res[2]  ;the photon number is returned from GMASK_FIT
                                		;(note: the x,y values are unchanged from the buffer)
                output[3, j]=i          ;the frame is the current frame
                FCS_FRAP_tally=FCS_FRAP_tally + 1
           endfor
          endif
      endif

    buffer=output   ;this buffer is for localization in FCS/FRAP mode.  If no spots are found, these
    ;print, i	   ;positions from a preceeding frame are used for calculating photons for the
   ; print, buffer   ;frame where no 'spots' are found.
    ;----------------------------------------------------------------------------------------

    printf, 2, FORMAT = '(4(f11.3, "    "))',  output
    widget_control, parameters.outputlocalizeID, set_value='spots:'+string(a/3),$
     /append, SET_TEXT_TOP_LINE=(i-1) > 0
     
     if keyword_set(reverse) then i=bookkeeper
endfor

close, 2

; this file is an output file containing the parameters

runtime=systime(1)-T

parameter_file = filename
parameter_file=change_extension(filename, 'par')
openw, 1, parameter_file, /append
printf, 1, '***********************************'
printf, 1, 'LOCALIZATION parameters'
printf, 1, '***********************************'
printf, 1, systime()
printf, 1, 'input file=', filename
printf, 1, 'output file=', output_file
printf, 1, 'psfwidth=', parameters.psfwidth
printf, 1, 'region (x0, y0, Nx, Ny):', x0, y0, Nx, Ny
if parameters.auto_bpass_val EQ 0 then printf, 1, 'auto threshold:', parameters.autoBpassThreshold
if parameters.auto_bpass_val EQ 1 then printf, 1, 'manual threshold:', parameters.bpassthreshold
if parameters.auto_black_level_val eq 0 then printf, 1, 'black level: median'
if parameters.auto_black_level_val eq 1 then printf, 1, 'black level: ', parameters.blacklevel
if parameters.auto_black_level_val eq 2 then printf, 1, 'black level: local (i.e. fit to a plane'
printf, 1, 'cut_size=', parameters.cutsize
printf, 1, 'photon_threshold=', parameters.photonthreshold
printf, 1, 'tolerance=', parameters.tolerance
printf, 1, 'number of spots:', a/3
printf, 1, 'algorithm: ', algorithm_name
printf, 1, '# frames localized in FCS_FRAP mode: ', FCS_frap_tally
if keyword_set(auxiliary) then printf, 1, 'auxiliary file used for localization:'+parameters.auxfile
printf, 1, 'run time:', runtime
close, 1

if keyword_set(batch) EQ 0 then begin 
  if keyword_set(manyROI) EQ 0 then begin
    res=dialog_message(title='LOCALIZE complete', $
    'run time: '+string(runtime)+' seconds'+string(10b)+$
    'data written to: '+output_file+string(10b)+ $
    'parameters written to: '+parameter_file+string(10b))
    endif
endif
return
end