;+
; NAME:
;   lsm_read
;
;
; PURPOSE:
;   Read ZEISS lsm5 data files
;
;
; CATEGORY:
;   image I/O
;
;
; CALLING SEQUENCE:
;   data=lsm_read(name)
;
;
; INPUTS:
;   ZEISS lsm5 data file
;
;
; OPTIONAL INPUTS:
;
;
;
; KEYWORD PARAMETERS:
;   ZEISS:    variable will contain ZEISS private header
;   PLANAR:   planar configuration
;   THUMBNAIL:    variable will contain thumbnail
;
;
; OUTPUTS:
;   data: image data
;
;
; OPTIONAL OUTPUTS:
;
;
;
; COMMON BLOCKS:
;
;
;
; SIDE EFFECTS:
;
;
;
; RESTRICTIONS:
;
;
;
; PROCEDURE:
;
;
;
; EXAMPLE:
;
;
;
; MODIFICATION HISTORY:
;
;       Thu Oct 8 10:07:37 1998, Karsten Rodenacker
;       <iliad@janus.gsf.de>
;
;4/20/2009, Daniel Larson, DLARSON@AECOM.YU>EDU
;1. lsm_read_point checks for LZW compression and returns -1 if images are compressed.
;
;-
FUNCTION lsm_data_read, resu, lun, x_offset, $
                        samples, stripbytec, stripoffsets, bitspersample
   FOR i=0, samples-1 DO BEGIN
     CASE bitspersample[i] OF
       16: data = intarr(stripbytec[i]/2)
       32: data = lonarr(stripbytec[i]/4)
       ELSE: data = bytarr(stripbytec[i])
     ENDCASE
     point_lun, lun, stripoffsets[i]+x_offset
     readu, lun, data
     IF (bitspersample[i] NE 8) and (!version.arch ne 'x86') THEN data = swap_endian(data)
     IF n_elements(resu) LE 1 THEN resu = temporary(data) $
     ELSE resu = [[resu], [temporary(data)]]
   ENDFOR
   return, resu
END

FUNCTION lsm_value_read, direntry, lun, x_offset
   CASE direntry.type OF
     3:typlen = 2
     4:typlen = 4
     5:typlen = 4;?
     ELSE:typlen = 1
   ENDCASE
   IF direntry.length*typlen LE 4 THEN BEGIN
     CASE direntry.type OF
       1: val = byte(direntry.value)
       2: val = string(direntry.value)
       3: val = fix(direntry.value)
       4: val = long(direntry.value)
       5: val = float(direntry.value)
     ELSE: val = byte(direntry.value)
   ENDCASE
   ENDIF ELSE BEGIN
     point_lun, lun, direntry.value+x_offset
     CASE direntry.type OF
       3:val = intarr(direntry.length)
       4:val = lonarr(direntry.length)
       5:val = fltarr(direntry.length)
       ELSE:val = bytarr(direntry.length)
     ENDCASE
     IF direntry.tag EQ 34412u THEN val = {zeiss5}
     readu, lun, val
     IF direntry.type EQ 2 THEN val = string(val) $
     ELSE if !version.arch ne 'x86' then val = swap_endian(val)
   ENDELSE
   return, val
 END

FUNCTION lsm_read_point, lun, x_offset, ZEISS=lminfo, PLANAR=planar, THUMBNAIL=tum

   tiff_head = {tiff_head, byteorder:0u, identifier:0u, offset:0ul}
   readu, lun, tiff_head
   if !version.arch ne 'x86' then tiff_head = swap_endian(tiff_head)
   if (tiff_head.offset eq 0ul) and (x_offset ne 0ul) then return,-1l

   img_dir = {img_dir, tag:0u, type:0u, length:0ul, value:0ul}
   ntags = 0u
   offs = 0ul
   offset = tiff_head.offset

   tum = 0
   tumz = 0
   resu = 0
   resuz = 0

   REPEAT BEGIN
     readu, lun, ntags
     if !version.arch ne 'x86' then ntags = swap_endian(ntags)

     dir = replicate(img_dir, ntags)
     readu, lun, dir
     readu, lun, offset
     if !version.arch ne 'x86' then offset = swap_endian(offset)
     if !version.arch ne 'x86' then dir = swap_endian(dir)
     FOR i=0, ntags-1 DO CASE dir[i].tag OF
       254u:newsubfiletype = lsm_value_read(dir[i], lun, x_offset)
       256u:imagewidth = lsm_value_read(dir[i], lun, x_offset)
       257u:imagelength = lsm_value_read(dir[i], lun, x_offset)
       258u:bitspersample = lsm_value_read(dir[i], lun, x_offset)
       259u:compression = lsm_value_read(dir[i], lun, x_offset)
       262u:photo = lsm_value_read(dir[i], lun, x_offset)
       ;271u:producer = lsm_value_read(dir[i], lun, x_offset)
       ;272u:microscope = lsm_value_read(dir[i], lun, x_offset)
       273u:stripoffsets = lsm_value_read(dir[i], lun, x_offset)
       277u:samples = lsm_value_read(dir[i], lun, x_offset)
       279u:stripbytec = lsm_value_read(dir[i], lun, x_offset)
       284u:planar = lsm_value_read(dir[i], lun, x_offset)
       ;305u:zif_vers = lsm_value_read(dir[i], lun, x_offset)
       317u:predictor = lsm_value_read(dir[i], lun, x_offset)
       320u:colormap = lsm_value_read(dir[i], lun, x_offset)
       34412u:lminfo = lsm_value_read(dir[i], lun, x_offset)
       ELSE:print, dir[i]
     ENDCASE

if compression EQ 5 then return, -1

;*****************************************
;DEBUGGING code
;print, 'newsubfiletype:', newsubfiletype
;print, 'image dim:', imagewidth, imagelength
;print, 'bits per sample:', bitspersample
;print, 'compression:', compression
;print, 'photo:', photo
;print, 'strip offsets:', stripoffsets
;print, 'samples:', samples
;print, 'strip byte rec:', stripbytec
;print, 'planar:', planar
;print, 'predictor:', predictor

;******************************************
     CASE newsubfiletype OF
       0:BEGIN
         resuz = resuz+1
         resuwidth = imagewidth
         resulength = imagelength
         ;resubits = bitspersample		;IS THIS LINE NEEDED? VARIABLE EVER USED?
         resusamples = samples
         resu = lsm_data_read(resu, lun, x_offset, samples, $
                              stripbytec, stripoffsets, bitspersample)
       END
       1:IF arg_present(tum) THEN BEGIN
         tumz = tumz+1
         tumwidth = imagewidth
         tumlength = imagelength
         tumbits = bitspersample
         tumsamples = samples
         tum = lsm_data_read(tum, lun, x_offset, samples, $
                             stripbytec, stripoffsets, bitspersample)
		ENDIF
       ELSE:
     ENDCASE
     IF offset NE 0 THEN point_lun, lun, offset+x_offset
    ENDREP UNTIL offset EQ 0  ;THIS is the original line
   IF n_elements(resu) GT 0 THEN $
    resu = reform(resu, resuwidth, resulength, resusamples, resuz, /over)
   IF arg_present(tum) AND (n_elements(tum) GT 0) $
     AND (size(tum, /n_dim) NE 0) THEN $
    tum = reform(tum, tumwidth, tumlength, tumsamples, tumz, /over)

   return, resu
 end

FUNCTION lsm_read, name, ZEISS=lminfo, PLANAR=planar, THUMBNAIL=tum, NEXT_RECORDING=nxrec

   inxrec=keyword_set(nxrec)?nxrec:0l

   openr, lun, name, /get_lun

   offset=0ul

	resu=lsm_read_point(lun, offset, ZEISS=lminfo, PLANAR=planar, THUMBNAIL=tum)

	if resu[0] EQ -1 then begin
		print, '*********LZW compression not supported.************'
		return, -1
		endif

   if (n_elements(resu) eq 1) and (resu[0] eq -1l) then message,'No image found'
   nxcount=0
   while (inxrec gt 0) do begin
     if lminfo.offs_nextrec ne 0ul then begin
       point_lun, lun, lminfo.offs_nextrec
       resu1=lsm_read_point(lun, lminfo.offs_nextrec, ZEISS=lminfo, PLANAR=planar, THUMBNAIL=tum)
       if not((n_elements(resu1) eq 1) and (resu1[0] eq -1l)) then begin
         resu=resu1
         inxrec = inxrec-1
         nxcount=nxcount+1
       endif else begin
         message,'No second image found',/continue
         inxrec = 0
         nxcount= 0
       endelse
     endif else inxrec=0l
   endwhile
   nxrec=nxcount

   free_lun, lun

   return, reform(resu, /over)
 END
