;****************************Sean Burke's RTFM module***************************************
; rtfm
; the Restricted Three-d Filter Module
; qualifies image sequences for focus level
; INPUTS:
; filename - filename (IPL)
; frame_score - a vector of percentages of "focusness" compared
;   to the most in-focus frame
PRO rtfm, filename, highFTarget, sliceArr, t_slice, outputID=outputID
    ; read first frame
    widget_control, outputID, set_value="RTFM(), processing 0th frame...", /append
    res = query_tiff(filename, FrameInfo, image_index= 0)
    image = read_tiff(filename, image_index=0)
    no_frames = FrameInfo.num_images
    ; rtfm first image
    rtfm_proc, image, vecSlice, len
    sliceArr = DBLARR(no_frames, (SIZE(vecSlice))[1])
    sliceArr[0, *] = vecSlice
    IF N_PARAMS() EQ 4 THEN curSlice = t_slice ELSE curSlice = 0
    FOR i = 1, no_frames - 1 DO BEGIN
            widget_control, outputID, set_value='RTFM(), processing frame ' + STRCOMPRESS(STRING(i),/remove_all) + ' out of ' + STRCOMPRESS(STRING(no_frames), /remove_all) + ', tSclice ' + STRCOMPRESS(curSlice, /REMOVE_ALL), /append
         IF N_PARAMS() EQ 4 THEN image = read_tiff(filename, image_index=i + t_slice*no_frames) ELSE image = read_tiff(filename, image_index=i)
            rtfm_proc, image, vecSlice, len
            sliceArr[i, *] = vecSlice
    ENDFOR
    ; next, we normalize across the unit mean frequency
    FOR j = 0, (SIZE(vecSlice))[1] - 1 DO BEGIN
       unitMean = TOTAL(sliceArr[*,j])/no_frames
       sliceArr[*,j] = sliceArr[*,j]/unitMean
    ENDFOR
;    print, size(sliceArr)
    ;write_chart, sliceArr, len, "output.ps"
    ;rank frames, arbitrarily, we select a cycle range of 60-80
    highFTarget = FLTARR(no_frames)
    FOR i = 0, no_frames - 1 DO BEGIN
         highFTarget[i] = (TOTAL(sliceArr[i,59:79])/20)
    ENDFOR
    highFTarget = highFTarget/MAX(highFTarget)
END

; rtfm_proc
; the meat and potatoes of the rtfm procedure
; based largely on information from the following site:
; http://www.ece.arizona.edu/~dial/trees/documentation/focustest/index.html
; first, we chop the image into a base 2 sized array,
; next subdivide the matrix into 9 equally overlapping sections
; then FFT, calculate the power spectra and radially average the images.
; the returned vector is a slice of the thre three-d power spectrum
PRO rtfm_proc, image, vecSlice, len
    ;first, lop extra data out
    a = FLOOR(ALOG(MIN([(SIZE(image))[1], (SIZE(image))[2]]))/ALOG(2))
    len = 2^a
    subImages = LONARR(9, 2^(a - 1), 2^(a - 1))
    ; i hope you studied your powers of 2!
    subImages[0,*,*] = image[0:(2^(a-1) - 1), 0:(2^(a - 1) - 1)];
    subImages[1,*,*] = image[2^(a-2):(3*2^(a-2) - 1), 0:(2^(a - 1) - 1)];
    subImages[2,*,*] = image[2^(a-1):2^a - 1, 0:(2^(a - 1) - 1)];
    subImages[3,*,*] = image[0:(2^(a-1) - 1), 2^(a-2):(3*2^(a-2) - 1)];
    subImages[4,*,*] = image[2^(a-2):(3*2^(a-2) - 1), 2^(a-2):(3*2^(a-2) - 1)];
    subImages[5,*,*] = image[2^(a-1):2^a - 1, 2^(a-2):(3*2^(a-2) - 1)];
    subImages[6,*,*] = image[0:(2^(a-1) - 1), 2^(a-1):2^a - 1];
    subImages[7,*,*] = image[2^(a-2):(3*2^(a-2) - 1), 2^(a-1):2^a - 1];
    subImages[8,*,*] = image[2^(a-1):2^a - 1, 2^(a-1):2^a - 1];


    ; temporary buffer for power spect
    psBuff = FLTARR(2^a, 2^a)
    vecSliceTemp = DBLARR(100)
    vecSlice = DBLARR(100)


    FOR i = 0, 8 DO BEGIN
       psBuff = (ABS(FFT(image)))^2
       ; next we average the image radially
       ; 1 degree step size should be more than enough
       ra, psBuff, 0, 0, 1, vecSliceTemp
       vecSlice = vecSlice + vecSliceTemp/8;
    ENDFOR
    vecSlice = vecSlice/vecSlice[0]
;    plot, vecSlice
END

; ra - the radial averager
; INPUTS
; image - a 2D image matrix
; xcen - x center value
; ycen - y center value
; stepSize - theta step size in decimal degrees
; rvec - the output vector.
pro ra, image, xcen, ycen, stepSize, rvec
    rad = (SIZE(rvec))[1]
    rprime = DBLARR(2,rad)
    theta = UINT(0)
    IF xcen EQ 0 AND ycen EQ 0 THEN thetaMax = 90 ELSE $
       thetaMax = 360


    mult = thetaMax/stepSize
    radialArray = DBLARR(mult + 1,rad)


    FOR theta = 0, mult DO BEGIN
       FOR r = 0, rad - 1 DO BEGIN
         angle = 3.1415*theta*stepSize/180
         rprime[0, r] = r*cos(angle)
         rprime[1, r] = r*sin(angle)
       ENDFOR
         ; We will ignore the first two data points
        radialArray[theta, *] = INTERPOLATE(image, rprime[0,0:(rad - 1)], $
         rprime[1,0:(rad - 1)])
    ENDFOR


    FOR r = 0, rad - 1 DO BEGIN
       rvec[r] = TOTAL(radialArray[*, r])/(mult + 1)
    ENDFOR
END
;********************************************************************************************