function [bSuccess] = applyUserConversion()
%APPLYUSERCONVERSION allow the user to select a conversion function to be
%applied on selected files
%     The goal of this function is to allow the user to select files for
%   conversion as well as a function file that will perform the actual
%   conversion. Once converted, the file should return a properly formed
%   data structure as defined in createDataStructDefinition. From that
%   point onwards, the session file can be loaded via the File menu and
%   processing is performed in the normal way. Use the function 
%   extractSucrosePulses.m provided with this software to look at an example 
%   of how to perform a conversion. Please note that the steps in this
%   file are just for illustration, and may not result in a data set
%   suitable for fitting.
%     An extraction function needs to conform to the following rules:
%       1. Two inputs: filecontainerversion and the files for conversion
%       2. Two outputs: conversion success and the data structure
%   The filecontainerversion should be equal to the version in the main
%   interface (this can be checked in the Help>about window). The files for
%   conversion should be a cell array of filenames, preferably with the
%   full path name. The conversion success indicates whether all files were
%   converted and is only used for handling errors and logging. The data
%   structure should contain all information necessary for further
%   processing, as described in the documentation.
%     After conversion, the data structure is verified to ensure that all the
%   required fields are present. This does not imply that the data is in a
%   correct state for further processing: the content of each field is not
%   verified. Check the log for any errors or warnings.
%
%See also: extractSucrosePulses.m, createDataStructDefinition.m,
%verifyDataStructure.m, appendSucroseLog.m

%#ok<*NASGU>

%start with default false
bSuccess = false;

hMain = getappdata(0, 'h_sucrose');
if isempty(hMain)
    %without the main interface, return false
    return;
else
    fileVersion = getappdata(hMain, 'fileContainerVersion');
    versionNr   = getappdata(hMain, 'versionNr');
    appRoot     = getappdata(hMain, 'appRoot');
    settings    = getappdata(hMain, 'settings');
    prevConvert = settings.previousConvert;
end

%{
    get the files for conversion using an 'All files' filterspec. The extract
    function will have to deal with processing the correct extensions. See
    also the example function extractSucrosePulses.m for an implementation
    of data extraction.
%}
[files, pathF, ~] = uigetfile('*.*', 'Select files for conversion', 'MultiSelect', 'on');

if pathF == 0
    return;
elseif ~iscell(files)
    files = {files};
end

%convert files to complete files with path
files = cellfun(@(C) fullfile(pathF, C), files, 'UniformOutput', false);

%{
    Ask the user for the file conversion function: this function will
    handle the conversion from electrophysiological recording to a suitable
    data structure. The converted data structure will be compared against a
    template to verify that it has the correct structure. Please note that
    that does not mean the data will be correctly formatted for analysis:
    the structure is only verified on the presence of required fields and
    sub-structures. The validity of the content of each field is handled by
    the main program. Keep in mind that error checking is not exhaustive!
%}
converters = dir(fullfile(appRoot, 'conversion', '*.m'));
converters = {converters.name};
convertfltr = ~cellfun('isempty',(regexp(converters, '^(extract)')));
converters = converters(convertfltr);
converters{end+1} = 'other...';

if isempty(converters)
    %no converters found, ask the user for the location
    [convertFun, pathF, ~] = uigetfile('*.m', 'Select the conversion function', fullfile(appRoot, 'conversion'), 'MultiSelect', 'off');
else
    %check for previous converter
    if isempty(prevConvert)
        initVal = 1;
    else
        initVal = find(strcmp(prevConvert, converters), 1, 'first');
    end
    
    [select, ok] = listdlg('ListString', converters,...
        'Name', 'Select the conversion function',...
        'SelectionMode', 'single',...
        'InitialValue', initVal,...
        'ListSize', [250, 250]);
    
    if ok == 1
        if strcmp(converters{select}, 'other...')
            %let the user specify the function
            [convertFun, pathF, ~] = uigetfile('*.m', 'Select the conversion function', fullfile(appRoot, 'conversion'), 'MultiSelect', 'off');
        else
            %use one of the 'extract...' functions
            convertFun = converters{select};
            pathF      = fullfile(appRoot, 'conversion');
            
            %update the settings with the selected conversion function
            settings.previousConvert = convertFun;
            setappdata(hMain, 'settings', settings);
        end
    else
        return;
    end
end

if pathF == 0
    return;
end

%check if the path selected is on the matlab search path
currentPath = path();
pathIdx = regexp(currentPath, pathF, 'once');

if isempty(pathIdx)
    %the file is not on the search path, add it temporarily
    try
        path(pathF, currentPath);
    catch me
        disp(getReport(me, 'basic'));
        appendSucroseLog(me, 'urgent');
        return;
    end
end

%get the function name: we only require the filename without the extension.
[~, convertFun, ~] = fileparts(convertFun);

%{
    evaluate the function using feval and supply the required parameters as
    defined for extraction functions. Please consult the documentation for
    further details on the structure of an extraction function. Also take a
    look at the example function provided with the analysis program
    (extractSucrosePules.m) on how to construct an extraction function.
    After the function si evaluated, the data structure generated is
    verified against an empty prototype to make sure the file can load
    in the analysis program. Please note that successfull verification does 
    not automatically mean that the data can be fitted!
%}
try
    [bSuccess, dataStruct] = feval(convertFun, fileVersion, files);
    if ~verifyDataStructure(dataStruct, fileVersion)
        %the data could not be verified, return false
        bSuccess = false;
        path(currentPath);
        return;
    end
catch me2
    disp(getReport(me2, 'basic'));
    appendSucroseLog(me2, 'urgent');
    %reset the search path to the original data
    path(currentPath);
    return;
end

%{
%store the data structure
[matFile, matPath, ~] = uiputfile('*.mat', 'Save the session file');
if matPath == 0
    bSuccess = false;
    path(currentPath);
    return;
end

try
    %save the structure
    save(fullfile(matPath, matFile), 'dataStruct', 'fileVersion', 'versionNr');
    appendSucroseLog('Files were succesfully converted', 'info');
catch me3
    %failed to save the data
    disp(getReport(me3, 'basic'));
    appendSucroseLog(me3, 'urgent');
    bSuccess = false;
    path(currentPath);
    return;
end
%}

%if we were successfull, add the structure to the appdata
if bSuccess
    setappdata(hMain, 'dataStruct', dataStruct);
end

%reset the search path to the original path
path(currentPath);
end