function [bSuccess, dataStruct] = extractSucrosePulses(fileContainerVer, dataFiles)
%EXTRACTSUCROSEPULSES Example of a user-defined function for creating a
%session file suitable for loading in Sucrose Analysis
%
%   This file shows how to create a function for converting
%   electrophysiology data to a format that can be used as input for
%   sucrose analysis. This template can be extended or replicated for use
%   within Sucrose Analysis. It can be called from within the main
%   interface via the file menu. The applyUserConversion allows the
%   selection of this file. It also handles the saving of the data
%   structure and the verification of the structure.
%
%   input (required): 
%       fileContainerVer:   the version of the output file
%       dataFiles:          cell array of file names for importing
%   output (required):
%       bSuccess:   indicates whether conversion was succesfull
%       dataStruct: the finished structure to be used by Sucrose Analysis
%
%See also: applyUserConversion.m, createDataStructDefinition.m,
%verifyDataStructure.m

%#ok<*AGROW>

%default values for return parameters
bSuccess = false;
dataStruct = [];

%basic checks
if nargin ~= 2
    return;
end

%{
    First, create an empty data structure based on the container version
    supplied to this function. As the software develops, the data structure
    will change. This will cause the list of defintions provided by
    createDataStructDefintion to grow as well. The definition will always be
    equal to the version used by the analysis program.
%}
dataStruct = createDataStructDefinition(fileContainerVer);
if isempty(dataStruct)
    %unsupported data container format
    appendSucroseLog('Unsupported data container format', 'urgent');
    return;
end

%check the format of dataFiles
if isempty(dataFiles) || ~iscell(dataFiles) 
    appendSucroseLog('No valid files supplied', 'urgent');
    return;
end

%{
    for each file use the appropriate load function to collect the data. Note
    that in this example only abf files are processed using the abfload
    function from the file exchange. If you wish to support more than 1 file
    type, expand the case list with the correct file extension (including
    the period!) for each supported file type. Otherwise, create a specifc
    conversion function for each file type.
%}
for f = 1:numel(dataFiles)
    %test if the data file exists
    file = dataFiles{f};
    if exist(file, 'file') ~= 2
        %it does not exist so skip
        continue;
    end
    
    %{
        Set the insertion index according to the current size. When an
        empty data set is generated, the number of elements in dataStruct
        is 1, but the filename field will be empty. If this is the case,
        then the insertion index should be 1, otherwise it should be the
        last index + 1.
    %}
    if isempty(dataStruct(1).filename)
        insertIdx = 1;
    else
        insertIdx = numel(dataStruct) + 1;
    end
    
    %split the file name into its components
    [path, file, ext] = fileparts(file);
    
    switch(ext)
        case '.abf'
            %{
                load the file using abfload and retrieve the data, sample
                interval and the header. In the case of abfload, all
                required information about gain and recording units can
                be extracted from the header information. Make sure
                there is only one column for the data in the final
                structure, as multiple sweeps are not supported by the
                analysis program at this time.
            %}
            try
                [data, sample_int, hdr] = abfload(fullfile(path, [file ext]));
                if sample_int > 0
                    %convert to seconds
                    sample_int = sample_int/1e6;
                end
                if size(data, 2) > 1
                    data = data(:,1);
                end
            catch me
                disp(getReport(me, 'basic'));
                errordlg({'The function abfload was not on the matlab search path.'; 
                          'Please download it from the matlab fileexchange or add';
                          'it to the search path.'}, 'Data import failed');
                appendSucroseLog('Function abfload was missing', 'urgent');
                return;
            end
            
            try
                %{
                    retrieve the scaling info from the header: the program
                    expects the data to be expressed in pA so for some files a
                    scaling factor is required. Such a list can be expanded
                    if necessary to support more scalings. The program
                    expects that data is recorded in voltage-clamp mode,
                    meaning the data is measured in amperes
                %}
                scalingVals = unique(hdr.recChUnits);
                idx = regexp(scalingVals, '.A', 'once');
                idx = ~cellfun('isempty', idx);
                idx = find(idx, 1, 'first');
                scalingVal = scalingVals{idx};
                
                switch(scalingVal)
                    case 'pA'
                        scaleFactor = 1;
                    case 'nA'
                        scaleFactor = 1e3;
                    otherwise
                        %no valid scaling found, skip file
                        continue;
                end
                
                %{
                    retrieve the gain value from the header: the data should
                    be set at gain level 1, so any other value requires
                    correction
                %}
                signalGain = hdr.fSignalGain;
                signalGain = signalGain(1);
            catch me2
                disp(getReport(me2, 'basic'));
                errordlg({'The function abfload returned an incomplete or bad header.';
                          'Please make sure your abf file is not damaged.'},...
                          'Data import failed');
                appendSucroseLog('Header information was incorrect', 'urgent');
                return;
            end
            
            %{
                process the data according to the protocol used for
                acquisition: in this case the data is split into two equal
                parts and the block is defined between a quarter and half the
                duration of each data part. Values for these parameters can be
                hard-coded or defined via input dialogs. This is just an
                example of how to load and convert the data. For fitting,
                please ensure that the values for the blocks (start time
                and duration) are set correctly.
            %}
            data250 = (data(1:end/2)*scaleFactor)/signalGain;
            data500 = (data(end/2+1:end)*scaleFactor)/signalGain;
            
            %create the filter data
            filterData250 = false(numel(data250),1);
            filterData250(floor(numel(data250)*0.25):floor(numel(data250)*0.50)) = true;
            filterData500 = false(numel(data500),1);
            filterData500(floor(numel(data500)*0.25):floor(numel(data500)*0.50)) = true;
            
            %{
                update the filenames to include the necessary marker at the
                end for sucrose analysis to enable linking. Linking of
                files is done by comparing the filename before the final 4
                digits: identical names will be assumed to belong to a
                linked file. The method used here assures that when using a
                single file for both conditions, a unique name is generated
                which can be used for linking.
            %}
            linkfile = [file 'a_0002'];
            file     = [file 'a_0001']; 
            
            %{
                Fill in the structure for submax. In this example, the
                rawdataCorr field uses the same data as the rawdata field.
                The purpose of the rawdataCorr field is to store data
                corrected for baseline leak and baseline drift. Ideally the
                data for fitting should start at 0.
            %}
            dataStruct(insertIdx).filename           = [file ext];
            dataStruct(insertIdx).linkedFile         = {[linkfile ext]};
            dataStruct(insertIdx).pathname           = path;
            dataStruct(insertIdx).groupname          = '250mM';
            dataStruct(insertIdx).genotype           = 'WT';
            dataStruct(insertIdx).concentration      = 250;
            dataStruct(insertIdx).rawdata            = data250;
            dataStruct(insertIdx).rawdataCorr        = data250;
            dataStruct(insertIdx).nrOfBlocks         = 1;
            dataStruct(insertIdx).sample_int         = sample_int;
            dataStruct(insertIdx).gain               = signalGain;
            dataStruct(insertIdx).filters.blck_fltrs = filterData250;
            
            %fill in the structure for 500
            dataStruct(insertIdx+1).filename           = [linkfile ext];
            dataStruct(insertIdx+1).linkedFile         = {[file ext]};
            dataStruct(insertIdx+1).pathname           = path;
            dataStruct(insertIdx+1).groupname          = '500mM';
            dataStruct(insertIdx+1).genotype           = 'WT';
            dataStruct(insertIdx+1).concentration      = 500;
            dataStruct(insertIdx+1).rawdata            = data500;
            dataStruct(insertIdx+1).rawdataCorr        = data500;
            dataStruct(insertIdx+1).nrOfBlocks         = 1;
            dataStruct(insertIdx+1).sample_int         = sample_int;
            dataStruct(insertIdx+1).gain               = signalGain;
            dataStruct(insertIdx+1).filters.blck_fltrs = filterData500;
        otherwise
            %skip this file
            continue;
    end
    
end

%finished with the data structure
if ~isempty(dataStruct)
    bSuccess = true;
end

end

