function [bVerified] = verifyDataStructure(dataStruct, fcv)
%VERIFYDATASTRUCTURE function for verifying the structure of the data for
%sucrose analysis
%   This function determines whether the supplied data structure matches
%   the structure definition of file container version fcv

%default return value
bVerified = false;

%basic checks
if nargin ~= 2
    return;
end

correctStruct = createDataStructDefinition(fcv);

%start comparing the structure to see if all the required fields are
%present
correctFields = sort(fieldnames(correctStruct));
dataFields    = sort(fieldnames(dataStruct));

%check the number of fields
if numel(correctFields) ~= numel(dataFields)
    %different, problem
    appendSucroseLog('Incorrect number of fields', 'critical');
    return;
end

%do a string compare to see if the fields match up
if ~all(strcmp(correctFields, dataFields))
    appendSucroseLog('Incorrect fieldnames were present', 'critical');
    return;
end

%check if any of the fields is a struct
correctStructFields = cellfun(@(C) isstruct(correctStruct.(C)), correctFields);
dataStructFields    = cellfun(@(C) isstruct(dataStruct(1).(C)), dataFields);

if sum(correctStructFields) ~= sum(all([correctStructFields, dataStructFields], 2))
    %the structures present are not the same fieldnames
    appendSucroseLog('Incorrect sub-structure definitions', 'critical');
    return;
end

%retrieve the fields that contain structures
structFieldsCrct = correctFields(correctStructFields);

%check if the sub-structures are equal
if numel(structFieldsCrct) > 0
    %loop over the sub-structures
    for fld = 1:numel(structFieldsCrct)
        fldName = structFieldsCrct{fld};
        crctFldnames = sort(fieldnames(correctStruct.(fldName)));
        dataFldNames = sort(fieldnames(dataStruct(1).(fldName)));
        
        %compare the number of fields
        if numel(crctFldnames) ~= numel(dataFldNames)
            appendSucroseLog('Incorrect sub-structure definition', 'critical');
            return;
        end
        
        %compare the field names
        if ~all(strcmp(crctFldnames, dataFldNames))
            appendSucroseLog('Incorrect sub-structure fieldnames were present', 'critical');
            return;
        end
    end
end

%everything checks out, return true
bVerified = true;

end

