function export2XL(dataStruct, statStruct, fileVersion, versionNr, xlFile)

hMain    = getappdata(0, 'h_sucrose');
settings = getappdata(hMain, 'settings');

genotypes = unique({dataStruct.genotype});
colHdrs   = {'','';
            'File name', 'Sucrose concentration (mM)'};
blockDef  = {'Block', '','','','','','','','',;
            'Priming rate k1 (1/s)','Unpriming rate k-1 (1/s)','Release rate k2max (1/s)','Onset delay tdel (s)', 'Rise time tau (s)', ...
            'Depot pool D (pC)','Primed pool R (pC)','Cumulative release (pC)','Cumulative refill (pC)'};

maxNumBlock = max([dataStruct.nrOfBlocks]);
colHdrs      = [colHdrs repmat(blockDef, 1, maxNumBlock)];         
numPar = size(blockDef,2);

try
    for g = 1:numel(genotypes)
        
        genotypeFltr = strcmp({dataStruct.genotype},genotypes(g));
        ds           = dataStruct(genotypeFltr);
        filenames    = {ds.filename};
        tabledata    = cell(length(ds), numPar*maxNumBlock);
        rowHdrs      = [filenames' num2cell([ds.concentration])'];
        
        maxSucrose = max([ds.concentration]);
        
        for i = 1:length(ds)
            
            ba = ds(i).blockAnalysis;
            fltr = ds(i).filters.blck_fltrs;
            si = ds(i).sample_int;
            numBlock     = numel(ba);
            conc = ds(i).concentration;
            
            for b = 1:numBlock
                
                parameters = ba(b).parameters_extended; %Get the fitted parameters
                
                if isempty(parameters)
                    parameters = ba(b).parameters;
                end
                
                if isempty(parameters)
                    continue;
                end
                
                tFitStart = (find(fltr(:,b),1,'first')-1)*si; %Convert index to seconds
                tFitEnd = (find(fltr(:,b),1,'last')-1)*si; %Convert index to seconds
                time = tFitStart:si:tFitEnd;
                
                %pulseLength = tFitEnd - tFitStart;
                if conc < maxSucrose
                    if ~isempty(ds(i).linkedFile)
                        linkedFile = ds(i).linkedFile; %File is linked with maxSucrose only
                        linkIdx = find(strcmp(linkedFile, filenames),1,'first');
                        linkedLinkedFiles = ds(linkIdx).linkedFile; %All linked files for maxSucrose of current cell
                        cnt = find(strcmp(ds(i).filename, linkedLinkedFiles));
                        
                        curr_pars = parameters([1:2, (3:5)+3*((numel(linkedLinkedFiles)+1) - cnt), end-1:end]);
                        
                        tabledata(i,(1:numPar-2)+(numPar*(b-1))) = num2cell(curr_pars); %Fill up the table with parameters
                        
                        %Calculate the charge due to the fitted release and the refill
                        k2_currConc = sucrose_sigmoid(time, tFitStart, tFitEnd, settings.k20,...
                            curr_pars(3), curr_pars(4), curr_pars(5), settings.sucrose_decay);
                        [~, release, states] = simulate_sucrose(time, curr_pars(end-1:end), curr_pars(1:5), k2_currConc);
                        refill = -(curr_pars(1).*states(:,1) - curr_pars(2).*states(:,2));
                        
                        cumulRelease = -sum(release)*si; %Total fitted release
                        cumulRefill = -sum(refill)*si; %Total fitted refill
                        
                        tabledata(i,(numPar-1:numPar)+(numPar*(b-1))) = num2cell([cumulRelease cumulRefill]);
                    end
                else
                    tabledata(i,(1:numPar-2)+(numPar*(b-1))) = num2cell(parameters([1:2, 3:5, end-1:end]));
                end
                %tabledata(i,(numPar-3:numPar-2)+(numPar*(b-1))) = num2cell([tFitStart pulseLength]);
            end
        end
        
        xlSheet = ['par_' genotypes{g}]; %Sheet name depends on genotype
        tabledata = sortrows([rowHdrs tabledata], 2); %Sort data by concentration
        outData = [colHdrs; tabledata];
        
        %Save data to file
        if ispc()
            %use MS Excel
            xlswrite(xlFile,outData,xlSheet);
        else
            hMain = getappdata(0, 'h_sucrose');
            settings = getappdata(hMain, 'settings');
            poi_path = settings.poi_path;
            if isempty(poi_path)
                %ask the user for the folder containing the poi library
                poi_path = uigetdir('', 'Select the Apache POI directory');
                if poi_path == 0
                    appendSucroseLog('Failed to export the data using Apache POI', 'critical');
                    return;
                else
                    %store the POI path for future reference
                    settings.poi_path = poi_path;
                    setappdata(hMain, 'settings', settings);
                    appendSucroseLog(['Added POI library path ' poi_path], 'info');
                end
            end
            
            if libisloaded(fullfile(poi_path, 'poi-3.8-20120326.jar'))
                %use the xlwrite function
                xlwrite(xlFile,outData,xlSheet);
            else
                %load the libraries of the Apache POI
                javaaddpath(fullfile(poi_path, 'poi-3.8-20120326.jar'));
                javaaddpath(fullfile(poi_path, 'poi-ooxml-3.8-20120326.jar'));
                javaaddpath(fullfile(poi_path, 'poi-ooxml-schemas-3.8-20120326.jar'));
                javaaddpath(fullfile(poi_path, 'xmlbeans-2.3.0.jar'));
                javaaddpath(fullfile(poi_path, 'dom4j-1.6.1.jar'));
                javaaddpath(fullfile(poi_path, 'stax-api-1.0.1.jar'));
                
                %use the xlwrite function
                xlwrite(xlFile,outData,xlSheet);
            end
        end
        
    end 
catch me
    appendSucroseLog(me, 'urgent');
    appendSucroseLog(['Debug ' getLineNumber()], 'urgent');
end

%Save data to file
xlSheet = 'Info';
colHdrs   = {'Software version', '', '';
    'Data structure version', '', '';
    '', '', '';
    'File name', 'Sucrose concentration (mM)', 'Genotype'};
blockDef  = {'Block', '', '', '', '', '', '', '';
    'Fitting method', 'Fit start (s)', 'Sucrose pulse (s)', 'Baseline current (pA)', 'Baseline slope (pA/s)', ...
    'Noise level (pA)', 'Peak current (pA)', 'Fit quality (pA)'};

temp = [cell(2, size(repmat(blockDef, 1, maxNumBlock),2));
    repmat(blockDef, 1, maxNumBlock)];
temp(1:2,1) = num2cell([fileVersion; versionNr]);%Store software version and data structure version
colHdrs      = [colHdrs temp];

numInfoPar = size(blockDef,2);
tabledata    = cell(length(dataStruct), numInfoPar*maxNumBlock);

filenames    = {dataStruct.filename};
genotypes    = {dataStruct.genotype};
rowHdrs      = [filenames' num2cell([dataStruct.concentration])' genotypes'];

baseLeak     = statStruct(1).baselineLeak;
baseSlope    = statStruct(1).baselineSlope;
noiseLvl     = statStruct(1).noiseLevel;
peakCurr     = statStruct(1).peakLevel;
fitQual      = statStruct(1).fitQuality;

for i = 1:length(dataStruct)
    
    ba       = dataStruct(i).blockAnalysis;
    numBlock = numel(ba);
    si       = dataStruct(i).sample_int;
    fltr     = dataStruct(i).filters.blck_fltrs;
    
    for b = 1:numBlock
        
        % Current block contains no fitting data
        if isempty(ba(b).cost)
            continue;
        end
        
        tFitStart   = (find(fltr(:,b),1,'first')-1)*si; %Convert index to seconds
        tFitEnd     = (find(fltr(:,b),1,'last')-1)*si; %Convert index to seconds
        pulseLength = tFitEnd - tFitStart;
        
        tabledata(i,1+(numInfoPar*(b-1))) = {ba(b).label}; %Fitting method
        tabledata(i,2+(numInfoPar*(b-1))) = num2cell(tFitStart);
        tabledata(i,3+(numInfoPar*(b-1))) = num2cell(pulseLength);
        tabledata(i,4+(numInfoPar*(b-1))) = num2cell(baseLeak(i,b));
        tabledata(i,5+(numInfoPar*(b-1))) = num2cell(baseSlope(i,b));
        tabledata(i,6+(numInfoPar*(b-1))) = num2cell(noiseLvl(i,b));
        tabledata(i,7+(numInfoPar*(b-1))) = num2cell(peakCurr(i,b));
        tabledata(i,8+(numInfoPar*(b-1))) = num2cell(fitQual(i,b));
    end
end

tabledata = sortrows([rowHdrs tabledata], 2); %Sort data by concentration
outData = [colHdrs; tabledata];

if ispc()
    %use MS Excel to write data
    xlswrite(xlFile,outData,xlSheet);
else
    %use Apache POI to write data
    xlwrite(xlFile,outData,xlSheet);
end

end