function exportFitData(dataStruct, filename, type)
%EXPORTFITDATA export data from sucrose analysis and store it in the
%provided file of certain type

if isempty(dataStruct)
    return;
end

if nargin == 2
    %try to discover what the extension is from the filename
    [~, ~, ext] = fileparts(filename);
    
    switch(ext)
        case {'.xlsx', '.xls'}
            type = 1;
        case {'.csv', '.txt'}
            type = 2;
        otherwise
            return;
    end
    %type should now be resolved
end

%collect the data for all types
tabledata = collectFormatData(dataStruct);


%store the data according to the file format
switch(type)
    case 1
        %Excel file, cell arrays can be stored directly
        for f = 1:numel(dataStruct)
            sheetname = [dataStruct(f).filename '_' dataStruct(f).groupname];
            try
                if ispc()
                    xlswrite(filename, tabledata{f,:}, sheetname);
                else
                    if libisloaded()
                        %use the poi library to write the data to Excel
                        xlwrite(filename, tabledata{f,:}, sheetname);
                    else
                        
                    end
                end
            catch me
                appendSucroseLog(me, 'critical');
                return;
            end
        end
        
    case 2
        %delimited text file
        try
            fid = fopen(filename, 'a');
            for f = 1:numel(dataStruct)
                sheetname = [dataStruct(f).filename '_' dataStruct(f).groupname];
                %write the filename
                fprintf(fid, '%s\n', sheetname);
                
                %write the data using the correct format
                blockFormat = ['%s' repmat(',%d', 1, size(tabledata{f,:}, 2)-1), '\n'];
                fprintf(fid, blockFormat, tabledata{f,:})
            end
            %finished writing
            fclose(fid);
        catch me
            appendSucroseLog(me, 'critical');
            %try to close the file if necessary
            if ~isempty(fid)
                fclose(fid);
            end
            
            return;
        end
        
    otherwise
        return;
end


end


function [tabledata] = collectFormatData(dataStruct)
%this helper function collects all data from the structure and formats it into
%a simple table for writing to a specified file
%   tabledata is a cell array with the data for each file along dimension 1
%   and the data for each block along dimension 2. Each cell has a cell
%   array consisting of 2 columns: the first column contains the row headers and
%   the second column contains the data values.


%collect some basic data
numFiles  = numel(dataStruct);
numBlocks = [dataStruct.nrOfBlocks];

%check if any of the files deviates in the number of blocks
numUqBlocks = unique(numBlocks);

if numel(numUqBlocks) > 1
    %different numbers of blocks within this data set
    tabledata = cell(numFiles, max(numUqBlocks));
else
    %only one number of blocks for all files: easy case
    %get a list of all the parameters
    tabledata = cell(numFiles, numUqBlocks);
end

%loop over all files
for f = 1:numFiles
    %for each file, loop over the blocks
    for b = 1:numBlocks(f)
        %loop over all fields and create a table of the results
        parFields = fieldnames(dataStruct(f).blockAnalysis(b));
        fieldData = [];
        rowHdr    = {};
        for p = 1:numel(parFields)
            fData = dataStruct(f).blockAnalysis(b).(parFields{p});
            %check if this is a string
            if ischar(fData)
                continue;
            end
            
            %convert to column
            fData = fData(:);
            
            %set the header data
            switch(parFields{p})
                case 'parameters'
                    par = {'k1' 'k-1' 'k2' 'onset' 'tau'};
                    rowHdr = [rowHdr; par(:)];
                otherwise
                    %by default add the fieldname itself
                    rowHdr = [rowHdr; {parFields{p}}];
            end
            
            if numel(fData) == 1
                fieldData = [fieldData; fData];
            else
                %add the data column at the end
                if size(fData, 2) == 1
                    %only 1 column, add data
                    fieldData = [fieldData; fData(:)];
                else
                    %this will not work, skip
                    continue;
                end
            end
        end
        if b == 1
            %store the row headers only in the first block
            tabledata(f,b) = {rowHdr, num2cell(fieldData)};
        else
            tabledata(f,b) = {num2cell(fieldData)};
        end
    end
end

end
