function [fighandle axeshandle results barhandles errorb_handle] = makeBarPlotSucrose(data,SEM,groupnames,y_label,fighandle,ShowIndividualValues,cmap,p,n)
%makeBarPlot creates a nice-looking bar plot with errorbars. 
%inputs:
%-data: vector of averages, or cell arrays of raw data:
%-SEM: vector of standard errors. Will be ignored if data is a cell array
%-groupnames: cell array of strings
%-y_label: string for the y-axis
%-fighandle: handle to figure or axes in which the data should be plotted
%-showIndividualValues: boolean indicating whether individual values should
%be plotted. 
%-cmap: n-by-3 matrix with the colormap of the bars.
%-p: statistical analysis between the two groups (Arthur's method), or
%request to do so. 

%Arthur de Jong
%VU University Amsterdam,2012


%% check input arguments and define variables
error(nargchk(1,9,nargin))

no_items = length(data);

if nargin == 1
    groupnames = num2str((1:no_items)');
    SEM        = [];    
    y_label    = 'joepie';
elseif nargin == 3
    y_label    = 'aha';
end

if isempty(groupnames)
    groupnames = num2str((1:no_items)');
end

if iscell(data) %rawdata provided, calculate averages...
    average = cellfun(@nanmean,data);
    STD     = cellfun(@nanstd,data);
    n       = cellfun(@(x) length(find(~isnan(x))),data);
    SEM     = STD./(n.^0.5);
else
    average = data;
end

%check if length of input arguments does not exceed max of the function,
%and is the same for all arguments
l(1) = length(average);
l(2) = length(SEM);
l(3) = length(groupnames);

if l(1) > 8
    error('maximum number of groups is 8')
end

if length(unique(l)) ~= 1
    error('average, SEM and groupnames should have the same number of elements');
end

%define colormap
if  exist('cmap','var') ~=1 || isempty(cmap)
    cmap = ArthursCmap(l(1));
end
%cmap = [0.7 0.7 0.7; 0.2 0.2 0.2];

%check if we should perform statistics
if exist('p','var') && isa(p,'char') && l(1) > 1
    if ~iscell(data)
        error('statistics can only be performed on raw data');
    else
        
        %gather groups
        statsgroupNames = [];
        for i = 1:l(1)
            if isempty(groupnames)
                name = num2str(i);
            else
                name  = groupnames(i);
            end
            name       = repmat(name,length(data{i}),1);
            statsgroupNames = vertcat(statsgroupNames,name); %#ok<AGROW>
        end
        rawdata = vertcat(data{:});
        
        switch p
            case 'kruskalwallis'
                %perform KW test:
                [p_all, ~, stats] = kruskalwallis(rawdata,statsgroupNames,'off');
            case 'anova'
                %check for normality:
                h_lilly    = 0;
                p_lillyMax = 0;
                p_lillyMin = 1;
                
                s = warning('off','stats:lillietest:OutOfRangeP');
                
                skipAnova = false;
                
                for i = 1:l(1)
                    %get the raw data of the group
                    dataIn     = data{i};
                    
                    %check if data meets requirements for lillitest
                    if length(dataIn) < 4
                        warning(s)
                        h_warn = warndlg( 'Too few observations to test for normality. ANOVA cannot not performed.'); 
                        uiwait(h_warn)
                        skipAnova = true;
                        break
                    end
                    
                    %perform lillitest
                    [h_l p_l] = lillietest(dataIn,0.05);
                    
                    
                    %find largest p and h sofar:
                    h_lilly    = max(h_lilly,h_l);
                    p_lillyMin = min(p_lillyMin,p_l);
                    p_lillyMax = max(p_lillyMax,p_l);
                end
                
                warning(s)
                
                if ~skipAnova
                    if h_lilly ~= 0
                        %at least one group does not have a normal distribution
                        disp(['WARNING: data is not normally distributed: p(min, Lillifors) = ' ...
                            num2str(p_lillyMin) '.'] );
                    else
                        %all groups are normally distributed.
                        disp(['Data is normally distributed: p(max, Lillifors) = ' ...
                            num2str(p_lillyMax) '.']);
                    end
                    
                    
                    %perform 1-way ANOVA:
                    [p_all, ~, stats] = anova1(rawdata,statsgroupNames,'off');
                else
                    p_all = 1;
                end
            otherwise
                error('unknown statistical test requested')
        end
        
        %check if the groups are different:
        if p_all > 0.05 || isnan(p_all)
            p = [];
            
        else
            
            %create default output array:
            %p = ones(3,n_groups);
            
            %perform pairwise tests:
            alpha = [0.05, 0.01, 0.001];
            p     = [];
            
            for i = 1:length(alpha)
                c = multcompare(stats,'alpha',alpha(i),'display','off');
                
                [a ~] = size(c);
                
                %find the significant ones:
                for j = 1:a
                    %select which groups were compared:
                    p(1,j) = c(j,1); %#ok<AGROW>
                    p(2,j) = c(j,2); %#ok<AGROW>
                    
                    if (c(j,3) < 0 && c(j,5) < 0) || (c(j,3) > 0 && c(j,5) > 0)
                        %this comparison was significant at this alpha, so store alpha
                        p(3,j) = alpha(i); %#ok<AGROW>
                    elseif i == 1
                        p(3,j) = 1;
                    end
                end
            end
        end
    end
else
    if exist('p','var') && ~isempty(p) && ~ischar(p)
        %keep p
    else
        p = [];
    end
end



%% make bar graphs
%make a figure window where the axes is always at the same absolute
%position (ensures that bars are of equal size

barwidth   = 70; %width of every bar in graph, in pix
leftmargin = 70; %space between Y-axis and left-edge of figure, in pix. def = 200
figpos     = [560 520 leftmargin+barwidth*no_items,300];
left       = 0.9*leftmargin/figpos(3);
width      = (barwidth*no_items)/figpos(3);
bottom     = 0.08;
height     = 0.84;
axespos    = [left bottom width height];
barhandles = zeros(no_items,1);

%create figure and axes:
if nargin >= 5 && isempty(fighandle) == 0
    if strcmp('axes',get(fighandle,'Type'))
        %user supplied an axes
        h_axes = fighandle;
        h      = [];
    else
        %user supplied an figuer
        h = figure(fighandle);
        delete(get(h,'children'))
        h_axes  = axes('Position', axespos,'Units','Pixels');
    end
else
    %user didn't supply anything.
    fighandle = figure('Position', figpos,'ResizeFcn',@makeBarPlotResizeFcn);
    h       = fighandle;
    h_axes  = axes('Position', axespos,'Units','Pixels');
end

if ~strcmp('axes',get(fighandle,'Type'))
    %save handles and AxesMargins in userdata (needed for ResizeFcn)
    axespos               = get(h_axes,'Position');
    u.p_axesMargins(1:2)  = axespos(1:2);
    u.p_axesMargins(3)    = figpos(3) - axespos(1) - axespos(3);
    u.p_axesMargins(4)    = figpos(4) - axespos(2) - axespos(4);
    u.h_axes  = h_axes;
    set(h,'Userdata',u);
end

%plot bars:
for i = 1:no_items
    barhandles(i) = bar(h_axes,i,average(i),...
                    'faceColor', [cmap(i,1) cmap(i,2) cmap(i,3)],...
                    'EdgeColor','none');
    hold(h_axes,'on')
end

%set labels on x-axis
set(h_axes,...
    'XTick',1:no_items,...
    'XTickLabel',groupnames)
%'Position',  [0.1300    0.1100    0.7750    0.8150])

%plot errorbars
if ~isempty(SEM)
    %{
    errorb_handle = errorbar(h_axes,1:no_items,average,zeros(1,no_items),SEM,...
        'lineStyle','none',...
        'marker','none',...
        'Color','k');
    %}
    errorb_handle = errorbar(h_axes,1:no_items,average,SEM,SEM,...
        'lineStyle','none',...
        'marker','none',...
        'Color','k');
    %make top errorbars longer, remove the others
    %
    children               = get(errorb_handle,'children');
    errorbarXData          = get(children(2),'XData');
    errorbarXData(4:9:end) = errorbarXData(1:9:end) - 0.05;
    errorbarXData(7:9:end) = errorbarXData(2:9:end) - 0.05;%0;
    errorbarXData(5:9:end) = errorbarXData(1:9:end) + 0.05;
    errorbarXData(8:9:end) = errorbarXData(2:9:end) + 0.05;%0;
    set(children(2),'XData',errorbarXData);
    %}
end

if nargin >= 6 && ~isempty(ShowIndividualValues) && ShowIndividualValues
    for i = 1:no_items
        plot(h_axes,i+0.3,data{i},'ok','markersize',3,'linewidth',0.75,'markerfacecolor','k')
    end
end

%create legend
%legend(h_axes,groupnames,'Location','Best')

%show statistics, if required
if ~isempty(p)
    %define stepsize
    yLimits  = get(h_axes,'yLim');
    stepsize = 0.04*(yLimits(2) - yLimits(1));
    currentStep = yLimits(2);
    if sign(yLimits(1)) == -1
        stepsize    = stepsize*-1;
        currentStep = yLimits(1);
    end
    
    %go over every test, and plot it
    [~,n_tests] = size(p);
    for i = 1:n_tests
        %plot the line:
        currentStep = currentStep + stepsize;
        plot(h_axes,[p(1,i) p(2,i)],[currentStep currentStep],'Color','k');
        
               
        %calculate position of text:
        currentStep = currentStep + stepsize;
        x = (p(2,i) - p(1,i))/2 + p(1,i);
        
        %select text string:
        switch p(3,i)
            case 0
                str = 'ns';
                fs  = 8;
            case 1
                str = 'ns';
                fs  = 8;
            case 0.05
                str = '*';
                fs  = 12;
            case 0.01
                str = '**';
                fs  = 8;
            case 0.001
                str = '***';
                fs  = 8;
        end
        
        %shox text:
        text(x,currentStep, str,...
            'Parent',h_axes,...
            'HorizontalAlignment','center',...
            'FontSize',fs);
    end
    
    %reset ylim:
    if sign(yLimits(1)) == -1
        ylim(h_axes,[currentStep + 2*stepsize, yLimits(1)]);
    else
        ylim(h_axes,[yLimits(1),currentStep + 2*stepsize]);
    end
        
end

%show number of observations if present
if exist('n','var')
    ylimits = get(h_axes,'ylim');
    ypos    = 0.05 * (ylimits(2) - ylimits(1));
    if sign(ylimits(1)) == -1
        ypos    = ypos*-1;
    end
    
    for i = 1:length(n)
        text(i,ypos,num2str(n(i)),...
            'Color',[1 1 1],...
            'parent',h_axes,...
            'HorizontalAlignment','center');
    end
end

%misc. options
xlim(h_axes,[0.5 no_items+0.5])
ylabel(h_axes,y_label)

%gather output variables:
switch nargout
    case 1
        fighandle = h;
    case 2
        fighandle = h;
        axeshandle = h_axes;
    case {3,4,5}
        fighandle       = h;
        axeshandle      = h_axes;
        results.average = average;
        results.STD     = STD;
        results.n       = n;
        results.SEM     = SEM;
            
end
end

function makeBarPlotResizeFcn(hObject, ~, ~)
u = get(hObject,'Userdata');

old = get(hObject,'Units');

set(hObject,'Units','Pixels');
set(u.h_axes,'Units','Pixels');

p_fig = get(hObject,'Position');
p_axes(1) = u.p_axesMargins(1);
p_axes(2) = u.p_axesMargins(2);
p_axes(3) = p_fig(3) - u.p_axesMargins(1) - u.p_axesMargins(3);
p_axes(4) = p_fig(4) - u.p_axesMargins(2) - u.p_axesMargins(4); 



set(u.h_axes,'Position',p_axes);

set(hObject,'Units',old);
end