function varargout = sucrose_GUI_v2(varargin)
% SUCROSE_GUI_V2 MATLAB code for sucrose_GUI_v2.fig
%      SUCROSE_GUI_V2, by itself, creates a new SUCROSE_GUI_V2 or raises the existing
%      singleton*.
%
%      H = SUCROSE_GUI_V2 returns the handle to a new SUCROSE_GUI_V2 or the handle to
%      the existing singleton*.
%
%      SUCROSE_GUI_V2('CALLBACK',hObject,eventData,handles,...) calls the local
%      function named CALLBACK in SUCROSE_GUI_V2.M with the given input arguments.
%
%      SUCROSE_GUI_V2('Property','Value',...) creates a new SUCROSE_GUI_V2 or raises the
%      existing singleton*.  Starting from the left, property value pairs are
%      applied to the GUI before sucrose_GUI_v2_OpeningFcn gets called.  An
%      unrecognized property name or invalid value makes property application
%      stop.  All inputs are passed to sucrose_GUI_v2_OpeningFcn via varargin.
%
%      *See GUI Options on GUIDE's Tools menu.  Choose "GUI allows only one
%      instance to run (singleton)".
%
% See also: GUIDE, GUIDATA, GUIHANDLES

% Edit the above text to modify the response to help sucrose_GUI_v2

% Last Modified by GUIDE v2.5 01-Jul-2014 11:59:29

% Begin initialization code - DO NOT EDIT
gui_Singleton = 1;
gui_State = struct('gui_Name',       mfilename, ...
                   'gui_Singleton',  gui_Singleton, ...
                   'gui_OpeningFcn', @sucrose_GUI_v2_OpeningFcn, ...
                   'gui_OutputFcn',  @sucrose_GUI_v2_OutputFcn, ...
                   'gui_LayoutFcn',  [] , ...
                   'gui_Callback',   []);
if nargin && ischar(varargin{1})
    gui_State.gui_Callback = str2func(varargin{1});
end

if nargout
    [varargout{1:nargout}] = gui_mainfcn(gui_State, varargin{:});
else
    gui_mainfcn(gui_State, varargin{:});
end
% End initialization code - DO NOT EDIT

%#ok<*DEFNU,*INUSL,*AGROW,*INUSD,*NASGU>


% --- Executes just before sucrose_GUI_v2 is made visible.
function sucrose_GUI_v2_OpeningFcn(hObject, eventdata, handles, varargin)
% This function has no output args, see OutputFcn.
% hObject    handle to figure
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
% varargin   command line arguments to sucrose_GUI_v2 (see VARARGIN)

% Choose default command line output for sucrose_GUI_v2
handles.output = hObject;

%set up the basic parameters for the interface

%major version
%   0:  initial design
majorVer = 0;

%minor version
%   65: conversion now returns the dataStruct via appdata
%   66: bug fixes for plotting and legend
%   67: simplified conversion
%   68: simplified conversion
%   70: minor bug fix for release
minorVer = 0.70;

versionNr = majorVer + minorVer;

%build number
%   0714.04:    tested conversion and plotting of converted data
%   0714.05:    tested simplified data conversion
%   0714.06:    updated export->plot function
%   0714.07:    fixed error in help file
%   0714.08:    minor bug fixes
buildNr = '0714.08';

%start building the appdata structure
setappdata(0, 'h_sucrose', gcf);    %store the handle of the current fig on workspace 0
setappdata(gcf, 'versionNr', versionNr);
setappdata(gcf, 'buildNr', buildNr);
%this is used for compatibility: an increase in this value will prohibit
%loading files that are made with a different file format.
setappdata(gcf, 'fileContainerVersion', 3.4);
setappdata(gcf, 'dataStruct', []);
setappdata(gcf, 'statStruct', []);
setappdata(gcf, 'fixed_parameters', []);
setappdata(gcf, 'handles', handles);

%set the name of the application
set(gcf, 'Name', ['Sucrose analysis :: version ' num2str(versionNr) ' (build ' buildNr ')']);

%get the important root locations
appRoot = strrep(mfilename('fullpath'), mfilename, '');
docRoot = fullfile(appRoot, 'doc');
conRoot = fullfile(appRoot, 'conversion');
logfile = fullfile(appRoot, 'log_sucrose.log');

%get the current search path
if isempty(regexp(path, conRoot, 'once'))
    %not found, add the conversion to the search path
    path(conRoot, path);
end

%and store them in appdata
setappdata(gcf, 'appRoot', appRoot);
setappdata(gcf, 'docRoot', docRoot);
setappdata(gcf, 'sessionfile', '');
setappdata(gcf, 'logfile', logfile)

%get the current settings from a file or select the default settings
settings = getSettings();
setappdata(gcf, 'settings', settings);

% Update handles structure
guidata(hObject, handles);

% UIWAIT makes sucrose_GUI_v2 wait for user response (see UIRESUME)
% uiwait(handles.figure1);


function [settings] = getSettings()
h = getappdata(0, 'h_sucrose');
appRoot = getappdata(h, 'appRoot');

settingsFile = fullfile(appRoot, 'sucrose_settings.mat');

if exist(settingsFile, 'file') == 2
    %there is a settings file, load it
    settings = loadSettings(settingsFile);
else
    %no settings, get the default values
    settings = getDefaultSettings();
    %and store them
    saveSettings(settings);
end


function [settings] = loadSettings(file)
try
    in = load(file);
    if isfield(in, 'settings')
        %merge with the default settings to make sure all settings are
        %present at runtime but preserve loaded values
        settings = mergeSettingsStructures(in.settings, getDefaultSettings(), true);
    else
        settings = getDefaultSettings();
    end
catch me
    appendSucroseLog(me, 'urgent');
    appendSucroseLog(['Debug ' getLineNumber()], 'urgent');
end

function saveSettings(settings)
h = getappdata(0, 'h_sucrose');
appRoot = getappdata(h, 'appRoot');

if nargin == 0
    settings = getappdata(h, 'settings');
end
settingsFile = fullfile(appRoot, 'sucrose_settings.mat');

%testto see if settings is empty
if isempty(settings)
    %empty: get default values
    settings = getDefaultSettings();
end

%save the settings to file
try
    save(settingsFile, 'settings');
catch me
    appendSucroseLog(me, 'urgent');
    appendSucroseLog(['Debug ' getLineNumber()], 'urgent');
end


function [settings] = getDefaultSettings()
h = getappdata(0, 'h_sucrose');
appRoot = getappdata(h, 'appRoot');

%define the default settings
settings.fixed_params   = false(1,7);
settings.model_type     = 'none';
settings.fitting_order  = 0; %0=500mM first than other concentrations
settings.recentDir      = appRoot;
settings.sucrose_decay  = 0.15;
settings.k20            = 0; %Basal release rate constant
settings.useRandomInit  = false;
settings.globalFitType  = 'default';
settings.outputDisplay  = 'final';  %show the final output
settings.reusePrevious  = true;
%initial values: 
%   1: k1
%   2: k-1
%   3: k2 500
%   4: onset 500
%   5: tau 500
%   6: k2 250
%   7: onset 250
%   8: tau 250
settings.par_init        = [0.063722279	0.127527542	2.92378832	25325.28703	0.197679809	0.093785141	20.37647824	0.484709987];
settings.par_previous    = settings.par_init;
settings.MaxFunEvals     = 400; 
settings.MaxIter         = 400;
settings.pulseLength     = 7;
settings.fitStartBlck    = [4.5 21];
settings.baselineLength  = .1;
settings.gain            = 1;
settings.explore_range   = 0.10;
settings.poi_path        = [];
settings.useExtended     = false;
settings.useWeightedFit  = false;
settings.useFixedParam   = false;
settings.useExperimental = false;
settings.useAdaptiveBase = true;
settings.previousConvert = [];


function [bOK] = saveSession()
%default is false
bOK = false;

h = getappdata(0, 'h_sucrose');
dataStruct  = getappdata(h, 'dataStruct');
statStruct  = getappdata(h, 'statStruct');
recentDir   = getappdata(h, 'recentDir');
sessionfile = getappdata(h, 'sessionfile');
%store the file version to reflect changes in file format and data
%structure to prevent errors during loading
fileVersion = getappdata(h, 'fileContainerVersion');

if isempty(dataStruct)
    %nothing to do here, everything is OK
    bOK = true;
    return;
end

if isempty(sessionfile)
    %ask the user for a file location
    [filename, path, ~] = uiputfile({'*.mat' 'Session file (*.mat)'}, 'Save session file', recentDir);
    if path == 0
        %user canceled
        return;
    else
        %user provided a valid filename
        filename = fullfile(path, filename);
        setappdata(h, 'sessionfile', filename);
    end
else
    filename = sessionfile;
end

%get the data to save, and save it
versionNr  = getappdata(h, 'versionNr');
try
    if ~isempty(dataStruct)
        if isempty(statStruct)
            statStruct = generateStatStructure(true);
        end
        save(filename, 'dataStruct', 'versionNr', 'fileVersion', 'statStruct');
        bOK = true;
    else
        return;
    end
catch me
    appendSucroseLog(me, 'urgent');
    appendSucroseLog(['Debug ' getLineNumber()], 'urgent');
    return
end


function [statS] = generateStatStructure(bExtensive)
hMain = getappdata(0, 'h_sucrose');
dataS = getappdata(hMain, 'dataStruct');
statS = getappdata(hMain, 'statStruct');
settings = getappdata(hMain, 'settings');
maxBlcks = max([dataS.nrOfBlocks]);

if isempty(dataS)
    %no data to process.
    return;
end

if nargin == 0
    bExtensive = false;
end

if bExtensive
    %%% calculate the basic file statistics first (exception)
    numRecords = numel(dataS);
    %initialize arrays
    baselineLeak  = nan(numRecords, maxBlcks);
    baselineSlope = nan(numRecords, maxBlcks);
    noiseLevel    = nan(numRecords, maxBlcks);
    peakLevel     = nan(numRecords, maxBlcks);
    fitQuality    = nan(numRecords, maxBlcks);
    
    for d = 1:numRecords
        %get the raw data
        trace    = dataS(d).rawdata;
        trace_sm = smooth(trace, 2000);
        blckfltrs = dataS(d).filters.blck_fltrs;
        si = dataS(d).sample_int;
        
        for p = 1:dataS(d).nrOfBlocks
            %get the block filter and determine the start and stop index
            blckfltr = blckfltrs(:,p);
            i  = find(blckfltr,1,'first');
            ii = i - floor(settings.baselineLength/si+1);
            
            %check if value is within bounds and calculate the leak
            %accordingly
            if ii <= 0
                ii = i + floor(settings.baselineLength/si+1);
                baselineLeak(d, p)  = mean(trace(i:ii));
            else
                baselineLeak(d, p)  = mean(trace(ii:i));
            end
            
            %adjust the stop index for the slope to extend beyond the fitted
            %response
            ii = find(blckfltr,1,'last');
            
            %calculate the peak and noise level during the pulse duration
            noiseLevel(d, p) = sqrt(sum((trace_sm(i:ii)-trace(i:ii)).^2)/(ii-i));
            peakLevel(d, p)  = min(trace_sm(i:ii));
            
            %for the slope, add 2 seconds to the pulse duration
            ii = ii + 2e4;
            if settings.useAdaptiveBase
                %calculate the slope in pA/sec
                baselineSlope(d, p) = (trace(ii)-trace(i))/((ii-i-1)/si);
            else
                %when adaptive baseline is not used, a baseline slop eof 0
                %is used instead (constant-value baseline)
                baselineSlope(d, p) = 0;
            end
            
            %when a fit is present, calculate the fit quality
            if isfield(dataS(d), 'blockAnalysis') && ~isempty(dataS(d).blockAnalysis)
                fitQuality(d, p) = determineFitQuality(d, p);
            end
        end
        
    end
    %add the values to the struct
    statS(1).baselineLeak  = baselineLeak;
    statS(1).baselineSlope = baselineSlope;
    statS(1).noiseLevel    = noiseLevel;
    statS(1).peakLevel     = peakLevel;
    statS(1).fitQuality    = fitQuality;
else
    baselineLeak  = statS(1).baselineLeak;
    baselineSlope = statS(1).baselineSlope;
    noiseLevel    = statS(1).noiseLevel;
    peakLevel     = statS(1).peakLevel;
    fitQuality    = statS(1).fitQuality;
end

%%% caclulate the mean, sd and SEM for the parameters
statS(2).baselineLeak  = struct('mean', nanmean(baselineLeak,1),...
                                'stdv', nanstd(baselineLeak,[],1),...
                                'sem', nanstd(baselineLeak,[],1)/sqrt(sum(isnan(baselineLeak))),...
                                'num', sum(isnan(baselineLeak)));
statS(2).baselineSlope = struct('mean', nanmean(baselineSlope,1),...
                                'stdv', nanstd(baselineSlope,[],1),...
                                'sem', nanstd(baselineSlope,[],1)/sqrt(sum(isnan(baselineSlope))),...
                                'num', sum(isnan(baselineSlope)));
statS(2).noiseLevel    = struct('mean', nanmean(noiseLevel,1),...
                                'stdv', nanstd(noiseLevel,[],1),...
                                'sem', nanstd(noiseLevel,[],1)/sqrt(sum(isnan(noiseLevel))),...
                                'num', sum(isnan(noiseLevel)));
statS(2).peakLevel     = struct('mean', nanmean(peakLevel,1),...
                                'stdv', nanstd(peakLevel,[],1),...
                                'sem', nanstd(peakLevel,[],1)/sqrt(sum(isnan(peakLevel))),...
                                'num', sum(isnan(peakLevel)));
statS(2).fitQuality    = struct('mean', nanmean(fitQuality,1),...
                                'stdv', nanstd(fitQuality,[],1),...
                                'sem', nanstd(fitQuality,[],1)/sqrt(sum(isnan(fitQuality))),...
                                'num', sum(isnan(fitQuality)));




function [fitQ] = determineFitQuality(idx, blckID)

[trace, ~] = getFitData(idx, []);
trace = trace(:,2);

%no trace data
if all(isnan(trace))
    fitQ = nan;
    return;
end
%only measure during the blocks
%nanfltr = ~isnan(trace);

hMain = getappdata(0, 'h_sucrose');
dataS = getappdata(hMain, 'dataStruct');
blckfltr = dataS(idx).filters.blck_fltrs(:,blckID);
%return the error (standard deviation equivalent)
fitQ = sqrt(sum((dataS(idx).rawdataCorr(blckfltr) - trace(blckfltr)).^2)/sum(blckfltr));


                           
function updateInterface()
h = getappdata(0, 'h_sucrose');
%get all relevant data
handles    = getappdata(h, 'handles');
dataStruct = getappdata(h, 'dataStruct');

%no data, nothing to update
if isempty(dataStruct);
    set(handles.popup_group, 'String', 'Groupname');
    return;
end

%prepare the group and file list controls
groupnames = {dataStruct.groupname};
filenames  = {dataStruct.filename};

%first, update the group box
currGroupIdx  = get(handles.popup_group, 'Value');
currGroupName = get(handles.popup_group, 'String');

if strcmpi(currGroupName, 'groupname')
    %first group, nothing special
    newGroups = unique(groupnames);
    set(handles.popup_group, 'String', newGroups);
    set(handles.list_files, 'String', '');
    currGroupName = newGroups{1};
else
    currGroupName = currGroupName{currGroupIdx};
    if any(strcmp(groupnames, currGroupName))
        %a previous group existed, check for the new idx due to sorting by
        %the unique function
        newGroups = unique(groupnames);
        newIdx    = find(strcmpi(currGroupName, newGroups));
    else
        %the group has disappeared
        newGroups = unique(groupnames);
        newIdx    = 1;
    end
    %set the controls
    set(handles.popup_group, 'String', newGroups);
    set(handles.popup_group, 'Value', newIdx);
    
end

%get the files belonging to the currently selected group
groupfltr = strcmpi(currGroupName, groupnames);
filenames = filenames(groupfltr);
%set the file list
set(handles.list_files, 'String', filenames);
set(handles.list_files, 'Value', 1);
set(handles.text_files, 'String', ['Files (' num2str(numel(filenames)) '):']);

if isfield(dataStruct, 'blockAnalysis')
    if ~all(cellfun('isempty', {dataStruct.blockAnalysis}))
        set(handles.radio_fitOnly, 'Enable', 'on');
        set(handles.radio_fitArea, 'Enable', 'on');
        da = [dataStruct.blockAnalysis];
        if ~all(isempty([da.parameters_extended]))
            set(handles.check_extended, 'Enable', 'on');
        else
            set(handles.check_extended, 'Enable', 'off');
        end
    else
        set(handles.radio_fitOnly, 'Enable', 'off');
        set(handles.radio_fitArea, 'Enable', 'off');
        set(handles.check_extended, 'Enable', 'off');
    end
else
    set(handles.radio_fitOnly, 'Enable', 'off');
    set(handles.radio_fitArea, 'Enable', 'off');
    set(handles.check_extended, 'Enable', 'off');
end

%update the stat structure and update the plots
statS = generateStatStructure(true);
setappdata(h, 'statStruct', statS);
updatePlot();


function updatePlot(level, bNew)
h = getappdata(0, 'h_sucrose');
%get relevant data
dataStruct = getappdata(h, 'dataStruct');
handles    = getappdata(h, 'handles');
settings   = getappdata(h, 'settings');

if nargin == 0
    if get(handles.check_showRaw, 'Value')
        level = 1;
    else
        level = 2;
    end
    bNew = false;
elseif nargin == 1
    bNew = false;
end

if bNew
    figure('Name', 'Data trace');
    hAxis = axes;
else
    hAxis = handles.axes_plot;
end

%find out which data needs to be plotted
idx = findCurrentDataIndex();
if idx == 0
    return;
end

bCorr = get(handles.check_baseCorr, 'Value');

if bCorr
    rawdata = dataStruct(idx).rawdataCorr;
    lgndData = {['Data ' dataStruct(idx).groupname ' (corr.)']};
else
    rawdata  = dataStruct(idx).rawdata;
    lgndData = {['Data ' dataStruct(idx).groupname ' (uncorr.)']};
end
noBlocks = dataStruct(idx).nrOfBlocks;

si_sec    = dataStruct(idx).sample_int;
blockFltr = any(dataStruct(idx).filters.blck_fltrs, 2);
bLinked   = get(handles.check_showLinked, 'Value');

%this function requires the multiple links approach
if bLinked
    [linkedFile, m, n] = unique(dataStruct(idx).linkedFile);
    
    if isempty(linkedFile)
        linkdata = nan(numel(rawdata),1);
        lnk      = [];
    else
        if numel(linkedFile) == numel(dataStruct(idx).linkedFile)
            linkedFile = linkedFile(n);
        else
            linkedFile = linkedFile(sort(m));
        end
        %there is a linked file, do some processing to retrieve the correct
        %index for further processing
        filenames = {dataStruct.filename};
        if dataStruct(idx).concentration == max([dataStruct.concentration])
            %this is the max concentration; so there can be multiple linked
            %files present
            if numel(linkedFile) > 1
                %more than 1 linked file (3 groups or more): for now only
                %retrieve the first one
                filefltr = strcmpi(linkedFile{1}, filenames);
                if bCorr
                    linkdata = dataStruct(filefltr).rawdataCorr;
                    lgndData(end+1) = {['Data ' dataStruct(filefltr).groupname ' (corr.)']};
                else
                    linkdata = dataStruct(filefltr).rawdata;
                    lgndData(end+1) = {['Data ' dataStruct(filefltr).groupname ' (uncorr.)']};
                end
                lnk = find(filefltr);
            else
                %only 2 groups: old way should still work
                filefltr = strcmpi(linkedFile{1}, filenames);
                if bCorr
                    linkdata = dataStruct(filefltr).rawdataCorr;
                    lgndData(end+1) = {['Data ' dataStruct(filefltr).groupname ' (corr.)']};
                else
                    linkdata = dataStruct(filefltr).rawdata;
                    lgndData(end+1) = {['Data ' dataStruct(filefltr).groupname ' (uncorr.)']};
                end
                lnk = find(filefltr);
            end
        else
            %only a link to the max concentration
            filefltr = strcmpi(linkedFile{1}, filenames);
            if bCorr
                linkdata = dataStruct(filefltr).rawdataCorr;
                lgndData(end+1) = {['Data ' dataStruct(filefltr).groupname ' (corr.)']};
            else
                linkdata = dataStruct(filefltr).rawdata;
                lgndData(end+1) = {['Data ' dataStruct(filefltr).groupname ' (uncorr.)']};
            end
            lnk = find(filefltr);
        end
        
    end
else
    lnk = [];
end

%plot the raw data
t_axis = (0:numel(rawdata)-1) * si_sec;
if level == 1
    lastIndex = numel(rawdata);
else
    lastIndex = find(blockFltr, 1, 'last') + 30000;
    if lastIndex > numel(t_axis)
        lastIndex = numel(t_axis);
    end
end

%clear the axes first
cla(hAxis);
hold(hAxis, 'on');

bRaw = get(handles.check_showRaw, 'Value');

if bRaw
    %show the entire trace: rawdata
    plot(hAxis, t_axis, rawdata, 'k');
    %do we need to show the linked file?
    if bLinked
        link_t_axis = (0:numel(linkdata)-1) * si_sec;
        plot(hAxis, link_t_axis, linkdata, 'Color', [0.6 0.6 0.6]);
    end
else
    %show only the sucrose data
    rawdata(~blockFltr) = nan;
    plot(hAxis, t_axis, rawdata, 'k');
    %do we need to show the linked file?
    if bLinked && ~isempty(lnk)
        %get the linked data
        linkBlockFltr = any(dataStruct(lnk).filters.blck_fltrs, 2);
        linkdata(~linkBlockFltr) = nan;
        link_t_axis = (0:numel(linkdata)-1) * si_sec;
        plot(hAxis, link_t_axis, linkdata, 'Color', [0.6 0.6 0.6]);
    end
end

if ~isfield(dataStruct(idx), 'blockAnalysis')
    %finished plotting
    hold(hAxis, 'off');
    
    %add decorations to hAxis
    xlim(hAxis, [0 t_axis(lastIndex)]);
    xlabel(hAxis, 'Time (sec)');
    ylabel(hAxis, 'Current (pA)');
    legend(hAxis, lgndData);
    
    title(hAxis, [dataStruct(idx).filename ' (' dataStruct(idx).groupname ')'], 'Interpreter', 'none');
    return;
end

if get(handles.radio_fitOnly, 'Value')
    [currdata, linkdata] = getFitData(idx, lnk);
    %plot the fit data of the current file
    plot(hAxis, currdata(:,1), currdata(:,2), 'Color', [0 1 0]);
    if ~all(isnan(currdata(:,2)))
        lgndData(end+1) = {['Fit ' dataStruct(idx).groupname]};
    end
    
    %plot the linked data?
    if bLinked && ~isempty(linkdata) && ~all(isnan(linkdata(:,2)))
        plot(hAxis, linkdata(:,1), linkdata(:,2), 'Color', [1 0 1]);
        lgndData(end+1) = {['Fit ' dataStruct(lnk).groupname]};
    end
else
    [currdata, linkdata] = getFitData(idx, lnk);
     %plot the fit data of the current file
    plot(hAxis, currdata(:,1), currdata(:,2), 'Color', [0 1 0]);
    if  ~all(isnan(currdata(:,2)))
        lgndData(end+1) = {['Fit ' dataStruct(idx).groupname]};
    end
    
    [currarea, linkarea] = getAreaData(idx, lnk);
    blockSize = floor(size(currarea, 1) / noBlocks);
    
    if ~isempty(dataStruct(idx).blockAnalysis) 
        patch(currarea(1:blockSize,1), currarea(1:blockSize,2), [0 1 0], 'FaceColor', [0 1 0], 'FaceAlpha', 0.6, 'Parent', hAxis);
        patch(currarea(blockSize+1:end,1), currarea(blockSize+1:end,2), [0 1 0], 'FaceColor', [0 1 0], 'FaceAlpha', 0.6, 'Parent', hAxis);
        lgndData(end+1) = {['Area ' dataStruct(idx).groupname ' (Block 1)']};
        lgndData(end+1) = {['Area ' dataStruct(idx).groupname ' (Block 2)']};
    end
     %plot the linked data?
    if bLinked && ~isempty(linkdata) && ~all(isnan(linkdata(:,2)))
        plot(hAxis, linkdata(:,1), linkdata(:,2), 'Color', [1 0 1]);
        lgndData(end+1) = {['Fit ' dataStruct(lnk).groupname]};
        patch(linkarea(1:blockSize,1), linkarea(1:blockSize,2), [1 0 1], 'FaceColor', [1 0 1], 'FaceAlpha', 0.6, 'Parent', hAxis);
        patch(linkarea(blockSize+1:end,1), linkarea(blockSize+1:end,2), [1 0 1], 'FaceColor', [1 0 1], 'FaceAlpha', 0.6, 'Parent', hAxis);
        lgndData(end+1) = {['Area ' dataStruct(lnk).groupname ' (Block 1)']};
        lgndData(end+1) = {['Area ' dataStruct(lnk).groupname ' (Block 2)']};
    end
end

%finished plotting
hold(hAxis, 'off');

%add decorations to hAxis
xlim(hAxis, [0 t_axis(lastIndex)]);
xlabel(hAxis, 'Time (sec)');
ylabel(hAxis, 'Current (pA)');
if strcmpi(get(handles.uitoggletool5, 'State'), 'on')
    %show the legend
    legend(hAxis, lgndData, 'Location', 'SouthEast');
else
    %do not show the legend
    hLgnd = legend(hAxis, lgndData, 'Location', 'SouthEast');
    legend(hLgnd, 'hide');
    set(handles.uitoggletool5, 'State', 'off');
end


title(hAxis, [dataStruct(idx).filename ' (' dataStruct(idx).groupname ')'], 'Interpreter', 'none');


function [currdata, linkdata] = getFitData(idx, lnk)
hMain    = getappdata(0, 'h_sucrose');
dataS    = getappdata(hMain, 'dataStruct');
settings = getappdata(hMain, 'settings');
handles  = getappdata(hMain, 'handles');

data     = dataS(idx);
si       = data.sample_int;
currConc = data.concentration;
bLinked  = ~isempty(lnk);

%create the data sets for plotting
rawE  = data.rawdata;
rawT  = (0:numel(rawE)-1)*si;

%init arrays
currdata = nan(numel(rawE),2);
if bLinked
    linkdata = currdata;
else
    linkdata = [];
end

sucrConcentrations = findConcentration(dataS, idx); %sort(unique([dataS.concentration]), 'descend');
currSucrIndex      = find(currConc==sucrConcentrations, 1, 'first');
startParIdx = 3 + (currSucrIndex-1)*3;

for blockID = 1:numel([data.blockAnalysis])
    %basel = data.blockAnalysis(blockID).baselineconc;
    fltr  = data.filters.blck_fltrs(:,blockID);
    
    %get the linked data if necessary
    if bLinked
        linkIdx  = lnk;
        linkRawE = dataS(linkIdx).rawdata;
        linkRawT = (0:numel(linkRawE)-1)*si;
        %linkBase = dataS(linkIdx).blockAnalysis(blockID).baselineconc;
        
        linkfltr = dataS(linkIdx).filters.blck_fltrs(:,blockID);
        %{
        %not sure why this code is here, but it works without it...
        if numel(fltr) > numel(linkfltr)
            fltr = fltr(1:numel(linkfltr));
        elseif numel(fltr) < numel(linkfltr)
            linkfltr = linkfltr(1:numel(fltr));
        end
        %}
        %linkEpsc = linkRawE(linkfltr) - linkBase;
        linkTime = linkRawT(linkfltr);
    end

    %epsc  = rawE(fltr)-basel;
    time  = rawT(fltr);
    
    %get the current parameters from the structure
    if get(handles.check_extended, 'Value')
        origPar = data.blockAnalysis(blockID).parameters_extended;
        if isempty(origPar)
            return;
        end
    else
        origPar = data.blockAnalysis(blockID).parameters;    
    end
    
    if isempty(origPar)
        continue;
    end
    %determine the start of this block
    fitStartBlck = find(fltr, 1, 'first');
    fitStartBlck = (fitStartBlck-1)*si;
    
    %check for unlinked file
    if startParIdx+2 > numel(origPar)
        %subtract one data set length off the start index
        startParIdx = startParIdx - 3;
        %check again
        if startParIdx+2 > numel(origPar)
            %subtract one data set length off the start index
            startParIdx = startParIdx - 3;
            %check again
            if startParIdx+2 > numel(origPar)
                %this file may not be properly linked
                bLinked = false;
            end
        end
    end
    
    if currConc < sucrConcentrations(1)
        %sub-maximal concentration
        origK2_subMax = sucrose_sigmoid(time, fitStartBlck, time(end), settings.k20, origPar(startParIdx), origPar(startParIdx+1), origPar(startParIdx+2), settings.sucrose_decay);
        if origPar(end-1) == 1000
            [origTimeM, origRelease_conc] = simulate_sucrose(time, origPar(end), origPar([1:2 startParIdx:startParIdx+2 end-1]), origK2_subMax);
            if bLinked
                origK2_max  = sucrose_sigmoid(linkTime, linkTime(1), linkTime(end), settings.k20, origPar(3), origPar(4), origPar(5), settings.sucrose_decay);
                [linkOrigTimeM, linkOrigRelease_conc] = simulate_sucrose(linkTime, origPar(end), origPar([1:5 end-1]), origK2_max);
            end
        else
            if get(handles.check_extended, 'Value')
                fixedPars = origPar(2);
                IC_states = origPar(end-1:end);
                [origTimeM, origRelease_conc] = simulate_sucrose(time, IC_states, origPar(1), origK2_subMax, fixedPars);
                if bLinked
                    origK2_max  = sucrose_sigmoid(linkTime, linkTime(1), linkTime(end), settings.k20, origPar(3), origPar(4), origPar(5), settings.sucrose_decay);
                    [linkOrigTimeM, linkOrigRelease_conc] = simulate_sucrose(linkTime, IC_states, origPar(1), origK2_max, fixedPars);
                end
            else
                [origTimeM, origRelease_conc] = simulate_sucrose(time, origPar(end-1:end), origPar([1:2 startParIdx:startParIdx+2 end-1]), origK2_subMax);
                if bLinked
                    origK2_max  = sucrose_sigmoid(linkTime, linkTime(1), linkTime(end), settings.k20, origPar(3), origPar(4), origPar(5), settings.sucrose_decay);
                    [linkOrigTimeM, linkOrigRelease_conc] = simulate_sucrose(linkTime, origPar(end-1:end), origPar([1:5 end-1]), origK2_max);
                end
            end
            
        end
    else
        %maxmimal concentration
        if get(handles.check_extended, 'Value')
            fixedPars = origPar(2);
            IC_states = origPar(end-1:end);
            origK2_max  = sucrose_sigmoid(time, fitStartBlck, time(end), settings.k20, origPar(3), origPar(4), origPar(5), settings.sucrose_decay);
            [origTimeM, origRelease_conc] = simulate_sucrose(time, IC_states, origPar(1), origK2_max, fixedPars);
            if bLinked
                origK2_subMax = sucrose_sigmoid(linkTime, linkTime(1), linkTime(end), settings.k20, origPar(end-4), origPar(end-3), origPar(end-2), settings.sucrose_decay);
                [linkOrigTimeM, linkOrigRelease_conc] = simulate_sucrose(linkTime, IC_states, origPar(1), origK2_subMax, fixedPars);
            end
        else
            origK2_max  = sucrose_sigmoid(time, fitStartBlck, time(end), settings.k20, origPar(3), origPar(4), origPar(5), settings.sucrose_decay);
            [origTimeM, origRelease_conc] = simulate_sucrose(time, origPar(end), origPar([1:5 end-1]), origK2_max);
            if bLinked
                origK2_subMax = sucrose_sigmoid(linkTime, linkTime(1), linkTime(end), settings.k20, origPar(end-4), origPar(end-3), origPar(end-2), settings.sucrose_decay);
                [linkOrigTimeM, linkOrigRelease_conc] = simulate_sucrose(linkTime, origPar(end-1:end), origPar([1:2 end-4:end-2 end-1]), origK2_subMax);
            end
        end
    end
    
    currdata(fltr,1) = origTimeM';
    currdata(fltr,2) = origRelease_conc';
    if bLinked
        linkdata(linkfltr,1) = linkOrigTimeM';
        linkdata(linkfltr,2) = linkOrigRelease_conc';
    end
end


function sucrConc = findConcentration(dataS, idx)

%Get all concentrations present in the current dataStruct
allConcs = sort(unique([dataS.concentration]), 'descend');
maxConc = allConcs(1);

if dataS(idx).concentration == maxConc %current conc = max conc
    maxIdx = idx;
else %current conc = submax conc
    maxIdx = find(ismember({dataS.filename}, dataS(idx).linkedFile));    
end

submaxIdx = find(ismember({dataS.filename}, dataS(maxIdx).linkedFile));

submaxConc = zeros(1, numel(submaxIdx));
for i = 1:numel(submaxIdx)
    submaxConc(i) = dataS(submaxIdx(i)).concentration;
end

sucrConc = sort([maxConc submaxConc], 'descend');


function [currarea, linkarea] = getAreaData(idx, lnk)
hMain    = getappdata(0, 'h_sucrose');
dataS    = getappdata(hMain, 'dataStruct');
settings = getappdata(hMain, 'settings');
handles  = getappdata(hMain, 'handles');

data     = dataS(idx);
si       = data.sample_int;
currConc = data.concentration;
bLinked  = ~isempty(lnk);

%create the data sets for plotting
rawE  = data.rawdataCorr;
rawT  = (0:numel(rawE)-1)*si;

%init arrays
currarea = [];
linkarea = [];

for blockID = 1:numel([data.blockAnalysis])
    fltr  = data.filters.blck_fltrs(:,blockID);
    epsc  = rawE(data.filters.blck_fltrs(:,blockID));
    time  = rawT(data.filters.blck_fltrs(:,blockID));
    
    %get the linked data if necessary
    if bLinked
        linkIdx  = lnk;
        linkRawE = dataS(linkIdx).rawdataCorr;
        linkRawT = (0:numel(linkRawE)-1)*si;
        linkEpsc = linkRawE(dataS(linkIdx).filters.blck_fltrs(:,blockID));
        linkTime = linkRawT(dataS(linkIdx).filters.blck_fltrs(:,blockID));
    end
    
    %get the current parameters from the text fields
    origPar = data.blockAnalysis(blockID).parameters;    
    
    %determine the start of this block
    fitStartBlck = find(data.filters.blck_fltrs(:,blockID), 1, 'first');
    fitStartBlck = (fitStartBlck-1)*si;
    sucrConcentrations = sort(unique([dataS.concentration]), 'descend');
    currSucrIndex      = find(currConc==sucrConcentrations, 1, 'first');
    startParIdx = 3 + (currSucrIndex-1)*3;
    
    if currConc < sucrConcentrations(1)
        origK2_subMax = sucrose_sigmoid(time, fitStartBlck, time(end), settings.k20, origPar(startParIdx), origPar(startParIdx+1), origPar(startParIdx+2), settings.sucrose_decay);
        [origTimeM, ~, origStates] = simulate_sucrose(time, origPar(end-1:end), origPar([1:2 startParIdx:startParIdx+2]), origK2_subMax);
        origRefill = -(origPar(1).*origStates(:,1) - origPar(2).*origStates(:,2));
        if bLinked
            origK2_max  = sucrose_sigmoid(linkTime, fitStartBlck, linkTime(end), settings.k20, origPar(3), origPar(4), origPar(5), settings.sucrose_decay);
            [linkOrigTimeM, ~, linkStates] = simulate_sucrose(linkTime, origPar(end-1:end), origPar(1:5), origK2_max);
            linkRefill = -(origPar(1).*linkStates(:,1) - origPar(2).*linkStates(:,2));
        end
    else
        origK2_max  = sucrose_sigmoid(time, fitStartBlck, time(end), settings.k20, origPar(3), origPar(4), origPar(5), settings.sucrose_decay);
        [origTimeM, ~, origStates] = simulate_sucrose(time, origPar(end-1:end), origPar(1:5), origK2_max);
        origRefill = -(origPar(1).*origStates(:,1) - origPar(2).*origStates(:,2));
        if bLinked
            origK2_subMax = sucrose_sigmoid(linkTime, fitStartBlck, linkTime(end), settings.k20, origPar(end-4), origPar(end-3), origPar(end-2), settings.sucrose_decay);
            [linkOrigTimeM, ~, linkStates] = simulate_sucrose(linkTime, origPar(end-1:end), origPar([1:2 end-4:end-2]), origK2_subMax);
            linkRefill = -(origPar(1).*linkStates(:,1) - origPar(2).*linkStates(:,2));
        end
    end
    %get the indices
    stepsize = 100;
    if blockID == 1
        currarea = [origTimeM(1:stepsize:end)' origRefill(1:stepsize:end); origTimeM(end) 0; origTimeM(1) origRefill(1)];
        if bLinked
            linkarea = [linkOrigTimeM(1:stepsize:end)' linkRefill(1:stepsize:end); linkOrigTimeM(end) 0; linkOrigTimeM(1) linkRefill(1)];
        end
    else
        currarea = [currarea; origTimeM(1) 0; origTimeM(1:stepsize:end)' origRefill(1:stepsize:end); origTimeM(end) 0];
        if bLinked
            linkarea = [linkarea; linkOrigTimeM(1) 0; linkOrigTimeM(1:stepsize:end)' linkRefill(1:stepsize:end); linkOrigTimeM(end) 0];
        end
    end
     
end



function [idx] = findCurrentDataIndex()
h = getappdata(0, 'h_sucrose');

dataStruct = getappdata(h, 'dataStruct');
handles    = getappdata(h, 'handles');

if isempty(dataStruct)
    idx = 0;
    return;
end

%get the selected group and file from the interface
selectGroup = get(handles.popup_group, 'String');
selectIdx   = get(handles.popup_group, 'Value');
selectGroup = selectGroup{selectIdx};

%get the groupnames from the structure
groupnames = {dataStruct.groupname};
groupfltr  = strcmpi(selectGroup, groupnames);
groupIdx   = find(groupfltr);

%return the required index
idx = groupIdx(get(handles.list_files, 'Value'));


% --- Outputs from this function are returned to the command line.
function varargout = sucrose_GUI_v2_OutputFcn(hObject, eventdata, handles) 
% varargout  cell array for returning output args (see VARARGOUT);
% hObject    handle to figure
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Get default command line output from handles structure
varargout{1} = handles.output;


% --- Executes on selection change in popup_group.
function popup_group_Callback(hObject, eventdata, handles)
%call updateInterface to make sure that everything is up to date
set(handles.list_files, 'Value', 1);
updateInterface();


% --- Executes during object creation, after setting all properties.
function popup_group_CreateFcn(hObject, eventdata, handles)
% hObject    handle to popup_group (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: popupmenu controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end


% --- Executes on selection change in list_files.
function list_files_Callback(hObject, eventdata, handles)
%call updatePlot to see the data
updatePlot();


% --- Executes during object creation, after setting all properties.
function list_files_CreateFcn(hObject, eventdata, handles)
% hObject    handle to list_files (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: listbox controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



function edit_concSucrose_Callback(hObject, eventdata, handles)
%get the entered value
sucrConc = str2double(get(hObject,'String'));

if isnan(sucrConc)
    %invalid number, do nothing
    return;
else
    if sucrConc < 0
        %we have a funny man...
        return;
    end
    
    %enable the button for loading files
    set(handles.push_loadData, 'Enable', 'on');
end

% --- Executes during object creation, after setting all properties.
function edit_concSucrose_CreateFcn(hObject, eventdata, handles)
% hObject    handle to edit_concSucrose (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end


% --- Executes on button press in push_loadData.
function push_loadData_Callback(hObject, eventdata, handles)
%check the value of sucrose concentration
sucrConc = get(handles.edit_concSucrose, 'String');
genotype = get(handles.edit_genotype, 'String');

if isnan(str2double(sucrConc)) || str2double(sucrConc) < 0
    %invalid concentration, disable button
    set(handles.push_loadData, 'Enable', 'off');
    return;
else
    groupname = [sucrConc 'mM'];
end

h = getappdata(0, 'h_sucrose');
settings   = getappdata(h, 'settings');
recentDir  = settings.recentDir;
dataStruct = getappdata(h, 'dataStruct');

%ask the user to select the files
[files, path, ~] = uigetfile({'*.abf',  'Axon files (*.abf)';'*.mat', 'Matlab data files (*.mat)'}, 'Select files',...
    recentDir, 'MultiSelect', 'on');

if path == 0
    %user canceled
    return;
else
    %update the recent directory value
    settings.recentDir = path;
    setappdata(h, 'settings', settings);
    saveSettings(settings);
end

if ~iscell(files)
    files = {files};
end

%ask the user for the fitting range
fitRange_prompt = {'Sucrose pulse length (s):','Start 1st pulse (s):','Start 2nd pulse (s) - leave blank for single pulse:'};
fitRangeDlg_title = 'Fitting procedure input';
def = {num2str(settings.pulseLength),num2str(settings.fitStartBlck(1)),num2str(settings.fitStartBlck(2))};
fitRange_answer = inputdlg(fitRange_prompt, fitRangeDlg_title, 1, def);

pulseLength = str2double(fitRange_answer{1});
if isnan(pulseLength)
    pulseLength = settings.pulseLength;
end
fitStartBlck1 = str2double(fitRange_answer{2});
fitStartBlck2 = str2double(fitRange_answer{3});

%check input
if fitStartBlck1 > fitStartBlck2 || isnan(fitStartBlck1)
    fitStartBlck_temp = fitStartBlck1;
    fitStartBlck1 = fitStartBlck2;
    fitStartBlck2 = fitStartBlck_temp;
end

%go over all the selected files and load them
progressbarUltra('Loading files');
filters = struct('blck1_fltr', [], 'blck2_fltr', []);

scaleFactor = 1;
bError = false;
[~, ext] = strtok(files{1}, '.');
if strcmp(ext, '.abf')
    for f = 1:numel(files)
        filename = fullfile(path, files{f});
        
        %get the data from the file
        [data, si, header] = abfload(filename);
        %if data contains more than 1 column: ignore the others
        if size(data, 2) > 1
            data = data(:,1);
        end
        
        %fitting procedure assumes pA-scale
        scaleUnit = header.recChUnits{1};
        switch scaleUnit
            case 'nA'
                scaleFactor = 1e3;
            otherwise
                scaleFactor = 1;
        end
        if isfield(header, 'fSignalGain')
            gain = header.fSignalGain(1);
            if isnan(gain)
                gain = settings.gain;
                appendSucroseLog(['Gain value in file ' files{f} ' was incorrect'], 'abnormal');
                appendSucroseLog(['Falling back to default gain value: ' num2str(settings.gain)], 'urgent');
                if ~bError
                    bError = true;
                end
            end
        else
            gain = settings.gain;
            appendSucroseLog(['Header information incomplete in file ' files{f} ': missing gain info.'], 'abnormal');
            appendSucroseLog(['Falling back to default gain value: ' num2str(settings.gain)], 'urgent');
            if ~bError
                bError = true;
            end
        end
        
        data = (data/gain)*scaleFactor;
        si = si/1e6;
        
        %check length data versus time points provided by user
        blck1_fltr = false(size(data));
        blck2_fltr = false(size(data));
        while fitStartBlck1+pulseLength > numel(data)*si || fitStartBlck1+pulseLength > fitStartBlck2 || fitStartBlck2+pulseLength > numel(data)*si
            
            qstring = sprintf('Check if the pulses overlap or exceed the maximal time: \n Sucrose pulse length (s): %f \n Start 1st pulse (s): %f \n Start 2nd pulse (s): %f' , ...
                pulseLength,fitStartBlck1,fitStartBlck2);
            input_question = questdlg(qstring,'Are these values correct?');
            switch input_question
                case 'Yes'
                    disp('Cannot fit data with these input values.')
                    break
                case 'No'
                    %ask the user again for the fitting range
                    fitRange_prompt = {'Sucrose pulse length (s):','Start 1st pulse (s):','Start 2nd pulse (s) - leave blank for single pulse:'};
                    fitRangeDlg_title = 'Fitting procedure input';
                    def = {num2str(pulseLength),num2str(fitStartBlck1),num2str(fitStartBlck2)};
                    fitRange_answer = inputdlg(fitRange_prompt, fitRangeDlg_title, 1, def);
                    
                    pulseLength = str2double(fitRange_answer{1});
                    if isnan(pulseLength)
                        pulseLength = settings.pulseLength;
                    end
                    fitStartBlck1 = str2double(fitRange_answer{2});
                    fitStartBlck2 = str2double(fitRange_answer{3});
                    
                    %check input
                    if fitStartBlck1 > fitStartBlck2 || isnan(fitStartBlck1)
                        fitStartBlck_temp = fitStartBlck1;
                        fitStartBlck1 = fitStartBlck2;
                        fitStartBlck2 = fitStartBlck_temp;
                    end
                case 'Cancel'
                    progressbarUltra(1);
                    return
            end
        end
        
        blck1_fltr((fitStartBlck1/si+1):(fitStartBlck1 + pulseLength)/si+1) = true;
        %create the array for the corrected data
        dataCorr = data;
        strtIdx  = find(blck1_fltr, 1, 'first');
        stopIdx  = find(blck1_fltr, 1, 'last');
        baseLen  = (settings.baselineLength/si)+1;
        %calculate the start and stop values of the baseline
        if settings.useAdaptiveBase
            %use the value before and after the response
            strtBase = mean(data((strtIdx-baseLen):strtIdx));
            stopBase = mean(data((stopIdx+2e4):(stopIdx+2e4+baseLen)));
        else
            %use the value before the response only
            strtBase = mean(data((strtIdx-baseLen):strtIdx));
            stopBase = strtBase;
        end
        
        %calculate how far the baseline is from 0
        baseCorr = 0 - interp1([strtIdx stopIdx], [strtBase stopBase], strtIdx:stopIdx);
        
        %apply the correction
        dataCorr(strtIdx:stopIdx) = dataCorr(strtIdx:stopIdx) + baseCorr(:);
        
        nrBlocks = 1;
        if ~isnan(fitStartBlck2)
            blck2_fltr(fitStartBlck2/si+1:(fitStartBlck2 + pulseLength)/si+1) = true;
            strtIdx = find(blck2_fltr, 1, 'first');
            stopIdx = find(blck2_fltr, 1, 'last');
            %calculate the start and stop values of the baseline
            if settings.useAdaptiveBase
                strtBase = mean(data((strtIdx-baseLen):strtIdx));
                stopBase = mean(data((stopIdx+2e4):(stopIdx+2e4+baseLen)));
            else
                strtBase = mean(data((strtIdx-baseLen):strtIdx));
                stopBase = strtBase;
            end
            
            baseCorr = 0 - interp1([strtIdx stopIdx], [strtBase stopBase], strtIdx:stopIdx);
            
            %apply the correction
            dataCorr(strtIdx:stopIdx) = dataCorr(strtIdx:stopIdx) + baseCorr(:);
            
            nrBlocks = nrBlocks + 1;
        end
        
        %and store it in the structure
        if isempty(dataStruct)
            dataStruct.filename      = files{f};
            dataStruct.linkedFile    = '';
            dataStruct.pathname      = path;
            dataStruct.groupname     = groupname;
            dataStruct.genotype      = genotype;
            dataStruct.concentration = str2double(sucrConc);
            dataStruct.rawdata       = data;
            dataStruct.rawdataCorr   = dataCorr;
            dataStruct.nrOfBlocks    = nrBlocks;
            dataStruct.sample_int    = si; %store sample interval in seconds
            dataStruct.filters.blck_fltrs = [blck1_fltr, blck2_fltr];
            dataStruct.gain          = gain;
        else
            %add to the end of the structure
            dataStruct(end+1).filename = files{f};
            dataStruct(end).linkedFile    = '';
            dataStruct(end).pathname      = path;
            dataStruct(end).groupname     = groupname;
            dataStruct(end).genotype      = genotype;
            dataStruct(end).concentration = str2double(sucrConc);
            dataStruct(end).rawdata       = data;
            dataStruct(end).rawdataCorr   = dataCorr;
            dataStruct(end).nrOfBlocks    = nrBlocks;
            dataStruct(end).sample_int    = si; %sample interval in seconds
            dataStruct(end).filters.blck_fltrs = [blck1_fltr, blck2_fltr];
            dataStruct(end).gain          = gain;
        end
        appendSucroseLog(['Added file ' files{f} ' in group ' groupname], 'info');
        progressbarUltra(f/numel(files));
    end
else %do the same except from data structure modified in modData.m
    for g = 1:numel(files)
        filename = fullfile(path, files{g});
        load(filename);
        %check for correct structure name
        if exist('modDataStruct', 'var')
            %get the data from the file
            for f = 1:length(modDataStruct)
                
                data = modDataStruct(f).data;
                si = modDataStruct(f).si;
                header = modDataStruct(f).header;
                %if data contains more than 1 column: ignore the others
                if size(data, 2) > 1
                    data = data(:,1);
                end
                
                %fitting procedure assumes pA-scale
                scaleUnit = header.recChUnits{1};
                switch scaleUnit
                    case 'nA'
                        scaleFactor = 1e3;
                    otherwise
                        scaleFactor = 1;
                end
                if isfield(header, 'fSignalGain')
                    gain = header.fSignalGain(1);
                    if isnan(gain)
                        gain = settings.gain;
                        appendSucroseLog(['Gain value in file ' modDataStruct(f).filename ' was incorrect'], 'abnormal');
                        appendSucroseLog(['Falling back to default gain value: ' num2str(settings.gain)], 'urgent');
                        if ~bError
                            bError = true;
                        end
                    end
                else
                    gain = settings.gain;
                    appendSucroseLog(['Header information incomplete in file ' modDataStruct(f).filename ': missing gain info.'], 'abnormal');
                    appendSucroseLog(['Falling back to default gain value: ' num2str(settings.gain)], 'urgent');
                    if ~bError
                        bError = true;
                    end
                end
                
                data = (data/gain)*scaleFactor;
                si = si/1e6;
                
                %check length data versus time points provided by user
                blck1_fltr = false(size(data));
                blck2_fltr = false(size(data));
                while fitStartBlck1+pulseLength > numel(data)*si || fitStartBlck1+pulseLength > fitStartBlck2 || fitStartBlck2+pulseLength > numel(data)*si
                    
                    qstring = sprintf('Check if the pulses overlap or exceed the maximal time: \n Sucrose pulse length (s): %f \n Start 1st pulse (s): %f \n Start 2nd pulse (s): %f' , ...
                        pulseLength,fitStartBlck1,fitStartBlck2);
                    input_question = questdlg(qstring,'Are these values correct?');
                    switch input_question
                        case 'Yes'
                            disp('Cannot fit data with these input values.')
                            break
                        case 'No'
                            %ask the user again for the fitting range
                            fitRange_prompt = {'Sucrose pulse length (s):','Start 1st pulse (s):','Start 2nd pulse (s) - leave blank for single pulse:'};
                            fitRangeDlg_title = 'Fitting procedure input';
                            def = {num2str(pulseLength),num2str(fitStartBlck1),num2str(fitStartBlck2)};
                            fitRange_answer = inputdlg(fitRange_prompt, fitRangeDlg_title, 1, def);
                            
                            pulseLength = str2double(fitRange_answer{1});
                            if isnan(pulseLength)
                                pulseLength = settings.pulseLength;
                            end
                            fitStartBlck1 = str2double(fitRange_answer{2});
                            fitStartBlck2 = str2double(fitRange_answer{3});
                            
                            %check input
                            if fitStartBlck1 > fitStartBlck2 || isnan(fitStartBlck1)
                                fitStartBlck_temp = fitStartBlck1;
                                fitStartBlck1 = fitStartBlck2;
                                fitStartBlck2 = fitStartBlck_temp;
                            end
                        case 'Cancel'
                            progressbarUltra(1);
                            return
                    end
                end
                
                blck1_fltr((fitStartBlck1/si+1):(fitStartBlck1 + pulseLength)/si+1) = true;
                %create the array for the corrected data
                dataCorr = data;
                strtIdx  = find(blck1_fltr, 1, 'first');
                stopIdx  = find(blck1_fltr, 1, 'last');
                baseLen  = (settings.baselineLength/si)+1;
                %calculate the start and stop values of the baseline
                if settings.useAdaptiveBase
                    %use the value before and after the response
                    strtBase = mean(data((strtIdx-baseLen):strtIdx));
                    stopBase = mean(data((stopIdx+2e4):(stopIdx+2e4+baseLen)));
                else
                    %use the value before the response only
                    strtBase = mean(data((strtIdx-baseLen):strtIdx));
                    stopBase = strtBase;
                end
                
                %calculate how far the baseline is from 0
                baseCorr = 0 - interp1([strtIdx stopIdx], [strtBase stopBase], strtIdx:stopIdx);
                
                %apply the correction
                dataCorr(strtIdx:stopIdx) = dataCorr(strtIdx:stopIdx) + baseCorr(:);
                
                nrBlocks = 1;
                if ~isnan(fitStartBlck2)
                    blck2_fltr(fitStartBlck2/si+1:(fitStartBlck2 + pulseLength)/si+1) = true;
                    strtIdx = find(blck2_fltr, 1, 'first');
                    stopIdx = find(blck2_fltr, 1, 'last');
                    %calculate the start and stop values of the baseline
                    if settings.useAdaptiveBase
                        strtBase = mean(data((strtIdx-baseLen):strtIdx));
                        stopBase = mean(data((stopIdx+2e4):(stopIdx+2e4+baseLen)));
                    else
                        strtBase = mean(data((strtIdx-baseLen):strtIdx));
                        stopBase = strtBase;
                    end
                    
                    baseCorr = 0 - interp1([strtIdx stopIdx], [strtBase stopBase], strtIdx:stopIdx);
                    
                    %apply the correction
                    dataCorr(strtIdx:stopIdx) = dataCorr(strtIdx:stopIdx) + baseCorr(:);
                    
                    nrBlocks = nrBlocks + 1;
                end
                
                %and store it in the structure
                if isempty(dataStruct)
                    dataStruct.filename      = modDataStruct(f).filename;
                    dataStruct.linkedFile    = '';
                    dataStruct.pathname      = path;
                    dataStruct.groupname     = groupname;
                    dataStruct.genotype      = genotype;
                    dataStruct.concentration = str2double(sucrConc);
                    dataStruct.rawdata       = data;
                    dataStruct.rawdataCorr   = dataCorr;
                    dataStruct.nrOfBlocks    = nrBlocks;
                    dataStruct.sample_int    = si; %store sample interval in seconds
                    dataStruct.filters.blck_fltrs = [blck1_fltr, blck2_fltr];
                    dataStruct.gain          = gain;
                else
                    %add to the end of the structure
                    dataStruct(end+1).filename = modDataStruct(f).filename;
                    dataStruct(end).linkedFile    = '';
                    dataStruct(end).pathname      = path;
                    dataStruct(end).groupname     = groupname;
                    dataStruct(end).genotype      = genotype;
                    dataStruct(end).concentration = str2double(sucrConc);
                    dataStruct(end).rawdata       = data;
                    dataStruct(end).rawdataCorr   = dataCorr;
                    dataStruct(end).nrOfBlocks    = nrBlocks;
                    dataStruct(end).sample_int    = si; %sample interval in seconds
                    dataStruct(end).filters.blck_fltrs = [blck1_fltr, blck2_fltr];
                    dataStruct(end).gain          = gain;
                end
                appendSucroseLog(['Added file ' modDataStruct(f).filename ' in group ' groupname], 'info');
                progressbarUltra(f/length(modDataStruct));
            end
        else
            progressbarUltra(1);
            msgbox('Incorrect data structure, see modData.m for loading *.mat files.', 'Error')
            return
        end
    end
end
%store values
settings.pulseLength = pulseLength;
settings.fitStartBlck = [fitStartBlck1 fitStartBlck2];

%log the number of files added
appendSucroseLog(['Added ' num2str(numel(files)) ' files to the data structure.']);

%reset the import parameters
set(handles.edit_concSucrose, 'String', 'Concentration');
set(handles.push_loadData, 'Enable', 'off');
%switch off the linkage check box
set(handles.check_linkage, 'Value', 0);

%add the data to the appdata structure
setappdata(h, 'dataStruct', dataStruct);

statStruct = generateStatStructure(true);
setappdata(h, 'statStruct', statStruct);


setappdata(h, 'settings', settings);

%sort the data and update the interface
applyDataSorting();
updateInterface();

if bError
    %an error occurred
    errordlg('An error occurred during the import, which may result in incorrect data. Please check the log for more details.', 'Import error');
end


% --------------------------------------------------------------------
function menu_file_Callback(hObject, eventdata, handles)
% --------------------------------------------------------------------
function menu_edit_Callback(hObject, eventdata, handles)
% --------------------------------------------------------------------
function menu_plot_Callback(hObject, eventdata, handles)

h = getappdata(0, 'h_sucrose');
dataStruct = getappdata(h, 'dataStruct');

if isempty(dataStruct)
    %no data...
    set(handles.menu_plot_extended_cost, 'Enable', 'off');
    set(handles.menu_plot_RRP_fraction, 'Enable', 'off');
else
    if isfield(dataStruct, 'blockAnalysis')
        %blockAnalysis field exists
        if ~all(cellfun('isempty', {dataStruct.blockAnalysis}))
            %not all blockAnalysis fields are empty
            ba = [dataStruct.blockAnalysis];
            
            if ~all(cellfun('isempty', {ba.parameters_extended}))
                %not all parameters_extended fields are empty
                set(handles.menu_plot_extended_cost, 'Enable', 'on');
                set(handles.menu_plot_RRP_fraction, 'Enable', 'on');
            else
                set(handles.menu_plot_extended_cost, 'Enable', 'off');
                set(handles.menu_plot_RRP_fraction, 'Enable', 'off');
            end
        else
            set(handles.menu_plot_extended_cost, 'Enable', 'off');
            set(handles.menu_plot_RRP_fraction, 'Enable', 'off');
        end
    else
        set(handles.menu_plot_extended_cost, 'Enable', 'off');
        set(handles.menu_plot_RRP_fraction, 'Enable', 'off');
    end
end

% --------------------------------------------------------------------
function menu_help_Callback(hObject, eventdata, handles)

% --------------------------------------------------------------------
function menu_help_help_Callback(hObject, eventdata, handles)
h = getappdata(0, 'h_sucrose');
docRoot = getappdata(h, 'docRoot');

docFile = fullfile(docRoot, 'index.html');

if exist(docFile, 'file') == 2
    %load the documentation
    web(docFile, '-new', '-notoolbar');
else
    return;
end

% --------------------------------------------------------------------
function menu_help_about_Callback(hObject, eventdata, handles)
h = getappdata(0, 'h_sucrose');

versionNr = getappdata(h, 'versionNr');
buildNr   = getappdata(h, 'buildNr');
fcv       = getappdata(h, 'fileContainerVersion');

msg = {'Sucrose Analysis'
       ''
       ['Version:' num2str(versionNr)];
       ['Build nr: ' buildNr];
       ['File container version: ' num2str(fcv)];
       ''
       'Interface by Jurjen Broeke'
       'Analysis code by Bas Schotten'
       'Biophysical model by Niels Cornelisse'
       'License: CC-BY-SA-NC'
       ''
       'This program was developed for fitting a biophysical model to hypertonic'
       'sucrose responses measured with electrophysiological recordings. The model'
       'was published in [#ref, 2014].'
       ''
       'Center for Neurogenomics and Cognitive Research'
       'Neuroscience Campus Amsterdam'
       'Vrije Universiteit (VU) and VU medical center'
       'De Boelelaan 1085'
       '1081HV Amsterdam'
       'The Netherlands'
       'http://www.cncr.nl'};
   
helpdlg(msg, 'About Sucrose Analysis...');

% --------------------------------------------------------------------
function menu_help_debug_Callback(hObject, eventdata, handles)
h = getappdata(0, 'h_sucrose');

dataStruct = getappdata(h, 'dataStruct');
statStruct = getappdata(h, 'statStruct');
settings   = getappdata(h, 'settings');

%remove the handle of the figure for safety reasons...
clear h;

%log the event as debug session
appendSucroseLog('Started DEBUG session', 'info');

disp('This is the DEBUG mode, use with caution.');
disp('Type RETURN and press enter to exit DEBUG mode!');
keyboard;

%log the event as debug session
appendSucroseLog('Ended DEBUG session', 'info');


% --------------------------------------------------------------------
function menu_plot_RRP_fraction_Callback(hObject, eventdata, handles)

hMain      = getappdata(0, 'h_sucrose');
dataS      = getappdata(hMain, 'dataStruct');
settings   = getappdata(hMain, 'settings');

si = dataS.sample_int;
pulseLength = settings.pulseLength; %Ask user input?

if isempty(dataS)
    %nothing to plot
    return;
else
    %plot released RRP fraction vs release rate constant
    ba = {dataS.blockAnalysis};
    fltrs = {dataS.filters};
    genotypes = {dataS.genotype};
    conc = [dataS.concentration];
    %fn = cellfun(@fieldnames, ba, 'UniformOutput', false);
    %numFields = cellfun(@numel, fn, 'UniformOutput', false);
    
    %clean up all entries that lack a blockAnalysis field
    emptyBaFltr = cellfun('isempty', ba);
    ba(emptyBaFltr) = [];
    fltrs(emptyBaFltr) = [];
    genotypes(emptyBaFltr) = [];
    conc(emptyBaFltr) = [];
    
    ba = cell2mat(ba);
    pars = {ba.parameters_extended};
    if isempty(pars)
        %do not plot anything if there was no extended fit
        return;
    end
    unqGenotype = unique(genotypes);
    
    %ask user for the control genotype, plot this data in black
    [ctrlGenotypeIdx, ok] = listdlg('ListString',unqGenotype, ...
        'SelectionMode','single',...
        'ListSize',[160 160], ...
        'PromptString','Select control genotype:');
    if ok == 0
        %do not plot anything
        return;
    else
        ctrlGenotype = unqGenotype{ctrlGenotypeIdx};
    end
    
    plotColours = {'r', 'b', 'g', 'c', 'm', 'y'};
    plotSymbols = {'o', 's', 'd', '^', 'v', '>', '<', '*', 'x', 'p', 'x', '+', '.'};
    
    k2_max = max(cell2mat(cellfun(@(C) C(3), pars, 'UniformOutput', false)));
    k2_min = min(cell2mat(cellfun(@(C) min(C(3:3:end-2)), pars, 'UniformOutput', false)));
    k2_max = ceil(log10(k2_max));
    k2_min = floor(log10(k2_min));
    
    %simulated step size is hardcoded
    if k2_min < -3
       k2_range = 10.^([k2_min:-3, -2.9:.1:k2_max]);
    else
       k2_range = 10.^(k2_min:.1:k2_max); 
    end 
    
    figure; 
    hComb = axes();
    hold(hComb, 'on');
    
    %create a separate figure for each genotype
    for i = 1:numel(unqGenotype)   
        genoFltr = strcmp(unqGenotype{i}, genotypes);
        
        geno_fltrs = fltrs(genoFltr);
        genoConc = conc(genoFltr);
        genoPars = pars(genoFltr);
        
        %calculate model prediction and extract fitted data
        [RRPFrac_model, k2_data, RRPFrac_data, conc_data] = calculateRRPFrac(k2_range, genoPars, genoConc, geno_fltrs, pulseLength, si);
        unqConc = unique(conc_data);
        unqConc = unqConc(~isnan(unqConc));
        
        if unqGenotype{i} == ctrlGenotype
            plotClr = 'k';
        else
            plotClr = plotColours{i};
        end
        
        lgndModel = {'Model prediction'};
        tmpLegend = cellfun(@(C) [num2str(C) 'mM'], num2cell(unqConc), 'UniformOutput', false);
        
        figure; 
        plot(log10(k2_range), RRPFrac_model, plotClr);
        hSingle = gca;
        legend(hSingle, lgndModel, 'Location', 'NorthWest');
        
        lgndSingle = [lgndModel; tmpLegend(:)];
        
        if i == 1
            plot(hComb, log10(k2_range), RRPFrac_model, plotClr);
            lgndComb = lgndSingle;
        else
            plot(hComb, log10(k2_range), RRPFrac_model, plotClr);
            lgndComb = [lgndComb; lgndSingle];
        end
        
       hold(hSingle, 'on'); 
       
        for j = 1:numel(unqConc)
            concFltr = (conc_data == unqConc(j));
            
            if numel(unqConc) <= numel(plotSymbols)
                plot(hSingle, log10(k2_data(concFltr)), RRPFrac_data(concFltr), cat(2, plotClr, plotSymbols{j}) );               
                plot(hComb, log10(k2_data(concFltr)), RRPFrac_data(concFltr), cat(2, plotClr, plotSymbols{j}) );                
            else        
                plot(hSingle, log10(k2_data(concFltr)), RRPFrac_data(concFltr), 'ko');
                plot(hComb, log10(k2_data(concFltr)), RRPFrac_data(concFltr), 'ko');
            end
        end
        
        title(hSingle,unqGenotype{i});
        legend(hSingle, lgndSingle,'Location','NorthWest');
        xlabel(hSingle,'log_{10}(Release rate constant) (s^{-1})');
        ylabel(hSingle,'Released RRP fraction'); 
        hold(hSingle, 'off'); 
        ylim(hSingle, [0 1.2]);
    end
    title(hComb, 'All genotypes');
    legend(hComb, lgndComb,'Location','NorthWest');
    xlabel(hComb,'log_{10}(Release rate constant) (s^{-1})');
    ylabel(hComb,'Released RRP fraction');
    hold(hComb, 'off');
    ylim(hComb, [0 1.2]);
end


function [RRPFrac_model, k2_data, RRPFrac_data, conc_data] = calculateRRPFrac(k2_range, genoPars, genoConc, geno_fltrs, pulseLength, si)

hMain      = getappdata(0, 'h_sucrose');
settings   = getappdata(hMain, 'settings');

%Extract mean parameters needed for generating the model-predicted curve
meanGenoPars = zeros(1,6);
meanGenoParsIdx = 1;

for j = [1, 2, 4, 5] %k1, k-1, tDel, tauOnset
    meanGenoPars(meanGenoParsIdx) = mean(cell2mat(cellfun(@(C) C(j), genoPars, 'UniformOutput', false)));
    meanGenoParsIdx = meanGenoParsIdx+1;
end

%meanGenoPars(1) = meanGenoPars(1)/2; %To test the effect of
%priming on the sigmoid
meanGenoPars(meanGenoParsIdx) = mean(cell2mat(cellfun(@(C) C(end-1), genoPars, 'UniformOutput', false)));
meanGenoPars(end) = meanGenoPars(1)*meanGenoPars(end-1)/meanGenoPars(2); %R = k1*D/k-1

time = 0:si:pulseLength;

for k = 1:numel(k2_range)   
    curr_pars = [meanGenoPars(1:2), k2_range(k), meanGenoPars(3:end)];
    
    k2_curr = sucrose_sigmoid(time, 0, pulseLength, settings.k20,...
        curr_pars(3), curr_pars(4), curr_pars(5), settings.sucrose_decay);
    [~, release, states] = simulate_sucrose(time, curr_pars(end-1:end), curr_pars(1:end-1), k2_curr);
    refill = -(curr_pars(1).*states(:,1) - curr_pars(2).*states(:,2));
    
    cumulRelease = -sum(release)*si; %Total fitted release (charge)
    cumulRefill = -sum(refill)*si; %Total fitted refill (charge)
    
    RRPFrac_model(k) = (cumulRelease-cumulRefill)/curr_pars(end);
end


%extract the fitted data points
k2_data = NaN(1,numel(genoPars));
RRPFrac_data = NaN(1,numel(genoPars));
conc_data = NaN(1,numel(genoPars));
unqConc = sort(unique(genoConc),'descend');

try
    for m = 1:numel(genoPars)
        %length of sucrose pulse can differ
        curr_fltr = geno_fltrs{m}.blck_fltrs(:,1);
        time = (0:sum(curr_fltr)-1)*si;
        tFitStart = time(1);
        tFitEnd = time(end);
        
        parameters = genoPars{m};
        conc = genoConc(m);
        concIdx = find(unqConc == conc);
        curr_pars = NaN(1,7);
        
        if 4+3*concIdx <= numel(parameters)
            curr_pars = parameters([1:2, (3:5)+3*(concIdx - 1), end-1:end]);
            
            k2_data(m) = curr_pars(3);
            if conc == unqConc(1)
                RRPFrac_data(m) = 1;
            else    
                %calculate the charge due to the fitted release and the refill
                k2_currConc = sucrose_sigmoid(time, tFitStart, tFitEnd, settings.k20,...
                    curr_pars(3), curr_pars(4), curr_pars(5), settings.sucrose_decay);
                [~, release, states] = simulate_sucrose(time, curr_pars(end-1:end), curr_pars(1:5), k2_currConc);
                refill = -(curr_pars(1).*states(:,1) - curr_pars(2).*states(:,2));
                
                cumulRelease = -sum(release)*si; %Total fitted release
                cumulRefill = -sum(refill)*si; %Total fitted refill
                RRPFrac_data(m) = (cumulRelease-cumulRefill)/parameters(end);
            end
            conc_data(m) = conc;
        end
    end
catch me
    appendSucroseLog(me, 'urgent');
    appendSucroseLog(['Debug ' getLineNumber()], 'urgent');
end


% --------------------------------------------------------------------
function menu_edit_params_Callback(hObject, eventdata, handles)
hMain    = getappdata(0, 'h_sucrose');
settings = getappdata(hMain, 'settings');

defAns  = settings.par_init(1:5);
defAnsS = num2cell(defAns);
defAnsS = cellfun(@num2str, defAnsS, 'UniformOutput', false);
prompts = {'Priming rate (k1)', 'Unpriming rate (k-1)', 'Release rate (k2max)',...
           'Onset delay (tdel)', 'Rise time (tau)'};
answers = inputdlg(prompts, 'Set initial values',1,defAnsS);

if isempty(answers)
    return;
end

answers = cellfun(@str2double, answers);
nanfltr = isnan(answers);
if any(nanfltr)
    answers(nanfltr) = defAns(nanfltr);
end

%create the correct format
answers = answers(:)';
new_par = [answers answers(3:5)];

%store the values in the settings
settings.par_init     = new_par;
settings.par_previous = new_par;
setappdata(hMain, 'settings', settings);
saveSettings();



% --------------------------------------------------------------------
function menu_edit_prefs_Callback(hObject, eventdata, handles)
% open the preference window
hPref = sucrose_preferences();
uiwait(hPref);
%at this point preferences should be correct
saveSettings();


% --------------------------------------------------------------------
function menu_file_loadSession_Callback(hObject, eventdata, handles)
%ask the user for a file location
hMain = getappdata(0, 'h_sucrose');

dataStruct  = getappdata(hMain, 'dataStruct');
recentDir   = getappdata(hMain, 'recentDir');
fileVersion = getappdata(hMain, 'fileContainerVersion');

[file, path, ~] = uigetfile({'*.mat' 'Session file (*.mat)'}, 'Load a session file', recentDir);

if path == 0
    return;
end

sessionFile = fullfile(path, file);
%create a new session file to prevent the old one from being overwritten:
%undesired side-effect?
if isempty(regexp(sessionFile, '-session.mat$', 'once'))
    %not a session yet
    newSessionFile = fullfile(path, strrep(file, '.mat', '-session.mat'));
else
    %re-use the session file
    newSessionFile = sessionFile;
end

try
    in = load(sessionFile);
    
    if isfield(in, 'dataStruct')
        if isfield(in, 'fileVersion')
            if fileVersion > in.fileVersion
                %the program uses a newer data structure, exit
                helpdlg('The selected file has an older data structure and can not be used.');
                return;
            end
        end
        if isempty(dataStruct)
            %load the session directly
            setappdata(hMain, 'dataStruct', in.dataStruct);
            setappdata(hMain, 'sessionfile', newSessionFile);
        else
            %append the new session to the existing data
            setappdata(hMain, 'dataStruct', [dataStruct in.dataStruct]);
            currSession = getappdata(hMain, 'sessionFile');
            if isempty(currSession)
                setappdata(hMain, 'sessionfile', sessionFile);
            else
                setappdata(hMain, 'sessionfile', newSessionFile);
            end
        end
        
        updateInterface();
    end
catch me
    appendSucroseLog(me, 'urgent');
    appendSucroseLog(['Debug ' getLineNumber()], 'urgent');
end


% --------------------------------------------------------------------
function menu_file_loadExample_Callback(hObject, eventdata, handles)
% hObject    handle to menu_file_loadExample (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)


% --------------------------------------------------------------------
function menu_file_exit_Callback(hObject, eventdata, handles)
%call the closerequest function
hMain = getappdata(0, 'h_sucrose');

figure1_CloseRequestFcn(hMain, eventdata, handles)


% --------------------------------------------------------------------
function menu_list_remove_Callback(hObject, eventdata, handles)
h = getappdata(0, 'h_sucrose');
%get the overall dataset
dataStruct = getappdata(h, 'dataStruct');

if isempty(dataStruct)
    %nothing to do
    return;
end

%get the selected group from the popup
groupnames = get(handles.popup_group, 'String');
groupIdx   = get(handles.popup_group, 'Value');
groupname  = groupnames{groupIdx};

%get the selected index from the list
selectIdx = get(handles.list_files, 'Value');

%get the groupnames from the data structure
groupnames = {dataStruct.groupname};

%get the filenames belonging to the specified group
group_fltr = strcmpi(groupname, groupnames);
filenames  = {dataStruct.filename};
allFiles   = filenames;
filenames  = filenames(group_fltr);
idx_group  = find(group_fltr);

filename = filenames{selectIdx};

%ask the user to really remove the selected file
answer = questdlg({'Do you wish to remove the data belonging to: ';
                    filename});

switch(answer)
    case 'Yes'
        %remove link in the donor file
        donor_link = dataStruct(idx_group(selectIdx)).linkedFile;
        fltr = strcmp(donor_link, allFiles);
        if any(fltr)
            dataStruct(fltr).linkedFile = '';
        end
        
        %get the filename for logging
        filename = dataStruct(idx_group(selectIdx)).filename;
        
        %remove the data and update the structure
        dataStruct(idx_group(selectIdx)) = [];
        setappdata(h, 'dataStruct', dataStruct);
        
        %file was removed, log it
        appendSucroseLog(['Removed data of file ' filename], 'info');
        
        %update the interface
        updateInterface();
    case {'No','Cancel'}
        %nothing to do
        return;
end

% --------------------------------------------------------------------
function menu_list_removeAll_Callback(hObject, eventdata, handles)
h = getappdata(0, 'h_sucrose');
%get the overall dataset
dataStruct = getappdata(h, 'dataStruct');

if isempty(dataStruct)
    %nothing to do
    return;
end

%get the selected group from the popup
groupnames = get(handles.popup_group, 'String');
groupIdx   = get(handles.popup_group, 'Value');
groupname  = groupnames{groupIdx};

%get the groupnames from the data structure
groupnames = {dataStruct.groupname};

%get the filenames belonging to the specified group
group_fltr = strcmpi(groupname, groupnames);

choice    = questdlg(['Are you sure you wish to remove data from ' num2str(sum(group_fltr)) ' files?']);
filenames = {dataStruct.filename};

switch(choice)
    case {'No','Cancel'}
        %nothing to do,
        return;
    case 'Yes'
        %remove the data belonging to this group
        dataStruct(group_fltr) = [];
        
        %update the data structure and then the interface
        setappdata(h, 'dataStruct', dataStruct);
        
        %check if the dataStruct is now empty
        if isempty(dataStruct)
            %restore the default string for the group popup
            set(handles.popup_group, 'String', 'Groupname');
            set(handles.list_files, 'String', 'Files...');
            cla(handles.axes_plot);
        else
            %get the groupnames from the dataStructure
            newGroupnames = {dataStruct.groupname};
            %add the correct list back
            set(handles.popup_group, 'String', newGroupnames);
        end
        %change the group selector to avoid errors
        set(handles.popup_group, 'Value', 1)
        
        %group was removed, log it
        filenames = filenames(group_fltr);
        for f = 1:numel(filenames)
            appendSucroseLog(['Removed data of file ' filenames{f}], 'info');
        end
        
        updateInterface();
end




% --------------------------------------------------------------------
function list_menu_Callback(hObject, eventdata, handles)

h = getappdata(0, 'h_sucrose');
dataStruct = getappdata(h, 'dataStruct');

if isempty(dataStruct)
    %no data...
    set(handles.menu_list_refit_extended, 'Enable', 'off');
else
    if isfield(dataStruct, 'blockAnalysis')
        if ~all(cellfun('isempty', {dataStruct.blockAnalysis}))
            set(handles.menu_list_refit_extended, 'Enable', 'on');
        else
            set(handles.menu_list_refit_extended, 'Enable', 'off');
        end
    else
        set(handles.menu_list_refit_extended, 'Enable', 'off');
    end
end

% --------------------------------------------------------------------
function menu_fit_Callback(hObject, eventdata, handles)

h = getappdata(0, 'h_sucrose');
dataStruct = getappdata(h, 'dataStruct');

if isempty(dataStruct)
    %no data...
    set(handles.menu_fit_fitAll, 'Enable', 'off');
    set(handles.menu_fit_fitExtended, 'Enable', 'off');
else
    set(handles.menu_fit_fitAll, 'Enable', 'on');
    if isfield(dataStruct, 'blockAnalysis')
        if ~all(cellfun('isempty', {dataStruct.blockAnalysis}))
            set(handles.menu_fit_fitExtended, 'Enable', 'on');
        else
            set(handles.menu_fit_fitExtended, 'Enable', 'off');
        end
    else
        set(handles.menu_fit_fitExtended, 'Enable', 'off');
    end
end

% --------------------------------------------------------------------
function menu_fit_fitAll_Callback(hObject, eventdata, handles)
fitAllData()

%determine the fit quality for the stat structure
hMain = getappdata(0, 'h_sucrose');
statS = generateStatStructure(true);
setappdata(hMain, 'statStruct', statS);



function fitAllData()
h = getappdata(0, 'h_sucrose');
%add explanatory text to statusbar
updateStatusbar('Fitting data...');

settings = getappdata(h, 'settings');
dataStruct = getappdata(h, 'dataStruct');
fileNames = {dataStruct.filename};

analysis = struct('label', [], 'baselineconc', [], 'baselinelssthn500', [], 'parameters', [], 'cost', [], 'release_infiniteconc', [], 'release_finiteconc', [], ...
    'release_infinitelssthn500', [], 'release_finitelssthn500', []);

sucrConcs  = unique({dataStruct.groupname}); 
maxSucrose = max([dataStruct.concentration]);

if settings.useFixedParam
    %ask the user which parameters should be fixed
    choice_list = {'priming rate (k1)', 'unpriming rate (k-1)', 'release rate (k2max)', 'onset delay (tdel)', 'rise time (tau)', 'depot pool (D)'};
    [select, btn] = listdlg('ListString', choice_list);
    if isempty(select)
        setappdata(h, 'fixed_parameters', []);
        return;
    end
    
    %ask the user for the correct values
    prompts = choice_list(select);
    prompts = cellfun(@(C) ['Value of ' C ' (blank: use block 1):'], prompts, 'UniformOutput', false);
    def_ans = num2cell(rand(1, numel(select)));
    def_ans = cellfun(@num2str, def_ans, 'UniformOutput', false);
    answers = inputdlg(prompts, 'Enter the fixed values', 1, def_ans);
    
    if isempty(answers)
        setappdata(h, 'fixed_parameters', []);
        return;
    end
    
    %process the values
    answers = cellfun(@str2double, answers);
    
    %create the fixed parameters list and at the answers
    nanfltr = isnan(answers);
    if any(nanfltr)
        fixed_par = nan(1, numel(choice_list));
        fixed_par(select)  = answers;
        fixed_par(select(nanfltr)) = -1;    %reuse values of previous block?
    else
        fixed_par = nan(1, numel(choice_list));
        fixed_par(select) = answers;
    end
    
    
    setappdata(h, 'fixed_parameters', fixed_par);
else
    %leave empty
    setappdata(h, 'fixed_parameters', []);
end

%add a line to the log (for performance measurement)
appendSucroseLog('Started fitting all data.');
tic;
%create a progressbar
progressbarUltra('Group progress', 'Fitting data');
bException = false;
try
    %Loop over all concentrations
    for i = 1:numel(sucrConcs)
        
        conc = str2double(sucrConcs{i}(1:end-2));
        
        conc_fltr = strcmp({dataStruct.groupname}, sucrConcs{i});
        conc_file_cntr = find(conc_fltr == 1);
        
        %update the progressbar for the groups
        progressbarUltra((i-0.5)/numel(sucrConcs), []);
        
        %Loop over all files for a single concentration
        for j = 1:numel(conc_file_cntr)
            if isempty(dataStruct(conc_file_cntr(j)).linkedFile)
                %unlinked file: always fit
                fitDataSingle(conc_file_cntr(j));
            else
                %linked file
                if conc < maxSucrose
                    %concentration less than the max sucrose concentration 
                    %and linked to 250: no fitting required (maxSucrose
                    %files will handle this).
                    continue;
                else
                    fitDataSingle(conc_file_cntr(j));
                end
            end
            %update progreesbar
            progressbarUltra([], (j)/numel(conc_file_cntr));
        end
        
        %update the progressbar for the groups
        progressbarUltra(i/numel(sucrConcs), []);
    end
    
catch me
    appendSucroseLog(me, 'urgent');
    appendSucroseLog(['Debug ' getLineNumber()], 'urgent');
    bException = true;
    progressbarUltra(1,1);
end
%switch the progressbar completely off
progressbarUltra(1,1);
elapsedTime = toc;
if elapsedTime > 300
    elapsedTime = floor(elapsedTime / 60);
    timeUnit = ' minutes.';
else
    timeUnit = ' seconds.';
end

if bException
    helpdlg('There were invalid parameters, please consult the log for information.');
end
%add a line to the log (for performance measurement)
appendSucroseLog(['Finished fitting all data in ' num2str(elapsedTime) timeUnit]);

%add explanatory text to statusbar
updateStatusbar('Finished fitting data.');
updateInterface();


function fitDataSingle(selIdx)
h = getappdata(0, 'h_sucrose');
%add explanatory text to statusbar
updateStatusbar('Refitting data...');

%get the relevant data
dataStruct = getappdata(h, 'dataStruct');
settings   = getappdata(h, 'settings');

if isempty(dataStruct)
    return;
end
filenames  = {dataStruct.filename};

if nargin == 0
    %get the handles
    handles = getappdata(h, 'handles');
    %get the currently selected file from the file list
    currFilename = get(handles.list_files, 'String');
    currFilename = currFilename(get(handles.list_files, 'Value'));
    
    %find the correct index
    selIdx = find(strcmp(currFilename, filenames));
    
    dataS = dataStruct(selIdx);
    conc  = dataS.concentration;
    maxConc = max([dataStruct.concentration]);
    
    if conc ~= maxConc
        linkFile = dataStruct(selIdx).linkedFile{1};
        selIdx = find(strcmp(linkFile, filenames));
    end
    
    %ask user for block to refit (leave empty for all)
    refitBlock_prompt = {'Refit sucrose pulse # - leave blank for both pulses:'};
    refitBlockDlg_title = 'Refitting procedure input';
    startBlock_answer = inputdlg(refitBlock_prompt, refitBlockDlg_title, 1);
    pause(0.1);
    noBlocks = dataStruct(selIdx).nrOfBlocks;
    
    if isempty(startBlock_answer)
        %user canceled, return.
        return;
    elseif isempty(startBlock_answer{1})
        startBlock = 1;
    else
        startBlock = str2double(startBlock_answer{1});
    end
else
    %set the startBlock to 1 by default
    startBlock = 1;
end

%get the data required for refitting
dataS = dataStruct(selIdx);
si    = dataS.sample_int;
epsc  = dataS.rawdataCorr;
%time  = {(0:numel(epsc)-1)*si};
conc  = dataS.concentration;

%get the value for maximum sucrose within the data set
maxSucrose = max([dataStruct.concentration]);

%check for a linked file
if ~isempty(dataS.linkedFile)
    %there is linked data, get it from the linked file field
    linkedFiles = dataS.linkedFile;
    for i = 1:numel(linkedFiles)
        linkIdx = find(strcmp(linkedFiles{i}, filenames));
        if i==1
            rd = {dataStruct(linkIdx).rawdataCorr};
            rd_corr = rd;
        else
            %concatenate the groups
            rd = [rd, {dataStruct(linkIdx).rawdataCorr}];
            rd_corr = [rd_corr rd];
        end
    end  
end
%epsc contains the correct data for fitting now
noBlocks       = dataS.nrOfBlocks;

if startBlock > noBlocks
    %what to do: use whole range
    startBlock = 1;
    stopBlock = noBlocks;
else
    %use the selected range
    if nargin == 0
        stopBlock = startBlock;
    else
        stopBlock = noBlocks;
    end
end

%change the status text
updateStatusbar('Busy refitting...');

%add a line to the log (for performance measurement)
appendSucroseLog('Started fitting data.');

%change pointer
%watchon;
%enableDisableFig(h, false);

try
    if ~isfield(dataStruct(selIdx),'blockAnalysis')
        dataStruct(selIdx).blockAnalysis = [];
        dataS.blockAnalysis = [];
    end
    
    %go over the number of blocks present
    for b = startBlock:stopBlock
        fltr     = dataS.filters.blck_fltrs(:,b);
        fit_EPSC = {epsc(fltr)};
        fit_time = {(0:numel(fit_EPSC{1})-1)*si};
        
        if ~isempty(dataS.linkedFile)
            for n =1:numel(dataS.linkedFile)
                linkIdx = find(strcmp(linkedFiles{n}, filenames), 1, 'first');
                linkfltr = dataStruct(linkIdx).filters.blck_fltrs(:,b);
               
                fit_rd(n) = {rd{n}(linkfltr)};
                time_rd(n) = {(0:numel(fit_rd{n})-1)*si};
            end
            
            if dataS.concentration < maxSucrose
                %current data first, then linked data
                fit_EPSC = [fit_EPSC, fit_rd];
                %apply smoothing?
                %fit_EPSC = cellfun(@(x) smooth(x, 100), fit_EPSC, 'UniformOutput', false);
                fit_time = [fit_time, time_rd];
            else
                %linked data first, then original data
                fit_EPSC = [fit_rd, fit_EPSC];
                %apply smoothing?
                %fit_EPSC = cellfun(@(x) smooth(x, 100), fit_EPSC, 'UniformOutput', false);
                fit_time = [time_rd, fit_time];
            end            
        end
        
        %use previous settings or new random settings?
        if settings.reusePrevious
            if ~isempty(dataS.blockAnalysis)
                %use the data from the previous fit
                if numel(dataS.blockAnalysis) < b
                    if numel(fit_EPSC) == 2
                        par_init = settings.par_init;
                    elseif numel(fit_EPSC) > 2
                        par_init = settings.par_init;
                        additionalPar = numel(fit_EPSC) - 2;
                        par_init = [par_init repmat(par_init(6:8), 1, additionalPar)];
                    else
                        par_init = settings.par_init(1:5);
                    end
                    par_init = [par_init 1e3];
                else
                    previous_par = dataS.blockAnalysis(b).parameters;
                    par_init = previous_par(1:numel(fit_EPSC)*3+2);
                    par_init(1) = (previous_par(1)*previous_par(end-1))/1e3; %'Reset' k1 to correspond to D = 1e3
                end
                
            else
                %use the data from the settings
                if numel(fit_EPSC) == 2 
                    par_init = settings.par_init;
                elseif numel(fit_EPSC) > 2
                    par_init = settings.par_init;
                    additionalPar = numel(fit_EPSC) - 2;
                    par_init = [par_init repmat(par_init(6:8), 1, additionalPar)];
                else
                    par_init = settings.par_init(1:5);
                end
                par_init = [par_init 1e3];
            end
        else
            if numel(fit_EPSC) == 2 
                par_init = settings.par_init;
            elseif numel(fit_EPSC) > 2
                par_init = settings.par_init;
                additionalPar = numel(fit_EPSC) - 2;
                par_init = [par_init repmat(par_init(6:8), 1, additionalPar)];
            else
                par_init = settings.par_init(1:5);
            end
            par_init = [par_init.*rand(1, numel(par_init)) 1e3];
        end
              
        logstr = ['fitting block ' num2str(b) ' of file ' dataS.filename];
        appendSucroseLog(logstr, 'info');
        
        %get the fixed parameters
        fixedPars  = getappdata(h, 'fixed_parameters');
        %no fixed pars supplied, use default
        if isempty(fixedPars)
            fixedPars = nan(1, numel(par_init));
            fixedPars(end) = 1e3;
        else
            prevfltr = (fixedPars == -1);
            if any(prevfltr) && b > 1
                %get the previous values
                if settings.useExtended
                    if ~isempty(dataS.blockAnalysis)
                        if isempty(dataS.blockAnalysis(1).parameters_extended)
                            par_prev = dataS.blockAnalysis(1).parameters(1:end-1);
                        else
                            par_prev = dataS.blockAnalysis(1).parameters_extended(1:end-1);
                        end
                        %apply the previous values
                        prevfltr  = [prevfltr(1:2) repmat(prevfltr(3:5), 1, numel(fit_EPSC)) prevfltr(end)];
                        fixedPars = [fixedPars(1:2) repmat(fixedPars(3:5), 1, numel(fit_EPSC)) fixedPars(end)];
                        fixedPars(prevfltr) = par_prev(prevfltr);
                        if isnan(fixedPars(end))
                            fixedPars(end) = 1e3;
                        end
                    else
                        fixedPars = nan(1, numel(par_init));
                        fixedPars(end) = 1e3;
                    end
                else
                    par_prev  = dataS.blockAnalysis(1).parameters(1:end-1);
                    prevfltr  = [prevfltr(1:2) repmat(prevfltr(3:5), 1, numel(fit_EPSC)) prevfltr(end)];
                    fixedPars = [fixedPars(1:2) repmat(fixedPars(3:5), 1, numel(fit_EPSC)) fixedPars(end)];
                    fixedPars(prevfltr) = par_prev(prevfltr);
                    if isnan(fixedPars(end))
                        fixedPars(end) = 1e3;
                    end
                end
            else
                %correct the length
                if any(prevfltr) && b == 1
                    fixedPars(prevfltr) = nan;
                end
                
                if isnan(fixedPars(end))
                    fixedPars = [fixedPars(1:2) repmat(fixedPars(3:5), 1, numel(fit_EPSC)) 1e3];
                else
                    fixedPars = [fixedPars(1:2) repmat(fixedPars(3:5), 1, numel(fit_EPSC)) fixedPars(end)];
                end
            end
        end
        
        if settings.useExtended
            if ~isempty(dataS.blockAnalysis)
                if isempty(dataS.blockAnalysis(b).parameters_extended)
                    par_extended = dataS.blockAnalysis(b).parameters(1:end-1);
                else
                    par_extended = dataS.blockAnalysis(b).parameters_extended(1:end-1);
                end
                cost = dataS.blockAnalysis(b).cost;
                [par_extended, cost_extended] = fit_sucrose(fit_time, fit_EPSC, par_extended, settings.globalFitType, fixedPars);
            else
                %par = par_init;
                %[par, cost] = fit_sucrose(fit_time, fit_EPSC, par, settings.globalFitType);
                %Fit expontential to obtain k1?
                par_extended = par_init;
                [par_extended, cost_extended] = fit_sucrose(fit_time, fit_EPSC, par_extended, settings.globalFitType, fixedPars);
            end
            par = par_extended;
            cost = cost_extended;
        else
            [par, cost] = fit_sucrose(fit_time, fit_EPSC, par_init, settings.globalFitType, fixedPars);
            par_extended = [];
            cost_extended = [];
        end
        
        analysis.label          = settings.globalFitType;
        analysis.label_extended = settings.globalFitType;
        analysis.parameters     = par;
        
        if settings.useExtended
            %R_extended = (par_extended(1)*par_extended(end))/par_extended(2);
            %par_extended = [par_extended R_extended];
            if ~isempty(dataS.blockAnalysis)
                if b <= numel(dataS.blockAnalysis) && ~isempty(dataS.blockAnalysis(b).parameters_extended)
                    %re-use the previous values
                    analysis.parameters_extended = dataS.blockAnalysis(b).parameters_extended;
                end
            else
                analysis.parameters_extended = par_extended;
            end
        else
            %R_extended = [];
            par_extended = [];
            if ~isempty(dataS.blockAnalysis)
                if b <= numel(dataS.blockAnalysis) && ~isempty(dataS.blockAnalysis(b).parameters_extended)
                    %re-use the previous values
                    analysis.parameters_extended = dataS.blockAnalysis(b).parameters_extended;
                end
            end
        end
        
        analysis.parameters_extended = par_extended;
        analysis.cost                = cost;
        analysis.cost_extended       = cost_extended;
        analysis.useWeightedFit      = settings.useWeightedFit;
        
        if conc == maxSucrose
            if b == 1
                if isfield(dataS, 'blockAnalysis') && ~isempty(dataS.blockAnalysis)
                    dataS.blockAnalysis(b) = analysis; %Store fitting results for concentration >= 500mM
                else
                    dataS.blockAnalysis    = analysis; %Store fitting results for concentration >= 500mM
                end
            else
                dataS.blockAnalysis(b) = analysis; %Store fitting results for concentration <500mM
            end
            
            %store the linked cost as well
            for n = 1:numel(dataS.linkedFile)
                linkIdx = find(strcmp(linkedFiles{n}, filenames));
                if b == 1
                    if isfield(dataStruct(linkIdx), 'blockAnalysis') && ~isempty(dataStruct(linkIdx).blockAnalysis)
                        dataStruct(linkIdx).blockAnalysis(b) = analysis; %Store fitting results for linked 500mM data
                    else
                        dataStruct(linkIdx).blockAnalysis    = analysis; %Store fitting results for linked 500mM data
                    end
                else
                    dataStruct(linkIdx).blockAnalysis(b) = analysis; %Store fitting results for linked 500mM data
                end
            end
            
        else 
            if b == 1
                if isfield(dataS, 'blockAnalysis') && ~isempty(dataS.blockAnalysis)
                    dataS.blockAnalysis(b) = analysis; %Store fitting results for concentration >= 500mM
                else
                    dataS.blockAnalysis    = analysis; %Store fitting results for concentration >= 500mM
                end
            else
                dataS.blockAnalysis(b) = analysis; %Store fitting results for concentration >= 500mM
            end
        end
        
    end
catch me
    appendSucroseLog(me, 'urgent');
    appendSucroseLog(['Debug ' getLineNumber()], 'urgent');
    updateStatusbar('An error occurred during refitting...');
    watchoff(h);
    enableDisableFig(h, true);
end

watchoff(h);
enableDisableFig(h, true);

%add a line to the log (for performance measurement)
appendSucroseLog('Finished fitting data.');

%finished refitting, store the data back into its structure
dataStruct(selIdx) = dataS;
setappdata(h, 'dataStruct', dataStruct);
updateStatusbar('Finished refitting.');
if nargin == 0
    updateInterface();
end


function updateStatusbar(msg)
hMain   = getappdata(0, 'h_sucrose');
handles = getappdata(hMain, 'handles');

if ~ischar(msg)
    if isnumeric(msg)
        msg = num2str(msg);
    else
        msg = 'Unhandled exception...';
    end
end

set(handles.text_status, 'String', msg);


% --- Executes when user attempts to close figure1.
function figure1_CloseRequestFcn(hObject, eventdata, handles)
% hObject    handle to figure1 (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

prepareForExit();

% Hint: delete(hObject) closes the figure
delete(hObject);


function prepareForExit()
%collect all possible windows
hChildren(1).h = getappdata(0, 'h_fitDiscovery');
hChildren(2).h = getappdata(0, 'h_sucrPrefs');

if ~isempty(hChildren(1).h)
    %call uiresume
    uiresume(hChildren(1).h);
    pause(0.2);
end

%prepare for exiting: do some cleanup and save the current state
bOK = saveSession();
if ~bOK
    %failed to save the session, error
    appendSucroseLog('Failed to save the current data to a session file.', 'abnormal');
end

%save the current settings
saveSettings();

%{
for i = 1:numel(hChildren)
    %close the windows
    try
        if close(hChildren(i)) == 0
            %show the info about what is going on
            hHelp = helpdlg({'Could not close because ' get(hChildren, 'Name') 'is still open. Please close that window first.'});
            uiwait(hHelp);
            
            %bring it to the fore
            figure(hChildren(i));
            
            %abort the closing
            return;
        end
    catch me
        disp(getReport(me, 'basic'));
        return;
    end
end
%}

%finished with cleanup, remove appdata
rmappdata(0, 'h_sucrose');


% --- Executes on button press in check_linkage.
function check_linkage_Callback(hObject, eventdata, handles)
% Hint: get(hObject,'Value') returns toggle state of check_linkage
h = getappdata(0, 'h_sucrose');
dataStruct = getappdata(h, 'dataStruct');

if isempty(dataStruct)
    return;
end

%get the group names
groupnames = {dataStruct.groupname};
genotypes  = {dataStruct.genotype};
filenames  = {dataStruct.filename};
clean_fltr = false(numel(filenames), 1);
sucroseConc = strrep(groupnames, 'mM', '');
%convert the concentrations to numbers
sucroseConc = str2double(sucroseConc);
maxSucrose  = max(sucroseConc);

if any(sucroseConc<maxSucrose) && ~all(sucroseConc<maxSucrose)
    %perform the linkage
    link_accept_fltr = sucroseConc==maxSucrose;
    link_donor_fltr  = sucroseConc<maxSucrose;
    
    %determine the genotypes for the acceptor files
    unqGeno = unique(genotypes(link_accept_fltr));
    numGeno = numel(unqGeno);
    
    %get the indices for the two classes
    group_accept = find(link_accept_fltr);

    
    %{
    %get the filenames for the donor group
    group_donor_files = filenames(link_donor_fltr);
    group_donor_conc  = sucroseConc(link_donor_fltr);
    %}
    
    %go over all the acceptors and try to find a donor
    for i = 1:numel(group_accept)
        idx = group_accept(i);
        currFilename = dataStruct(idx).filename;
        currGenotype = dataStruct(idx).genotype;
        genotypefltr = all([link_donor_fltr(:) strcmp(currGenotype, genotypes(:))], 2);
        idxPrefix    = regexp(currFilename, '[_- ]\d{3,4}.abf$');
        filePrefix   = currFilename(1:idxPrefix-1);
        
        %clean up the donor filenames according to genotype
        group_donor_files = filenames(genotypefltr);
        group_donor_conc  = sucroseConc(genotypefltr);
        group_donor  = find(genotypefltr);
        
        %try to locate the donor
        filePrefix = regexprep(filePrefix, '\(', '\\(');
        filePrefix = regexprep(filePrefix, '\)', '\\)');
        donor_idx = ~cellfun('isempty', regexp(group_donor_files, filePrefix, 'once'));
        donor_idx = find(donor_idx);
        
        %get a filename from the index
        if isempty(donor_idx)
            %no donor could be found, clean the data
            clean_fltr(idx) = true;
        else
            %found it, create the links
            donor_filename = group_donor_files(donor_idx);
            donor_conc     = group_donor_conc(donor_idx);
            
            %apply a sort to make sure the order is from low to high
            [~,sortIdx] = sort(donor_conc);
            donor_filename(sortIdx) = donor_filename;
            
            %first add the donor to the acceptor
            dataStruct(idx).linkedFile = donor_filename;
            %then add the acceptor to the donor
            for j = 1:numel(donor_filename)
                dataStruct(group_donor(donor_idx(j))).linkedFile = {currFilename};
            end
        end
    end
else
    %no linking necessary, so linkage is correct
    set(handles.check_linkage, 'Value', 1);
    return;
end

%run the cleanup if necessary
if any(clean_fltr) && ~all(clean_fltr)
    %get the filenames that will be cleaned for logging
    filenames = {dataStruct.filename};
    filenames = filenames(clean_fltr);
    
    %remove the actual file data
    dataStruct(clean_fltr) = [];
    
    %append the log
    for f = 1:numel(filenames)
        appendSucroseLog(['Removed data of file ' filenames{f}], 'info');
    end
end

%all files were properly linked, update the checkbox
set(handles.check_linkage, 'Value', 1);

setappdata(h, 'dataStruct', dataStruct);
statS = generateStatStructure(true);
setappdata(h, 'statStruct', statS);
updateInterface();


% --------------------------------------------------------------------
function plot_menu_explore_Callback(hObject, eventdata, handles)
%get the current selected data and open the explore GUI
hMain = getappdata(0, 'h_sucrose');
dataStruct = getappdata(hMain, 'dataStruct');

%check if fitting has been performed succesfully
if ~isfield(dataStruct, 'blockAnalysis')
    updateStatusbar('Perform fitting first...');
    return;
else
    updateStatusbar('Busy exploring the model...');
end

%get the proper index
filenames = {dataStruct.filename};
currFile  = get(handles.list_files, 'String');
currIdx   = get(handles.list_files, 'Value');
currFile  = currFile{currIdx};

idx = find(strcmp(currFile, filenames));

%find the linked file
linkFile = dataStruct(idx).linkedFile;
if isempty(linkFile)
    linkIdx = 0;
else
    %if this is a max response
    if dataStruct(idx).concentration == max([dataStruct.concentration])
        %max response, get the min response
        %get all the minimal sucrose positions
        minSucrFltr  = [dataStruct.concentration] == min([dataStruct.concentration]);
        minSucrIdx   = find(minSucrFltr);
        minSucrNames = filenames(minSucrFltr);
        
        if numel(linkFile) > 1
            for m = 1:numel(linkFile)
                if any(strcmp(linkFile{m}, minSucrNames))
                    linkIdx = minSucrIdx(m);
                    break;
                end
            end
        else
            %only one linked file, get the index
            linkIdx = find(strcmp(linkFile{1}, minSucrNames),1, 'first');
        end
    else
        %this is a sub-max response, so the linked file is the max response
        linkIdx = find(strcmp(linkFile{1}, filenames),1 , 'first');
    end
    
    
end

clear('dataStruct');
hExplore = fitDiscovery_GUI([idx linkIdx]);
uiwait(hExplore);

%check if the discovery interface closed properly
hDisc = getappdata(0, 'h_fitDiscovery');
if ~isempty(hDisc)
    %it did not close properly
    % TO DO: handle this event
end
updateStatusbar('Finished exploring the model.');
appendSucroseLog(['Finished exploring ' currFile]);

saveSession();


% --------------------------------------------------------------------
function plot_menu_Callback(hObject, eventdata, handles)
set(handles.plot_menu_explore, 'Enable', 'off');


% --------------------------------------------------------------------
function menu_list_refit_Callback(hObject, eventdata, handles)
%use the empty call to let the interface find the correct file
fitDataSingle();

% --------------------------------------------------------------------
function menu_list_refitGroup_Callback(hObject, eventdata, handles)
%get all the indices for the current group and refit them all
hMain = getappdata(0, 'h_sucrose');
dataS = getappdata(hMain, 'dataStruct');

if isempty(dataS)
    return;
end

currGroup = get(handles.popup_group, 'String');
currGroup = currGroup{get(handles.popup_group, 'Value')};

%get all the groups from the data structure
groupnames = {dataS.groupname};
%and get their indices
groupIdx = find(strcmp(currGroup, groupnames));

if isempty(groupIdx)
    %something went wrong, abort
    return;
end

for grp = 1:numel(groupIdx)
    %fit this cell of the current group
    fitDataSingle(groupIdx(grp));
end

if saveSession()
    updateStatusbar('Session was saved.');
end


% --------------------------------------------------------------------
function menu_file_saveSession_Callback(hObject, eventdata, handles)
if saveSession()
    updateStatusbar('Session was saved.');
end


% --------------------------------------------------------------------
function menu_help_log_Callback(hObject, eventdata, handles)
%get the log data
hMain = getappdata(0, 'h_sucrose');
logfile = getappdata(hMain, 'logfile');

try
    if exist(logfile, 'file') == 2
        %open the file for reading
        fid = fopen(logfile, 'r');
        %get the text and reverse it
        logstr = textscan(fid, '%s', 'Delimiter', '\n');
        logstr = flipud(logstr{:,1});
        fclose(fid);
    else
        return;
    end
catch me
    updateStatusbar('Failed to open the log file');
    appendSucroseLog(me, 'urgent');
    fclose(fid);
    return;
end

%open the log viewer, and wait for winking owl to return...
[~,p] = listdlg('ListString', logstr, 'ListSize', [600, 300],...
                'CancelString', 'Close',...
                'OKString', 'Clear log',...
                'SelectionMode', 'single');

if p == 1
    %clear the log
    try
        %ask the user if he is sure
        button = questdlg('Are you sure you want to delete the log?', 'Delete log');
        
        switch(button)
            case 'Yes'
                %delete the log file
                delete(logfile);
            case {'No', 'Cancel'}
                %do nothing
                return;
            otherwise
                return;
        end
        
    catch me
        appendSucroseLog('Isn''t it ironic... clearing the log failed, and we are logging it');
        appendSucroseLog(me);
    end
end


% --------------------------------------------------------------------
function menu_file_export_Callback(hObject, eventdata, handles)
hMain = getappdata(0, 'h_sucrose');
dataStruct = getappdata(hMain, 'dataStruct');

if isempty(dataStruct)
    set(handles.menu_file_export_fit, 'Enable', 'off');
    set(handles.menu_file_export_plot, 'Enable', 'off');
else
    if isfield(dataStruct, 'blockAnalysis')
        if ~all(cellfun('isempty', {dataStruct.blockAnalysis}))
            set(handles.menu_file_export_fit, 'Enable', 'on');
            set(handles.menu_file_export_plot, 'Enable', 'on');
        else
            set(handles.menu_file_export_fit, 'Enable', 'off');
            set(handles.menu_file_export_plot, 'Enable', 'off');
        end
    else
        set(handles.menu_file_export_fit, 'Enable', 'off');
        set(handles.menu_file_export_plot, 'Enable', 'off');
    end
end


% --------------------------------------------------------------------
function menu_file_export_fit_Callback(hObject, eventdata, handles)
hMain = getappdata(0, 'h_sucrose');
dataStruct = getappdata(hMain, 'dataStruct');
statStruct = getappdata(hMain, 'statStruct');
versionNr = getappdata(hMain, 'versionNr');
fileContainerVersion = getappdata(hMain, 'fileContainerVersion');

if isempty(dataStruct) || isempty(statStruct)
    %nothing to export
    return;
else
    %prepare for export
    %ask for a filename and export the data
    [file, path, ~] = uiputfile({'*.xls, *.xlsx' 'Excel files (*.xls, *.xlsx)'}, ...
                                 'Select an output file');
    if path == 0
        return;
    end
    
    %get the full file name
    file = fullfile(path, file);
    %and export the data
    export2XL(dataStruct, statStruct, fileContainerVersion, versionNr, file);
end



% --------------------------------------------------------------------
function menu_file_export_plot_Callback(hObject, eventdata, handles)
if get(handles.check_showRaw, 'Value')
    level = 1;
else
    level = 2;
end
updatePlot(level,true);



function edit_genotype_Callback(hObject, eventdata, handles)
% hObject    handle to edit_genotype (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of edit_genotype as text
%        str2double(get(hObject,'String')) returns contents of edit_genotype as a double


% --- Executes during object creation, after setting all properties.
function edit_genotype_CreateFcn(hObject, eventdata, handles)
% hObject    handle to edit_genotype (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end


% --- Executes on button press in check_showRaw.
function check_showRaw_Callback(hObject, eventdata, handles)
% hObject    handle to check_showRaw (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

updatePlot();


% --- Executes on button press in check_showLinked.
function check_showLinked_Callback(hObject, eventdata, handles)
% hObject    handle to check_showLinked (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

updatePlot();


% --- Executes when selected object is changed in uipanel_fitOptions.
function uipanel_fitOptions_SelectionChangeFcn(hObject, eventdata, handles)
hMain = getappdata(0, 'h_sucrose');
dataStruct = getappdata(hMain, 'dataStruct');
idx = findCurrentDataIndex();

if isempty(dataStruct)
    return; 
else
    if isfield(dataStruct(idx), 'blockAnalysis')
        %enable the controls
        set(handles.radio_fitOnly, 'Enable', 'on');
        set(handles.radio_fitArea, 'Enable', 'on');
    else
        %disable the controls
        set(handles.radio_fitOnly, 'Enable', 'off');
        set(handles.radio_fitArea, 'Enable', 'off');
        return;
    end
end
%find out which button was selected
if get(handles.radio_fitOnly, 'Value')
    level = 3;
else
    level = 4;
end


updatePlot(level);


% --------------------------------------------------------------------
function menu_file_convertData_Callback(hObject, eventdata, handles)

if applyUserConversion()
    %successfully created the session file
    hHelp = helpdlg('The file was succesfully converted.', 'Finished conversion');
    uiwait(hHelp);
    pause(0.1);
    %the dataStruct should be added to the appdata, so update the interface
    updateInterface();
else
    errordlg('The file could not be converted, check the log for more information', 'Conversion failed');
end


% --- Executes on button press in check_baseCorr.
function check_baseCorr_Callback(hObject, eventdata, handles)
% hObject    handle to check_baseCorr (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of check_baseCorr
updatePlot();


% --------------------------------------------------------------------
function menu_edit_reviewData_Callback(hObject, eventdata, handles)
hMain = getappdata(0, 'h_sucrose');
dataS = getappdata(hMain, 'dataStruct');

if isempty(dataS)
    return;
end

hRev = sucrose_review();
uiwait(hRev);
pause(0.25);

%sort the data and update the interface
applyDataSorting();
updateInterface();


function applyDataSorting()
hMain = getappdata(0, 'h_sucrose');
dataS = getappdata(hMain, 'dataStruct');

if isempty(dataS)
    return;
end

%get the concentrations and sort them from high to low
sucroseConc = [dataS.concentration];
[~, sortIdx] = sort(sucroseConc, 'descend');

%apply the sorting to the structure
dataS = dataS(sortIdx);

%put the sorted data back
setappdata(hMain, 'dataStruct', dataS);



% --------------------------------------------------------------------
function menu_edit_reset_Callback(hObject, eventdata, handles)
% code for handling the resetting of the interface
hMain = getappdata(0, 'h_sucrose');
setappdata(hMain, 'dataStruct', []);
setappdata(hMain, 'statStruct', []);

cla(handles.axes_plot, 'reset');
legend(handles.axes_plot, 'off');
title(handles.axes_plot, '');
set(handles.list_files, 'String', 'Files...');
set(handles.list_files, 'Value', 1);
set(handles.text_files, 'String', 'Files:');
set(handles.popup_group, 'String', 'Groupname');
set(handles.popup_group, 'Value', 1);
set(handles.edit_concSucrose, 'String', 'Concentration');
set(handles.check_linkage, 'Value', 0);
set(handles.check_extended, 'Value', 0);
set(handles.check_extended, 'Enable', 'off');
updateStatusbar('Ready...');


% --------------------------------------------------------------------
function menu_fit_fitExtended_Callback(hObject, eventdata, handles)

h = getappdata(0, 'h_sucrose');

%add explanatory text to statusbar
updateStatusbar('Fitting data...');

settings = getappdata(h, 'settings');
dataStruct = getappdata(h, 'dataStruct');
fileNames = {dataStruct.filename};

sucrConcs  = unique({dataStruct.groupname}); 
maxSucrose = max([dataStruct.concentration]);

%add a line to the log (for performance measurement)
appendSucroseLog('Started fitting all data.');
tic;
%create a progressbar
progressbarUltra('Group progress', 'Fitting data');
bException = false;

try
    %Loop over all concentrations
    for i = 1:numel(sucrConcs)
        
        conc = str2double(sucrConcs{i}(1:end-2));
        
        conc_fltr = strcmp({dataStruct.groupname}, sucrConcs{i});
        conc_file_cntr = find(conc_fltr == 1);
        
        %update the progressbar for the groups
        progressbarUltra((i-0.5)/numel(sucrConcs), []);
        
        %Loop over all files for a single concentration
        for j = 1:numel(conc_file_cntr)
            if isempty(dataStruct(conc_file_cntr(j)).linkedFile)
                %unlinked file: always fit
                fitDataSingle_Extended(conc_file_cntr(j));
            else
                %linked file
                if conc < maxSucrose
                    %concentration less than the max sucrose concentration 
                    %and linked to 250: no fitting required (maxSucrose
                    %files will handle this).
                    continue;
                else
                    fitDataSingle_Extended(conc_file_cntr(j));
                end
            end
            %update progreesbar
            progressbarUltra([], (j)/numel(conc_file_cntr));
        end
        
        %update the progressbar for the groups
        progressbarUltra(i/numel(sucrConcs), []);
    end
    
catch me
    appendSucroseLog(me, 'urgent');
    appendSucroseLog(['Debug ' getLineNumber()], 'urgent');
    bException = true;
    progressbarUltra(1,1);
end

%switch the progressbar completely off
progressbarUltra(1,1);
elapsedTime = toc;
if elapsedTime > 300
    elapsedTime = floor(elapsedTime / 60);
    timeUnit = ' minutes.';
else
    timeUnit = ' seconds.';
end

if bException
    helpdlg('There were invalid parameters, please consult the log for information.');
end

%add a line to the log (for performance measurement)
appendSucroseLog(['Finished fitting all data in ' num2str(elapsedTime) timeUnit]);

%add explanatory text to statusbar
updateStatusbar('Finished fitting data.');
set(handles.check_extended, 'Enable', 'on');
updateInterface();


function fitDataSingle_Extended(selIdx)
h = getappdata(0, 'h_sucrose');
%add explanatory text to statusbar
updateStatusbar('Refitting data...');

%get the relevant data
dataStruct = getappdata(h, 'dataStruct');
settings   = getappdata(h, 'settings');
filenames  = {dataStruct.filename};

if nargin == 0
    %get the handles
    handles = getappdata(h, 'handles');
    %get the currently selected file from the file list
    currFilename = get(handles.list_files, 'String');
    currFilename = currFilename(get(handles.list_files, 'Value'));
    
    %find the correct index
    selIdx = find(strcmp(currFilename, filenames));
end

%get the data required for refitting
dataS = dataStruct(selIdx);
si    = dataS.sample_int;
epsc  = dataS.rawdataCorr;
%time  = {(0:numel(epsc)-1)*si};
conc  = dataS.concentration;

%get the value for maximum sucrose within the data set
maxSucrose = max([dataStruct.concentration]);

%check for a linked file
if ~isempty(dataS.linkedFile)
    %there is linked data, get it from the linked file field
    linkedFiles = dataS.linkedFile;
    for i = 1:numel(linkedFiles)
        linkIdx = find(strcmp(linkedFiles{i}, filenames));
        if i==1
            rd = {dataStruct(linkIdx).rawdataCorr};
            rd_corr = rd;
        else
            %concatenate the groups
            rd = [rd, {dataStruct(linkIdx).rawdataCorr}];
            rd_corr = [rd_corr rd];
        end
    end  
end
%epsc contains the correct data for fitting now
noBlocks       = dataS.nrOfBlocks;

%change the status text
updateStatusbar('Busy refitting...');

%add a line to the log (for performance measurement)
appendSucroseLog('Started fitting data.');

try
    if ~isfield(dataStruct(selIdx),'blockAnalysis')
        dataStruct(selIdx).blockAnalysis = [];
        dataS.blockAnalysis = [];
        dataStruct(selIdx) = dataS;
        setappdata(h, 'dataStruct', dataStruct);
        return;
    end
    
    %go over the number of blocks present
    for b = 1:noBlocks
        analysis = dataS.blockAnalysis(b);
        fltr     = dataS.filters.blck_fltrs(:,b);
        fit_EPSC = {epsc(fltr)};
        fit_time = {(0:numel(fit_EPSC{1})-1)*si};
        
        if ~isempty(dataS.linkedFile)
            for n = 1:numel(dataS.linkedFile)
                linkIdx = find(strcmp(linkedFiles{n}, filenames), 1, 'first');
                linkfltr = dataStruct(linkIdx).filters.blck_fltrs(:,b);
               
                fit_rd(n) = {rd{n}(linkfltr)};
                time_rd(n) = {(0:numel(fit_rd{n})-1)*si};
            end
            
            if dataS.concentration < maxSucrose
                %current data first, then linked data
                fit_EPSC = [fit_EPSC, fit_rd];
                %apply smoothing?
                %fit_EPSC = cellfun(@(x) smooth(x, 100), fit_EPSC, 'UniformOutput', false);
                fit_time = [fit_time, time_rd];
            else
                %linked data first, then original data
                fit_EPSC = [fit_rd, fit_EPSC];
                %apply smoothing?
                %fit_EPSC = cellfun(@(x) smooth(x, 100), fit_EPSC, 'UniformOutput', false);
                fit_time = [time_rd, fit_time];
            end            
        end
        
        par_init = analysis.parameters([1 end-1]);
        fixedPars = analysis.parameters(2:end-2);
        
        logstr = ['fitting block ' num2str(b) ' of file ' dataS.filename];
        appendSucroseLog(logstr, 'info');
        
        [par_extended, cost_extended] = fit_DnRsucrose(fit_time, fit_EPSC, par_init, settings.globalFitType, fixedPars);
        
        analysis.label_extended = settings.globalFitType;      
        analysis.parameters_extended = par_extended;
        analysis.cost_extended       = cost_extended;
        analysis.useWeightedFit = settings.useWeightedFit;
        
        if conc == maxSucrose
            if b == 1
                if isfield(dataS, 'blockAnalysis') && ~isempty(dataS.blockAnalysis) && numel(dataStruct(linkIdx).blockAnalysis) > 1
                    dataS.blockAnalysis(b) = analysis; %Store fitting results for concentration >= 500mM
                else
                    dataS.blockAnalysis    = analysis; %Store fitting results for concentration >= 500mM
                end
            else
                dataS.blockAnalysis(b) = analysis; %Store fitting results for concentration <500mM
            end
            
            %store the linked cost as well
            for n = 1:numel(dataS.linkedFile)
                linkIdx = find(strcmp(linkedFiles{n}, filenames));
                if b == 1
                    if isfield(dataStruct(linkIdx), 'blockAnalysis') && ~isempty(dataStruct(linkIdx).blockAnalysis) && numel(dataStruct(linkIdx).blockAnalysis) > 1
                        dataStruct(linkIdx).blockAnalysis(b) = analysis; %Store fitting results for linked 500mM data
                    else
                        dataStruct(linkIdx).blockAnalysis    = analysis; %Store fitting results for linked 500mM data
                    end
                else
                    dataStruct(linkIdx).blockAnalysis(b) = analysis; %Store fitting results for linked 500mM data
                end
            end
            
        else 
            if b == 1
                if isfield(dataS, 'blockAnalysis') && ~isempty(dataS.blockAnalysis) && numel(dataStruct(linkIdx).blockAnalysis) > 1
                    dataS.blockAnalysis(b) = analysis; %Store fitting results for concentration >= 500mM
                else
                    dataS.blockAnalysis    = analysis; %Store fitting results for concentration >= 500mM
                end
            else
                dataS.blockAnalysis(b) = analysis; %Store fitting results for concentration >= 500mM
            end
        end
        
    end
catch me
    appendSucroseLog(me, 'urgent');
    appendSucroseLog(['Debug ' getLineNumber()], 'urgent');
    updateStatusbar('An error occurred during refitting...');
    watchoff(h);
    enableDisableFig(h, true);
end

watchoff(h);
enableDisableFig(h, true);

%add a line to the log (for performance measurement)
appendSucroseLog('Finished fitting data.');

%finished refitting, store the data back into its structure
dataStruct(selIdx) = dataS;
setappdata(h, 'dataStruct', dataStruct);
updateStatusbar('Finished refitting.');
if nargin == 0
    updateInterface();
end


% --------------------------------------------------------------------
function menu_plot_extended_cost_Callback(hObject, eventdata, handles)
h = getappdata(0, 'h_sucrose');

%get the relevant data
dataStruct = getappdata(h, 'dataStruct');
settings   = getappdata(h, 'settings');
filenames  = {dataStruct.filename};

%get the handles
handles = getappdata(h, 'handles');
%get the currently selected file from the file list
currFilename = get(handles.list_files, 'String');
currFilename = currFilename(get(handles.list_files, 'Value'));

%find the correct index
selIdx = find(strcmp(currFilename, filenames));

dataS = dataStruct(selIdx);
conc  = dataS.concentration;
maxSucrose = max([dataStruct.concentration]);
si    = dataS.sample_int;
epsc  = dataS.rawdataCorr;

if conc ~= maxSucrose
    linkFile = dataStruct(selIdx).linkedFile{1};
    selIdx = find(strcmp(linkFile, filenames));
end

%ask user for block#
costBlock_prompt = {'Plot cost landscape for sucrose pulse # (blank: 1st pulse)'};
costBlockDlg_title = 'Cost landscape input';
costBlock_answer = inputdlg(costBlock_prompt, costBlockDlg_title, 1);
pause(0.1);
noBlocks = dataStruct(selIdx).nrOfBlocks;

if isempty(costBlock_answer)
    %user canceled, return.
    return;
elseif isempty(costBlock_answer{1})
    costBlock = 1;
else
    costBlock = str2double(costBlock_answer{1});
end

%ask user for number of data points
dataPoints_prompt = {'Number of values per parameter # (blank: 10)'};
dataPointsDlg_title = 'Cost landscape input';
dataPoints_answer = inputdlg(dataPoints_prompt, dataPointsDlg_title, 1);
pause(0.1);

if isempty(dataPoints_answer)
    %user canceled, return.
    return;
elseif isempty(dataPoints_answer{1})
    noDataPoints = 10;
else
    noDataPoints = str2double(dataPoints_answer{1});
    if isnan(noDataPoints)
        noDataPoints = 10; %default value
    else
        noDataPoints = floor(noDataPoints);
    end
end

%check for a linked file
if ~isempty(dataS.linkedFile)
    %there is linked data, get it from the linked file field
    linkedFiles = dataS.linkedFile;
    for i = 1:numel(linkedFiles)
        linkIdx = find(strcmp(linkedFiles{i}, filenames));
        if i==1
            rd = {dataStruct(linkIdx).rawdataCorr};
            rd_corr = rd;
        else
            %concatenate the groups
            rd = [rd, {dataStruct(linkIdx).rawdataCorr}];
            rd_corr = [rd_corr rd];
        end
    end
end
%epsc contains the correct data for cost calculation now

try
    if ~isfield(dataStruct(selIdx),'blockAnalysis')
        dataStruct(selIdx).blockAnalysis = [];
        dataS.blockAnalysis = [];
        dataStruct(selIdx) = dataS;
        setappdata(h, 'dataStruct', dataStruct);
        return;
    end
    
    analysis = dataS.blockAnalysis(costBlock);
    if isfield(analysis, 'useWeightedFit')
       useWeightedFit = analysis.useWeightedFit; 
    else
        useWeightedFit = settings.useWeightedFit;
        appendSucroseLog('Using weighted fit setting', 'urgent');
    end
    
    fltr     = dataS.filters.blck_fltrs(:,costBlock);
    fit_EPSC = {epsc(fltr)};
    fit_time = {(0:numel(fit_EPSC{1})-1)*si};
    
    if ~isempty(dataS.linkedFile)
        for n = 1:numel(dataS.linkedFile)
            linkIdx = find(strcmp(linkedFiles{n}, filenames), 1, 'first');
            linkfltr = dataStruct(linkIdx).filters.blck_fltrs(:,costBlock);
            
            fit_rd(n) = {rd{n}(linkfltr)};
            time_rd(n) = {(0:numel(fit_rd{n})-1)*si};
        end
        
        if dataS.concentration < maxSucrose
            %current data first, then linked data
            fit_EPSC = [fit_EPSC, fit_rd];
            fit_time = [fit_time, time_rd];
        else
            %linked data first, then original data
            fit_EPSC = [fit_rd, fit_EPSC];
            fit_time = [time_rd, fit_time];
        end
    end
    
    %change the status text
    updateStatusbar('Busy calculating cost landscape...');
    
    logstr = ['Calculating cost landscape for block ' num2str(costBlock) ' of file ' dataS.filename];
    appendSucroseLog(logstr, 'info');
    
    k1D_opt = analysis.parameters_extended([1 end-1]);
    fixedPars = analysis.parameters_extended(2:end-2);
    cost_opt = analysis.cost_extended;
    
    if noDataPoints > 1e2
        noDataPoints = 1e2;
    elseif noDataPoints < 1
        noDataPoints = 10;
    end
    nContour = noDataPoints*10;
    
    for n = 1:2
        
        if n == 1
            k1Space = linspace(k1D_opt(1).*0.1,k1D_opt(1)*10,noDataPoints);
            DSpace = linspace(k1D_opt(2).*0.1,k1D_opt(2)*10,noDataPoints);
            %k1Space = linspace(k1D_opt(1).*0.5,k1D_opt(1)*10,noDataPoints);
            %DSpace = linspace(0.01,k1D_opt(2)*1.5,noDataPoints);
        else
            k1Space = linspace(k1D_opt(1).*0.995,k1D_opt(1)*1.005,noDataPoints);
            DSpace = linspace(k1D_opt(2).*0.995,k1D_opt(2)*1.005,noDataPoints);
        end
        
        costMtrx = zeros(noDataPoints);
        
        %Take into account weights used for fitting
        if useWeightedFit
            %use a weight function
            weights = cell(1,numel(fit_EPSC));
            for i = 1:numel(fit_EPSC)-1
                peakRatio = floor(min(fit_EPSC{end})/min(fit_EPSC{i}));
                if peakRatio <= 0
                    peakRatio = 1;
                end
                weights(i) = {sucroseWeightedFit(fit_EPSC{i}, 0.1, peakRatio)};
            end
            weights(end) = {ones( size(fit_EPSC{end}, 1), 1)};
        else
            weights = cell(1,numel(fit_EPSC));
            for i = 1:numel(fit_EPSC)
                weights(i) = {ones(size(fit_EPSC{i}, 1), 1)};
            end
        end
        
        for k = 1:numel(k1Space)
            for d = 1:numel(DSpace)
                costMtrx(k,d) = cost_DnRstate_model(fit_time,[k1Space(k),DSpace(d)],fit_EPSC,0,settings.sucrose_decay, weights, fixedPars);
            end
        end
        
        if n == 1
            figure('name','Overview');
            makeCostPlot(k1Space,DSpace,costMtrx,nContour,true,k1D_opt,cost_opt);
        else
            figure('name','Zoom');
            makeCostPlot(k1Space,DSpace,costMtrx,nContour,true,k1D_opt,cost_opt);
        end
        
    end
    
    updateStatusbar('Finished calculating cost landscape.');
    %{
    if min(size(costMtrx)) > 1 %surf needs a matrix input
        
        %surf(k1Space,DSpace,costMtrx,'LineStyle','none');
        
        %use spline interpolation to create a smooth cost landscape
        [k1, D] = meshgrid(k1Space, DSpace);
        nInterp = 1001; 
        k1i = linspace(k1(1), k1(end), nInterp);
        Di = linspace(D(1), D(end), nInterp);
        [k1i, Di] = meshgrid(k1i, Di);
        costi = interp2(k1, D, costMtrx, k1i, Di, 'spline');
        
        %Plot fit quality as contours
        figure
        %zoomStart = 496;
        %zoomEnd = 505;
        %contour(k1i(zoomStart:zoomEnd,zoomStart:zoomEnd), Di(zoomStart:zoomEnd,zoomStart:zoomEnd), ...
        %    costi(zoomStart:zoomEnd,zoomStart:zoomEnd),300); 
        contour(k1i, Di, costi,300); 
        hold on
        plot(k1D_opt(1),k1D_opt(2),'r.','MarkerSize',20); %Location of the optimum
        
        xlabel('Priming rate k_1 (s^{-1})');
        ylabel('Depot pool D (pC)');
        %zlabel('Cost');
        
        figure;
        zoomStart = 496;
        zoomEnd = 505;
        surf(k1i(zoomStart:zoomEnd,zoomStart:zoomEnd), Di(zoomStart:zoomEnd,zoomStart:zoomEnd), ...
            costi(zoomStart:zoomEnd,zoomStart:zoomEnd), 'LineStyle', 'none', 'FaceColor', 'interp');
        hold on;
        plot3(k1D_opt(1),k1D_opt(2),cost_opt,'r.','MarkerSize',20); %Location of the optimum
        
        %Plot fit quality as 'higher z-value = better fit'
        %surf(k1i, Di, max(max(costi))-costi, 'LineStyle', 'none','FaceColor', 'interp');
        xlabel('Priming rate k_1 (s^{-1})');
        ylabel('Depot pool D (pC)');
        zlabel('Cost');

    else
        %cost 'landscape' consisting of a single point... 
        return; 
    end
    %}
catch me
    appendSucroseLog(me, 'urgent');
    appendSucroseLog(['Debug ' getLineNumber()], 'urgent');
    updateStatusbar('An error occurred while calculating the cost landscape...');
    watchoff(h);
    enableDisableFig(h, true);
end

function makeCostPlot(k1Space,DSpace,costMtrx,nContour,bContour,k1D_opt,cost_opt)

if min(size(costMtrx)) > 1 %surf needs a matrix input
    
    %use spline interpolation to create a smooth cost landscape
    [k1, D] = meshgrid(k1Space, DSpace);
    nInterp = nContour*10;
    if nInterp > 1e3
        nInterp = 1e3;
    end
    
    k1i = linspace(k1(1), k1(end), nInterp);
    Di = linspace(D(1), D(end), nInterp);
    [k1i, Di] = meshgrid(k1i, Di);
    costi = interp2(k1, D, costMtrx, k1i, Di, 'spline');
    
    surf(k1i, Di, costi, 'LineStyle', 'none', 'FaceColor', 'interp');
    hold on;
    plot3(k1D_opt(1),k1D_opt(2),cost_opt,'r.','MarkerSize',20); %Location of the optimum
    
    xlabel('Priming rate k_1 (s^{-1})');
    ylabel('Depot pool D (pC)');
    zlabel('Cost');
    
    %Plot fit quality as contours
    if bContour
        figure('name','Contour zoom');
        vContour = min(min(costi)) + ((0:1/nContour:1).^2).*(max(max(costi))-min(min(costi)));
        vContour = [linspace(0, vContour(1)*0.9, 10) vContour];
        contour(k1i, Di, costi, vContour);
        hold on
        plot(k1D_opt(1),k1D_opt(2),'r.','MarkerSize',20); %Location of the optimum
        
        xlabel('Priming rate k_1 (s^{-1})');
        ylabel('Depot pool D (pC)');
    end
    
else
    %cost 'landscape' consisting of a single point...
    return;
end


% --- Executes on button press in check_extended.
function check_extended_Callback(hObject, eventdata, handles)
% hObject    handle to check_extended (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of check_extended
hMain = getappdata(0, 'h_sucrose');
dataS = getappdata(hMain, 'dataStruct');

if isempty(dataS)
    set(handles.check_extended, 'Value', 0);
    return;
end

updatePlot();

% --------------------------------------------------------------------
function menu_list_refit_extended_Callback(hObject, eventdata, handles)
uiwait(0.1);
fitDataSingle_Extended();


% --------------------------------------------------------------------
function menu_plot_stats_Callback(hObject, eventdata, handles)
% hObject    handle to menu_plot_stats (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
hMain = getappdata(0, 'h_sucrose');
dataS = getappdata(hMain, 'dataStruct');

if isempty(dataS)
    %nothing to do here
    errordlg('This function is not available. No data was loaded.', 'Error');
    return;
end
%load the sucrose statistics interface
sucrose_stats;


% --------------------------------------------------------------------
function plot_menu_newWindow_Callback(hObject, eventdata, handles)
% hObject    handle to plot_menu_newWindow (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
if get(handles.check_showRaw, 'Value')
    level = 1;
else
    level = 2;
end
updatePlot(level,true);