%%
%% Existence of oscillations in terms of K3 and Td
%%
clear;
close all;

% nominal parameters (gene 2 and 3)
N = 3; % number of genes
nu = 2; % hill coefficient for gene 2 and 3
a = log(2)/8; %mRNA half life 8min
b = log(2)/90; % protein half life 90min
c = 0.5; %TL rate
beta = 0.4; %TX rate
plasmid_conc = 5; %plasmid concentration (nm)
K = 5; %MM constant

%%
%% main code
%%
% parameters to be screened
K1 = 1:0.5:100; 

beta = beta*plasmid_conc;

cbeta = c*beta;
deps = 1e-7;
boudnary = zeros(length(K1),1);

for dim1 = 1:length(K1)
    dmin = -0.01;
    dmax = 0.5;
    while(abs(dmax - dmin) > deps)
        d = mean([dmin,dmax]);
        Rsq = cbeta/(a+d)/(b+d);
        eq = equilibrium_point(N, [Rsq,Rsq,Rsq], [K1(dim1), K, K], [nu, nu, nu]);

        %% stability analysis of the unique equilibrium
        fderiv = @(p,nu,K) (1/K)*(-nu*(p/K)^(nu-1))/(1+(p/K)^nu)^2;
        
        % Calculation of loop gain
        f_ave = 1;
        for i=1:N
            if (i==1)
                f_ave = f_ave * fderiv(eq(N),nu,K1(dim1));
            else
                f_ave = f_ave * fderiv(eq(i-1), nu, K);
            end
        end
        f_ave = nthroot(f_ave, N);
        
        % Construct Jacoian and compute eigenvalues
        A_1 = [-a-d 0;1 -b-d];
        A_2 = [0 cbeta*f_ave; 0 0];
        circA = diag(ones(1,N-1),-1); circA(1,N) = 1;
        
        A = kron(eye(N), A_1) + kron(circA, A_2);
        eigval = max(real(eig(A)));
        
        if(eigval < 0) %stable, too much dilution, d should be less than current value
            dmax = mean([dmin, dmax]);
        else
            dmin = mean([dmin, dmax]);
        end
        
    end
    boundary(dim1) = mean([dmin,dmax]);
    
end %dim 1

    figure;
    plot(K1, boundary,'b-');
    hold on;
    xlabel('K_1','FontSize',18);    
    ylabel('d','FontSize',18);
    axis square;

    %% plot Td (=1/eigAreal/log(2))
    figure;
    critical_Td = 1./boundary*log(2);
    
    % to make a polygon
    K1 = [K1, 100, 1];
    critical_Td = [critical_Td, 0,0];
    
    % fill
    fill(critical_Td, K1, [0.85,0.85,0.85]);
    xlabel('Td (min)','FontSize',18);    
    ylabel('K','FontSize',18);
    axis square;
