%%
%% Simulation of cyclic oscillators
%%
clear;
close all;

filename = 'n3_period.csv'; %filename

N = 3; % number of nodes
T_a = [8,8,8]; %mRNA half life time
T_b = 2*[45, 45, 45]; %protein half life time *(1um/hour) e^{-a*60} = 1000
c = 0.5*[1,1,1]; % TL rate (nM/min)
beta = 0.4*[1,1,1]; % TX rate (nM/(min * nM of DNA))
K = 5*[1,1,1]; % MM constant nm
plasmid_conc = 5*[1,1,1]; %[nM]
nu = [2,2,2]; % Hill coeff

T_d = [20,30,40,50,60,70,80,90,100,110]; %Td's to simulate
iv = [30,0,0,  30,0,0]; %initial concentrations [r1,r2,r3,  p1,p2,p3]

simtime = 60*60*2;  %simulation time
sampling = 0.5;

%%
%% main code
%%

a = log(2)./T_a.*ones(1,N); % degradation of mRNA (1/min)
b = log(2)./T_b.*ones(1,N); % degradation of protein (1/min)
beta = plasmid_conc.*beta; % transcription rate (1/min*plasmid conc)

actual_period = zeros(1,length(T_d)); %simulation

for TDloop=1:length(T_d)
    
    d = log(2)/T_d(TDloop); %2/60; %dilution by microfludic rig (1/min)    
    repressilator_ode = @(t,x) repressilator_hetero(t,x,N,a,b,c,d,beta,nu,K);
    [T,Y] = ode45(repressilator_ode, 0:sampling:simtime, iv(1:2*N));
    
    figure;
    plot(T,Y(:,N+1:2*N));
    
    %% Actual frequency (calculate autocorrelation)
    t_begin = 2000; %t_begin must be < simtime.
    sample_begin = ceil(t_begin/sampling);
    tau_max = 1500;
    
    autocor = zeros(1,ceil(tau_max/sampling)-30+1);
    T_cor = zeros(1,ceil(tau_max/sampling)-30+1);
    for tau_sample=30:ceil(tau_max/sampling)
        tmp = 0;
        for i=sample_begin:size(T,1)
            tmp = tmp + Y(i,N+1)*Y(i-tau_sample,N+1);
        end
        autocor(tau_sample-30+1) = tmp;
        T_cor(tau_sample-30+1) = tau_sample*sampling;
    end
    
    [max_cor,arg_max] = max(autocor);
    for i=2:length(autocor)
        if ((autocor(i)-autocor(i-1))*(autocor(i+1)-autocor(i)) < 0 && autocor(i)-autocor(i-1) > 0)
            arg_max = i;
            break;
        end
    end
    
    actual_period(TDloop) = T_cor(arg_max);
    
end

% write to file
csvwrite(filename, [T_d', actual_period']);

% plot
figure;
plot(T_d', actual_period','k');
