// Generalized channel segmentation for all images in a directory.
// This macro recursively goes through directory and detects all images matching a certain image extension.
// Written 03272023 by Michelle J

#@ string(value = '.lif', persist = false, label = "Ext of file to search for matching tif name", description = "Unique extension to find tiff file of slice. E.g. Truncating '.lif' from `325_1_1.lif` allows identification of `325_1_1.tif`") ext 
#@ string(value = 'C2', persist = true, label = "Channel", description = "Channel of the tiff image that will be segmented. e.g. `C1`, `C2`, `C3`, etc... ") channel 
#@ string(value = 'cfos', persist = true, label = "Channel name", description = "Name of channel/marker or protein being stained, e.g. cfos") channel_name 
#@ string(value = '50', persist = true, label = "Bandpass filter larger structures (pix)", description = "Bandpass filter larger structures down to this no. pixels") BandpassLarge 
#@ string(value = '5', persist = true, label = "Bandpass filter smaller structures (pix)", description = "Bandpass filter smaller structures up to to this no. pixels") BandpassSmall
#@ string(value = '8', persist = true, label = "Background subtraction rolling ball radius (pix)", description = "Rolling ball radius used for background subtraction") RollingBallRadius
#@ string(value = '8.0', persist = true, label = "x radius for 3D local maximum filtering (pix)", description = "Rolling ball radius used for background subtraction") ffRadX
#@ string(value = '8.0', persist = true, label = "y radius for 3D local maximum filtering (pix)", description = "Rolling ball radius used for background subtraction") ffRadY
#@ string(value = '1.0', persist = true, label = "z radius for 3D local maximum filtering (pix)", description = "Rolling ball radius used for background subtraction") ffRadZ
#@ string(value = '50', persist = true, label = "3DSpotSegmentation: local diff", description = "A constant difference is used to compute local threshold (ValueOfSeed – dif)") LocalDiff
#@ string(value = '2', persist = true, label = "3DSpotSegmentation: local radius_0", description = "Three circles are drawn. The user defines the radius (in pixels) of each circle, given that the first circle should be located within the object, while the two other circles should be located outside of the object. The mean intensities of the object (within the first circle), and of the background (in between the two other circles) are measured and the threshold calculated. By default, the threshold is the mean of the two mean intensity values (weight=0.5). The user can however shift this parameter: for a weight value of 0.75, the threshold will be closer to the background value.") Rad0
#@ string(value = '10', persist = true, label = "3DSpotSegmentation: local radius_1", description = "Three circles are drawn. The user defines the radius (in pixels) of each circle, given that the first circle should be located within the object, while the two other circles should be located outside of the object. The mean intensities of the object (within the first circle), and of the background (in between the two other circles) are measured and the threshold calculated. By default, the threshold is the mean of the two mean intensity values (weight=0.5). The user can however shift this parameter: for a weight value of 0.75, the threshold will be closer to the background value.") Rad1
#@ string(value = '11', persist = true, label = "3DSpotSegmentation: local radius_2", description = "Three circles are drawn. The user defines the radius (in pixels) of each circle, given that the first circle should be located within the object, while the two other circles should be located outside of the object. The mean intensities of the object (within the first circle), and of the background (in between the two other circles) are measured and the threshold calculated. By default, the threshold is the mean of the two mean intensity values (weight=0.5). The user can however shift this parameter: for a weight value of 0.75, the threshold will be closer to the background value.") Rad2
#@ string(value = '10', persist = true, label = "3DSpotSegmentation: Max Radius (Gaussian fit)", description = "See `https://imagej.net/media/3d-seg-spot-tutorial.pdf` for explanation of parameters.") RadMax
#@ string(value = '50', persist = true, label = "3DSpotSegmentation: Min Volume", description = "See `https://imagej.net/media/3d-seg-spot-tutorial.pdf` for explanation of parameters.") VolMin
#@ string(value = '1000', persist = true, label = "3DSpotSegmentation: Max Volume", description = "See `https://imagej.net/media/3d-seg-spot-tutorial.pdf` for explanation of parameters.") VolMax
#@ string(value = '32', persist = true, label = "No cpus to use", description = "Number of cpus to use") n_cpus
#@ Boolean (value=true, label = "Save segmentation binary?", description ="Yes, to check quality. No, to save space.", persist=true) SaveSegBinary
#@ Boolean (value=true, label = "Save log output?", description ="Save the log output file to the image folder directory.", persist=true) SaveLog
#@ Boolean (value=true, label = "batchmode", description ="Process in batchmode (recommended)", persist=true) BatchMode

setBatchMode(BatchMode);
// Recursively lists the files in a user-specified directory.
dir = getDirectory("Choose a root directory containing images. Can have subfolders. ");
count = 1;
processFolder(dir); 

 function processFolder(dir) {
    list = getFileList(dir);
    
    for (i=0; i<list.length; i++) {
       if (endsWith(list[i], "/"))
          processFolder(""+dir+list[i]);
       else
          if (endsWith(list[i], ext)){
          	print((count++) + ": " + dir + list[i]);
          	path = dir + list[i];
          	processFile(dir, list[i]);
          }          
    }
 }

function processFile(dir, name) {
	
		// Prints file
	print("Processing: " + dir + name);

	// Get file name minus extension
	 dir = replace(dir, "\\", "/");
	 dotIndex = indexOf(name, ".");
     name = substring(name, 0, dotIndex);

	// Opening the image
	id = dir + name + '.tif';
	open(id);
	Stack.getDimensions(width, height, channels, slices, frames) ;
	print("Channels: "+channels);
	
	// Get the time
	start = getTime();
	
	// Get the cfos image title, and directory
	imageTitle = split(getTitle(),".");
	imageName = imageTitle[0];
	print(imageName);

	// split the channels
	selectWindow(imageName+".tif");
	
	if (channels > 1){
		run("Split Channels");
	} else {
		rename(channel +"-" + imageName + ".tif");
	}
	
	selectWindow(channel +"-" + imageName + ".tif");
	close("\\Others"); // Close all the other windows except for the front facing window

	// Bandpass filter channel
	run("Duplicate...", "duplicate");
	selectWindow(channel +"-" + imageName + ".tif");
	run("Bandpass Filter...", "filter_large="+BandpassLarge+" filter_small="+BandpassSmall+" suppress=None tolerance=5 process");
	print("Done FFT");

	// Background subtraction channel
	selectWindow(channel +"-" + imageName + ".tif");
	run("Subtract Background...", "rolling="+RollingBallRadius+" stack");
	print("Done post-FFT background subtraction on channel");

	// Get maxima of z-projection average intensity
	selectWindow(channel +"-" + imageName + ".tif");
	run("Z Project...", "projection=[Average Intensity]");
	selectWindow("AVG_" + channel +"-" + imageName + ".tif");
	run("Find Maxima...", "noise=1 output=[Point Selection]");
	run("Measure");
	run("Summarize");

	// Get mean
	mean = getResult("Mean", nResults-4);
	print("Done mean calculation");
	print(mean);

	// Close windows
	selectWindow("Results");
	run("Close");
	selectWindow("AVG_" + channel +"-" + imageName + ".tif");
	close();

	// pass 3D fast filters
	selectWindow(channel +"-" + imageName + ".tif");
	run("3D Fast Filters","filter=MaximumLocal radius_x_pix="+ffRadX+" radius_y_pix="+ffRadY+" radius_z_pix="+ffRadZ+" Nb_cpus=" + n_cpus);
	print("Done 3D Maxima.");
	
	// Delete below
	print("3D Fast Filters","filter=MaximumLocal radius_x_pix="+ffRadX+" radius_y_pix="+ffRadY+" radius_z_pix="+ffRadZ+" Nb_cpus=" + n_cpus);

	// Use the local mean maxima as seed thresholds for the 3D spot segmentation
	selectWindow(channel +"-" + imageName + ".tif");
	run("3D Spot Segmentation", "seeds_threshold="+(mean)+" local_background="+(mean)+" local_diff="+LocalDiff+" radius_0="+Rad0+" radius_1="+Rad1+" radius_2="+Rad2+" weigth=0.50 radius_max="+RadMax+" sd_value=1.17 local_threshold=Constant seg_spot=Maximum watershed volume_min="+VolMin+" volume_max="+VolMax+" seeds=3D_MaximumLocal spots=[" + channel + "-" +imageName+"] radius_for_seeds=2 output=[Label Image] 32-bits");
	print("Done seg");
	
	// Set up 3D manager
	run("3D Manager Options", "volume surface compactness 3d_moments integrated_density mean_grey_value std_dev_grey_value mode_grey_value minimum_grey_value maximum_grey_value centroid_(pix) use distance_between_centers=10 distance_max_contact=1.80 drawing=Contour");
	run("3D Manager");
	
	// Measure the results
	// Workaround for different window naming conventions
	if (isOpen("seg")){
		selectWindow("seg");
	} else {
		selectWindow("Index");
	}
	

	// Quantify the measurements of segmented object masks  on the original channel pixel intensities
	Ext.Manager3D_AddImage();
	selectWindow(channel +"-" + imageName + "-1.tif");
	Ext.Manager3D_Measure();
	Ext.Manager3D_SaveResult("M", dir+channel+"_"+channel_name+"_"+imageName+".txt");
	Ext.Manager3D_Quantif();
	Ext.Manager3D_SaveResult("Q", dir+channel+"_"+channel_name+"_"+imageName+"_"+channel_name+ ".txt");
	Ext.Manager3D_CloseResult("All");
	// Modification to avoid recounting?
	Ext.Manager3D_Reset(); 
	Ext.Manager3D_Close();
	
	// Workaround for different naming conventions
	if (isOpen("seg")){
		selectWindow("seg");
	} else {
		selectWindow("Index");
	}
	
	// Save the binary segmentation image
	if (SaveSegBinary){
		saveAs("Tiff", dir + imageName + "_" + channel_name+"_SpotSegmentation.tif");
	}
	
	// Clean up 
	print("Done saving images");
	run("Close All");
	
	// Timekeep
	end = getTime();
	time = end-start;
	print("Run time for image= "+time/3600000+"h "+time/60000+"m "+time/1000+"s.");

	// Save log
	selectWindow("Log");
	if (SaveLog){
		saveAs("text", dir+imageName+"_3DSpotSegmentation_" + channel_name + "_Log.txt");
	}
	run("Close");
}
