// Colocalization macro
// 4.21.2023 Michelle J
// Colocalization is done one image A "C2_SpotSegmentation" and image_b = "Fast_G_C1_LabelImage". This automatically skips  colocalization of an image if segmentation data for both channels is not present. 
// Input: Root directory where subfolders contain the C2 & C1 segmentation tiffs. The image names are based off the original .lif file name stem.
// Output: Saved colocalization files with the extensions "ColocOnlyTable.txt", "Coloc_B_C1.txt", "Coloc_A_C2.txt"
// Recursively lists the files in a user-specified directory.
#@ string(value = '.lif', persist = true, label = "Ext of file to search for matching tif name", description = "Unique extension to find tiff file of slice. E.g. Truncating '.lif' from `325_1_1.lif` allows identification of `325_1_1.tif`") ext 
#@ string(value = '_Fast_G_C1_LabelImage.tif', persist = true, label = "C1 segmentation file ext", description = "What is the identifying string and extension for your C1 segmentation image?") C1_seg_ext
#@ string(value = '_Fast_R_C2_SpotSegmentation.tif' , persist = true, label = "C2 segmentation file ext", description = "What is the identifying string and extension for your C2 segmentation image?") C2_seg_ext
#@ Boolean (value=true, label = "Save log output?", description ="Save the log output file to the image folder directory.", persist=true) SaveLog

setBatchMode(true);
 dir = getDirectory("Choose a Directory ");
 count = 1;
 processFolder(dir); 

 function processFolder(dir) {
    list = getFileList(dir);
    
    for (i=0; i<list.length; i++) {
       if (endsWith(list[i], "/"))
          processFolder(""+dir+list[i]);
       else
          if (endsWith(list[i], ext)){
          	print((count++) + ": " + dir + list[i]);
          	path = dir + list[i];
          	processFile(dir, list[i]);
          }          
    }
 }

function processFile(dir, name) {
	// Boolean to exit in case both segmentation files don't exists
	seg_files_exist = 1;
	
	// Prints file
	print("Processing: " + dir + name);

	// Get file name minus extension
	 dir = replace(dir, "\\", "/");
	 dotIndex = indexOf(name, ".");
     name = substring(name, 0, dotIndex); 
	 s = getTime();
	 	
	// Multichannel C1 label image path
	id = dir + name + C1_seg_ext;
	if (File.exists(id) == 0){
		seg_files_exist = 0;
		print("C1 segmentation file doesn't exist! Skipping colocalization of image...");
	} else {
		open(id);
		C1LabelImage = getTitle();
		C1Dir = getInfo("image.directory");
		imageTitle = split(getTitle(),".");
		C1ImageName = imageTitle[0];
	}
	
	// Multichannel C2 label image path
	id = dir + name + C2_seg_ext;
	if (File.exists(id) == 0){
		seg_files_exist = 0;
		print("C2 segmentation file doesn't exist! Skipping colocalization of image...");
	} else if (seg_files_exist ){
		open(id);
		C2LabelImage = getTitle();
		C2Dir = getInfo("image.directory");
		imageTitle = split(getTitle(),".");
		C2ImageName = imageTitle[0];
	}

	if (seg_files_exist){

		// Run the colocalization scripts
		print("Running colocalization");
		run("3D MultiColoc", "image_a="+C2ImageName+" image_b="+C1ImageName+" results=[Only colocs]");
		
		// Accomodate different colocalization window names based on 3D ImageJ suite updates
		if (isOpen("Coloc only")){
			selectWindow("Coloc only");
		} else {
				selectWindow("Colocalisation");
		}
				
		saveAs("text", dir+name+"_ColocOnlyTable.txt");
		run("Close");
		print("Done colocalization");
	
		// Measure C1 cell counts
		selectWindow(C1LabelImage);
		run("3D Manager");
		Ext.Manager3D_AddImage();
		Ext.Manager3D_Measure();
		Ext.Manager3D_SaveResult("M", dir+name+"_Coloc_B_C1.txt");
		Ext.Manager3D_CloseResult("All");
		Ext.Manager3D_Reset(); 
		Ext.Manager3D_Close();
		
		// Measure C2 cell counts
		selectWindow(C2LabelImage);
		run("3D Manager");
		Ext.Manager3D_AddImage();
		Ext.Manager3D_Measure();
		Ext.Manager3D_SaveResult("M", dir+name+"_Coloc_A_C2.txt");
		Ext.Manager3D_CloseResult("All");
		Ext.Manager3D_Close();
		
		//Print the time, save the log, close all
	    f = getTime();
	    print("Time =",(f-s)/1000,"seconds");
	    
	    if (SaveLog){
	    	selectWindow("Log");
			saveAs("text", dir+name+"_coloc_Log.txt");
			run("Close");
		}	
	}

	// Exit everything
	print("Done all!");
	run("Close All");
	
}
