function fcn = makeConstrainToRectFcn(type,xlim,ylim)
%makeConstrainToRectFcn Create rectangularly bounded drag constraint function.
%   fcn = makeConstrainToRectFcn(type,xlim,ylim) creates a drag constraint
%   function for draggable tools of a given type, where type is the string:
%   'impoint','imline', or 'imrect'. The rectangular boundaries of the
%   drag constraint function are described by the vectors xlim and
%   ylim where xlim = [xmin xmax] and ylim = [ymin ymax].
%
%   Example
%   -------
%   % Constrain drag of impoint within axis limits 
%   figure, plot(1:10);
%   h = impoint(gca,2,6);
%   api = iptgetapi(h);
%   fcn = makeConstrainToRectFcn('impoint',get(gca,'XLim'),get(gca,'YLim'));
%   api.setDragConstraintFcn(fcn);
%
%   See also impoint, imrect, imline, imdistline 
  
%   Copyright 2005 The MathWorks, Inc.  
%   $Revision: 1.1.6.1 $  $Date: 2005/12/12 23:22:22 $


iptcheckstrs(lower(type),...
            {'imline','imrect','impoint'},...
            mfilename,'type',1);
        
iptcheckinput(xlim,{'numeric'},{'vector'},mfilename,'xlim',2);
iptcheckinput(ylim,{'numeric'},{'vector'},mfilename,'ylim',3);

if length(xlim) ~=2
    error('Images:makeConstrainToRectFcn:xLimSize', ...
        'XLIM must be specified as a two-element vector.');
end

if length(ylim) ~=2
    error('Images:makeConstrainToRectFcn:yLimSize', ...
        'YLIM must be specified as a two-element vector.');
end
            
switch lower(type)
    case 'imline'
        fcn = @constrainLineToRect;
    case 'imrect'
        fcn = @constrainRectToRect;
    case 'impoint'
        fcn = @constrainPointToRect;
end

%Store previous position matrix for use in constrainLineToRect.  Create NaN
%matrix to ensure any comparison of positions will yield false.
line_pos_last = NaN .*zeros(2,2);

    %-----------------------------------------
    function new_pos = constrainLineToRect(pos)
        
        is_end_point_drag = any(pos(:,1) == line_pos_last(:,1)) &&...
                            any(pos(:,2) == line_pos_last(:,2));
        
        if is_end_point_drag
            new_pos = [constrainPointToRect(pos(1,:));constrainPointToRect(pos(2,:))];
        else

            %define rectangle enclosing line
            x_min = min(pos(:,1));
            y_min = min(pos(:,2));

            w = abs(diff(pos(:,1)));
            h = abs(diff(pos(:,2)));

            %Sort points to find left most endpoint
            pos = sortrows(pos,1);
            theta = atan2(diff(pos(:,2)),diff(pos(:,1)));

            rect_pos = constrainRectToRect([x_min y_min w h]);

            xmin = rect_pos(1);
            ymin = rect_pos(2);
            w = rect_pos(3);
            h = rect_pos(4);

            % Line is oriented along one of two diagonals of the enclosing
            % rectangle.  Determine orientation using sign of arctan.
            if theta >= 0
                new_pos = [xmin, ymin; xmin+w, ymin+h];
            else
                new_pos = [xmin, ymin+h; xmin + w, ymin];
            end
        end
        
        line_pos_last = new_pos;
    end

    %-----------------------------------------
    function new_pos = constrainRectToRect(pos)

        x_min = pos(1);
        y_min = pos(2);
        w     = pos(3);
        h     = pos(4);

        x_min = min( xlim(2) - w, max(x_min, xlim(1)) );
        y_min = min( ylim(2) - h, max(y_min, ylim(1)) );

        new_pos = [x_min y_min w h];
    end

    %------------------------------------------
    function new_pos = constrainPointToRect(pos)

        x_candidate = pos(1);
        y_candidate = pos(2);

        x_new = min( xlim(2), max(x_candidate, xlim(1) ) );
        y_new = min( ylim(2), max(y_candidate, ylim(1) ) );

        new_pos = [x_new y_new];

    end

end
