import sys
sys.path.append('py/')
sys.path.append('py/helpers/')
import brianutils
from brianutils import units

def eval_func(M, eq, statemon=None, i=None, **kwargs):
    """Evaluates equation from dictionary with model equations.

    Parameters
    ----------
    M : dict
        dictionary with model equations
    eq : string
        equation to evaluate
    statemon : brian2.StateMonitor, optional
        brian2 statemonitor to use as inputs to solve eq (default is None)
    i : int, optional
        neuron index to take from statemon (default is None)
    **kwargs
        inputs to solve eq

   Returns
   -------
   numpy array
       a numpy array with solution to eq using defined inputs
   """

    # look in functions and definitions of M
    if eq in M["definitions"]:
        eq = M["definitions"][eq]
    elif eq in M["functions"]:
        eq = M["functions"][eq]
    elif eq in M["parameters"]:
        if eq not in statemon.get_states().keys():
            return eval(M["parameters"][eq],units)
    else:
        print("error, function %s not found in given model"%eq)

    bifpars = {}
    inp = {}
    inp.update(units)

    if statemon:
        # use statemon as input, remove any given inputs from M.
        # look for vars in statemon that arent in M.ode. then create variable and (temorarily delete from params..)
        for j in M["init_states"].keys():
            if i is not None:
                inp.update({j : eval('statemon.%s[%d]' % (j,i))})
            else:
                inp.update({j : eval('statemon.%s' % j)})

        try:
            rec_var = statemon.record_variables
        except:
            rec_var = statemon.__dict__.keys()

        for v in rec_var:
            if v in M["parameters"].keys():
                if i is not None:
                    inp.update({v : eval('statemon.%s[%d]' % (v,i))})
                else:
                    inp.update({v : eval('statemon.%s' % v)})
                bifpars.update({v: M["parameters"][v]})
                M["parameters"].pop(v)

    for v in kwargs:
        if v in M["parameters"].keys():
            bifpars.update({v: M["parameters"][v]})
            M["parameters"].pop(v)

    inp.update(kwargs)
    if i is not None:
        inp.update({'i':i})

    out = eval(str(brianutils.rsubs(eq,M["definitions"],M["functions"],M["parameters"],r=3)).replace('int',''),inp)
    M["parameters"].update(bifpars)

    return out

def init_states(ode,neurons,M):
    """Sets initial states of neurons

    Parameters
    ----------
    ode : brian2.Equations
        model equations
    neurons : brian2.NeuronGroup
        brian2 simulation object
    M : dict
        dictionary with model equations
   """
    for k in ode.diff_eq_names:
        setattr(neurons,k,eval(M['init_states'][k],units))

def create_equation_obj(M,newbifpar=None):
    """Creates brian2 equation object from dict

    Parameters
    ----------
    M : dict
        dictionary with model equations
    newbifpar : dict
        dictionary with parameters that are to be set as variables

    Parameters
    ----------
    ode : brian2.Equations
        model equations in brian2
    bifpar : dict
        all variables
   """
    # create a brian2 Equation object from the dictionary and add bifurcation params
    ode = brianutils.load_model(M,newbifpar)
    bifpar= dict(((i,j) for i,j in M["parameters"].items() if i in ode.parameter_names))
    bifpar.update(newbifpar)
    return ode, bifpar
