rm(list = ls())

library("brms")
library("ggplot2")
library("dplyr")
library("ggdist")
library("lme4")

setwd("C:/Users/guyze/OneDrive/Documents/moths/T_maze_experiment") 

df <- read.csv("Coridor_experiment_notebook.csv")
df <- data.frame(
  individual = factor(df$Moth.number),
  distance  = df$Cluster.s.location - 75,  # Centers distance around 0
  egg_count  = df$Number.of.eggs.in.the.cluster*(0.012/243),
  day = factor(df$Date),
  cluster = factor(df$Cluster_number),
  Speaker_side = as.integer(df$Speaker.s.location=='West'),
  cluster_count = rep(1,nrow(df))
)

scaling_factor <- 243 / 0.012  #  scaling factor

ggplot() +
  stat_density(aes(x = distance, y = after_stat(density) , weight = egg_count), 
               data = df, geom = "line", color = "black", size = 3) +
  stat_density(aes(x = distance, y = after_stat(density), weight = cluster_count), 
               data = df, geom = "line", color = "black", linetype = "dotdash", size = 3) +
  geom_point(aes(x = distance, y = egg_count), data = df, color = "black", size = 5) +
  labs(x = "Distance (cm)", y = "Egg count (#)") +
  
  scale_x_continuous(breaks = c(-75, 0, 75), limits = c(-80, 80)) +
  
  scale_y_continuous(
    name = "Density",
    labels = scales::number_format(accuracy = 0.001), 
    sec.axis = sec_axis(~ . * scaling_factor, name = "Egg count (#)")  
  ) +
  theme(
    text = element_text(size = 45),  
    plot.title = element_blank(),    
    panel.background = element_blank(),  
    panel.border = element_blank(),   
    axis.line = element_line(size = 1.5), 
    axis.title = element_text(size = 55),
    plot.margin = margin(20, 10, 10, 10)  
  )

# Dividing the data to 3 areas
df$areas <- ifelse(df$distance >= -30 & df$distance <= 30, "Center",
            ifelse(df$distance >= -80 & df$distance < -30, "Speaker",
            ifelse(df$distance > 30 & df$distance <= 80, "Control", NA)))


# Ensure areas is a factor and egg_count is numeric
df$areas <- factor(df$areas, levels = c("Speaker", "Center", "Control"))
df$egg_count <- as.numeric(df$egg_count)

# Calculate mean and median for each group, excluding "Control" for lines
summary_data <- df %>%
  group_by(areas) %>%
  summarize(
    mean_value = mean(egg_count * (243 / 0.012)),
    median_value = median(egg_count * (243 / 0.012)),
    n = n()
  ) %>%
  filter(areas != "Control") # Exclude "Control" for lines

# Convert areas to numeric positions for plotting lines
summary_data <- summary_data %>%
  mutate(x_pos = as.numeric(areas))

ggplot(df, aes(x = areas, y = egg_count * (243 / 0.012))) +
  # Add horizontal line for mean (black)
  geom_segment(
    data = summary_data,
    aes(
      x = x_pos - 0.3, xend = x_pos + 0.3,
      y = mean_value, yend = mean_value
    ),
    inherit.aes = FALSE,
    color = "black", size = 3
  ) +
  # Add horizontal line for median (red)
  geom_segment(
    data = summary_data,
    aes(
      x = x_pos - 0.3, xend = x_pos + 0.3,
      y = median_value, yend = median_value
    ),
    inherit.aes = FALSE,
    color = "red", size = 3
  ) +
  # Add black circles for individual data points
  geom_point(
    position = position_jitter(width = 0.2),
    shape = 21,
    color = "black",
    fill = "black",
    size = 5,
    alpha = 0.7
  ) +
  scale_x_discrete(labels = levels(df$areas)) + # Correct x-axis for discrete categories
  labs(x = "Area", y = "Egg Count (#)") + # Enhanced axis titles
  theme_minimal() +
  theme(
    text = element_text(size = 45),             # General text size
    axis.title = element_text(size = 55),       # Axis title size
    axis.text = element_text(size = 45),        # Axis text size
    axis.line = element_line(size = 1.5),       # Axis line size
    plot.title = element_blank(),               # Remove plot title
    panel.background = element_blank(),         # Remove panel background
    panel.grid = element_blank(),               # Remove grid
    plot.margin = margin(20, 10, 10, 10)        # Adjust plot margins
  )
glm_model <- glm(egg_count ~ areas, data = df, family = poisson()) # Poisson regression for count data
summary(glm_model)




#### Glmm tests----

poisson_glmm <- glmer(egg_count ~ distance +(1 | Speaker_side)+(1|day), 
                      data = df, family = poisson(link = "log"))
summary(poisson_glmm)

