# plotTree.R
# Script for mapping nitrogenase structure data to a phylogenetic tree
# Updated: 2023-09-09, Amanda Garcia (akgarcia3@wisc.edu)

# Clear environment
rm(list = ls())

# Load libraries
library(ggtree)
library(treeio)
library(ggplot2)
library(viridis)
library(dplyr)
library(TDbook)

# Import tree
tree <- read.iqtree("Figure4-SourceData1.tree")

# Attach data
taxa_df <- read.csv("Figure4-SourceData2.csv") # data for tree has first column = tip labels
taxa_df$Taxonomy <- gsub("(?i).*proteobacteria|Acidithiobacillia", "Proteobacteria", taxa_df$Taxonomy)

ka_data <- read.csv("Figure4-SourceData4.csv")
ka_data$nif_id <- gsub("_map", "", ka_data$nif_id)

# PREP METADATA #
ddkk_df <- read.csv("Figure4-SourceData3.csv", sep="\t") # data for tree has first column = tip labels
ddkk_df$original_name <- gsub("Anc_", "", ddkk_df$original_name)
ddkk_df = subset(ddkk_df, select = -c(row) )
names(ddkk_df)[names(ddkk_df) == "original_name"] <- "label"

ddkk_df <- merge(ddkk_df, ka_data[, c("nif_id", "nif_kd_charged_res_frac")],
                 by.x = "label", by.y = "nif_id")

# Calculate new parameters
ddkk_df$percent_buried_area <- ddkk_df$buried_area / ddkk_df$total_area
ddkk_df$percent_polar_area <- ddkk_df$polar_area / ddkk_df$total_area
ddkk_df$percent_buried_polar_area <- ddkk_df$buried_polar_area / ddkk_df$buried_area
ddkk_df$percent_apolar_area <- ddkk_df$apolar_area / ddkk_df$total_area
ddkk_df$percent_buried_apolar_area <- ddkk_df$buried_apolar_area / ddkk_df$buried_area
ddkk_df$percent_charged_charged_contacts <- ddkk_df$charged_charged_contacts / ddkk_df$intermolecular_contacts
ddkk_df$percent_charged_polar_contacts <- ddkk_df$charged_polar_contacts / ddkk_df$intermolecular_contacts
ddkk_df$percent_charged_apolar_contacts <- ddkk_df$charged_apolar_contacts / ddkk_df$intermolecular_contacts
ddkk_df$percent_apolar_polar_contacts <- ddkk_df$apolar_polar_contacts / ddkk_df$intermolecular_contacts
ddkk_df$percent_apolar_apolar_contacts <- ddkk_df$apolar_apolar_contacts / ddkk_df$intermolecular_contacts

tree@data$ancestor <- tree@data$node #generate ancestor labels from node numbers


########################
# Taxonomy (if needed) #
########################

palette_breaks <-c("Acidobacteria","Actinobacteria","Aquificae","Armatimonadetes",
                   "Atribacteria","Bdellovibrio","Ignavibacteria","Zixibacteria",
                   "Omnitrophica",
                   "Bacteroidetes","Balneolaeota","Caldiserica","Calditrichaeota",
                   "Chlamydiae","Chlymidae","Chlorobi","Chloroflexi",
                   "Chrysiogenetes","Cyanobacteria","Melainabacteria","Deferribacteres",
                   "Deinococcus","Dictyoglomi","Elusimicrobia",
                   "Fibrobacteres","FibrobacteresAcidobacteria","Firmicutes","Fusobacteria","Gemmatimonadetes",
                   "Ignavibacteriae","Kiritimatiellaeota","Lentisphaerae","Nitrospinae",
                   "Nitrospirae","Planctomycetes","Proteobacteria","Spirochaetes",
                   "Synergistetes","Tenericutes","Thermodesulfobacteria",
                   "Thermotogae","Verrucomicrobia","Crenarchaeota","Euryarchaeota",
                   "Thaumarchaeota","DPANN","Hadesarchaea","Korarchaeota",
                   "Nanoarchaeota","Parvarchaeota",
                   "unclassified", "Unclassified","Peregrinibacteria",
                   "CP","CPR","CPR2","PER-ii","GN02","TA06","TM6",
                   "WOR-1","WOR","NC10","OP3X","SM2F11",
                   "Opisthokonta","stramenopiles","Amoebozoa","Euglenozoa",
                   "Archaeplastida","Cryptophyta","Apusozoa","Alveolata",
                   "Rhodophyta","Haptophyceae","Parabasalia","Fornicata",
                   "Heterolobosea")

palette_tax <-c("#51AEE7","#A2DB64","#ADE0ED","#63EB5E",
                "#CDA434","#8A6642","#59351F","#59351F",
                "#B38FE3",
                "#E7F0C9","#9D8F7E","#E3E56C","#6847A1",
                "#B38FE3","#B38FE3","#E6D19D","#B1F1BB",
                "#43EEE3","#6CD8C5","#6CD8C5","#5560E8",
                "#598A55","#E1E0D1","#723CA2",
                "#E5A83F","#E5A83F","#59B0B0","#E9C073","#D7B7AF",
                "#DFE5EE","#7C33EA","#CACAF0","#79EBEF",
                "#9BC4EC","#7C7799","#8281EB","#AB6A3B",
                "#E1E73A","#72C896","#7496E0",
                "#D7E0E5","#9E92B6","#E948E5","#E0407B",
                "#D33AB2","#924E7D","#CF3476","#D36E70",
                "#924E7D","#924E7D",
                "grey", "grey","grey",
                "grey","grey","grey","grey","grey","grey","grey",
                "grey","grey","grey","grey","grey",
                "#781F19","#CB2821","#A2231D","#CB3234",
                "#E55137","#F80000","#9B111E","#CB2821",
                "#C93C20","#F75E25","#F54021","#F54021",
                "#CB3234"
)


# TEST TREE PLOT #

tree_plot <- ggtree(tree, right=TRUE) +
  geom_nodelab(aes(label=node), hjust = -.1, color = "black", size = 2)
tree_plot


# LINK METADATA TO TREE BEFORE RE-ROOTING #
tree_data <- tree %>% 
  left_join(ddkk_df, by=c("label" = "label"))


#Re-root. CHECK node labels are preserved!
tree_root <- root(tree_data, node = 822,edgelabel = FALSE)


### GENERATE PLOTS ###
for (i in tail(colnames(ddkk_df), -14)) {
  plot <- ggtree(tree_root, right=TRUE) +
    geom_point2(aes(fill=!!sym(i), color=is.na(!!sym(i))),
                shape=21, size=4,
                na.rm=TRUE) +
    scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                       guide = guide_colorbar()) +
    scale_fill_viridis(option="G",
                       direction=-1,
                       na.value=NA)
  ggsave(paste("gtreefig.ddkk.",i,".pdf", sep=""), plot = plot,
         width=9, height=6.5, units="in") 
}











tree_ddkk_totarea_plot <- ggtree(tree_root, right=TRUE) +
  geom_point2(aes(fill=total_area, color=is.na(total_area)),
              # color="red",
              shape=21, size=4,
              na.rm=TRUE) +
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  geom_nodelab(aes(label=ancestor), hjust = -.1, color = "black", size = 2) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA)
tree_ddkk_totarea_plot











#Re-root. CHECK node labels are preserved!
tree_root <- root(tree, node = 822,edgelabel = FALSE)
tree_plot <- ggtree(tree_root, right=TRUE) +
  geom_nodelab(aes(label=ancestor), hjust = -.1, color = "black", size = 2)
tree_plot

### VERY IMPORTANT TO RELABEL DATAFRAME AFTER REROOTING ###
ddkk_df$new_label <- tree_root@data$node[match(ddkk_df$old_label, tree_root@data$ancestor)]
ddkk_df <- ddkk_df[, c("new_label", setdiff(names(ddkk_df), "new_label"))]
ddkk_df$new_label[is.na(ddkk_df$new_label)] <- ddkk_df$old_label[is.na(ddkk_df$new_label)]

##########################
# View with structure data
##########################


i <- tail(colnames(ddkk_df), -38)


plot <- ggtree(tree_root, right=TRUE) %<+% ddkk_df +
  geom_point2(aes(fill=!!sym(percent_buried_area, color=is.na(percent_buried_area), data=ddkk_df),
              shape=21, size=4,
              na.rm=TRUE)) +
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA)





for (i in tail(colnames(ddkk_df), -14)) {
  plot <- ggtree(tree, right=TRUE) %<+% ddkk_df +
    geom_point2(aes(fill=!!sym(i), color=is.na(!!sym(i))),
                shape=21, size=4,
                na.rm=TRUE) +
    scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                       guide = guide_colorbar()) +
    scale_fill_viridis(option="G",
                       direction=-1,
                       na.value=NA)
  ggsave(paste("gtreefig.ddkk.",i,".pdf", sep=""), plot = plot,
         width=9, height=6.5, units="in") 
} 



# Total area
tree_ddkk_totarea_plot <- ggtree(tree_root, right=TRUE) %<+% ddkk_df +
  geom_point2(aes(fill=total_area, color=is.na(total_area)),
              # color="red",
              shape=21, size=4,
              na.rm=TRUE) +
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  geom_nodelab(aes(label=node), hjust = -.1, color = "black", size = 2) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA)
tree_ddkk_totarea_plot

ggsave("gtreefig.ddkk.totarea.pdf", plot = tree_ddkk_totarea_plot,
       width=9, height=6.5, units="in")

# Buried area
tree_ddkk_burarea_plot <- ggtree(tree_root, right=TRUE) %<+% ddkk_df +
  geom_point2(aes(fill=buried_area, color=is.na(buried_area)),
              # color="red",
              shape=21, size=4,
              na.rm=TRUE) +
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA)
ggsave("gtreefig.ddkk.burarea.pdf", plot = tree_ddkk_burarea_plot,
       width=9, height=6.5, units="in")

# Percentage buried area
tree_ddkk_perburarea_plot <- ggtree(tree_root, right=TRUE) %<+% ddkk_df +
  geom_point2(aes(fill=percent_buried_area, color=is.na(percent_buried_area)),
              # color="red",
              shape=21, size=4,
              na.rm=TRUE) +
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA)
ggsave("gtreefig.ddkk.perburarea.pdf", plot = tree_ddkk_perburarea_plot,
       width=9, height=6.5, units="in")

# Polar area
tree_ddkk_polararea_plot <- ggtree(tree_root, right=TRUE) %<+% ddkk_df +
  geom_point2(aes(fill=polar_area, color=is.na(polar_area)),
              # color="red",
              shape=21, size=4,
              na.rm=TRUE) +
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA)
ggsave("gtreefig.ddkk.polararea.pdf", plot = tree_ddkk_polararea_plot,
       width=9, height=6.5, units="in")

# Buried polar area
tree_ddkk_burpolararea_plot <- ggtree(tree_root, right=TRUE) %<+% ddkk_df +
  geom_point2(aes(fill=buried_polar_area, color=is.na(buried_polar_area)),
              # color="red",
              shape=21, size=4,
              na.rm=TRUE) +
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA)
ggsave("gtreefig.ddkk.burpolararea.pdf", plot = tree_ddkk_burpolararea_plot,
       width=9, height=6.5, units="in")

# Apolar area
tree_ddkk_apolararea_plot <- ggtree(tree_root, right=TRUE) %<+% ddkk_df +
  geom_point2(aes(fill=apolar_area, color=is.na(apolar_area)),
              # color="red",
              shape=21, size=4,
              na.rm=TRUE) +
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA)
ggsave("gtreefig.ddkk.apolararea.pdf", plot = tree_ddkk_apolararea_plot,
       width=9, height=6.5, units="in")

# Buried apolar area
tree_ddkk_burapolararea_plot <- ggtree(tree_root, right=TRUE) %<+% ddkk_df +
  geom_point2(aes(fill=buried_apolar_area, color=is.na(buried_apolar_area)),
              # color="red",
              shape=21, size=4,
              na.rm=TRUE) +
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA)
ggsave("gtreefig.ddkk.burapolararea.pdf", plot = tree_ddkk_burapolararea_plot,
       width=9, height=6.5, units="in")

# Intermolecular contacts
tree_ddkk_intercontact_plot <- ggtree(tree_root, right=TRUE) %<+% ddkk_df +
  geom_point2(aes(fill=intermolecular_contacts, color=is.na(intermolecular_contacts)),
              # color="red",
              shape=21, size=4,
              na.rm=TRUE) +
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA)
ggsave("gtreefig.ddkk.intercontact.pdf", plot = tree_ddkk_intercontact_plot,
       width=9, height=6.5, units="in")








# Generate plot
tree_plot <- ggtree(tree, right=TRUE) %<+% taxa_df +
  geom_tippoint(aes(color=Taxonomy), size = 2) + 
  scale_color_manual(values=palette_tax,
                     breaks=palette_breaks,
                     labels=palette_breaks,
                     name="ancestor", na.value="black") +
  # geom_nodelab(aes(label=node), hjust = -.1, color = "black", size = 2) +
  geom_treescale() +
  theme(legend.key.size = unit(0.1, 'cm'))
tree_plot

tree_root_plot <- ggtree(tree_root, right=TRUE) %<+% taxa_df +
  geom_tippoint(aes(color=Taxonomy), size = 2) + 
  scale_color_manual(values=palette_tax,
                     breaks=palette_breaks,
                     labels=palette_breaks,
                     name="Taxonomy", na.value="black") +
  # geom_nodelab(aes(label=ancestor), hjust = -.1, color = "black", size = 2) +
  geom_treescale() +
  theme(legend.key.size = unit(0.1, 'cm'))

tree_root_zoom_plot <- scaleClade(tree_root_plot, node=822, scale=.05)
tree_root_zoom_plot <- viewClade(tree_root_zoom_plot, node=822)
tree_root_zoom_plot

##########################
# View with structure data
##########################

tree_ddkk_burarea_plot <- ggtree(tree_root, right=TRUE) %<+% ddkk_df +
  geom_point2(aes(fill=percent_buried_area, color=is.na(percent_buried_area)),
              # color="red",
              shape=21, size=4,
              na.rm=TRUE) +
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA)
  
  
  
  
  
  geom_point2(aes(fill=buried_area, color=is.na(buried_area)),
              # color="black",
              shape=21, size=4,
              na.rm=TRUE) + 
  # geom_tippoint(aes(color=D.D), size = 4, na.rm = TRUE) + 
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA) +
  geom_treescale() +
  theme(legend.key.size = unit(0.5, 'cm'))
tree_ddkk_burarea_plot <- scaleClade(tree_ddkk_burarea_plot, node=822, scale=.05)
tree_ddkk_burarea_plot <- viewClade(tree_ddkk_burarea_plot, node=822, xmax_adjust = 1)


tree_ddkk_kk_plot <- ggtree(tree_root, right=TRUE) %<+% ddkk_df +
  geom_point2(aes(fill=K.K, color=is.na(K.K)),
              # color="black",
              shape=21, size=4,
              na.rm=TRUE) + 
  # geom_tippoint(aes(color=D.D), size = 4, na.rm = TRUE) + 
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA) +
  geom_treescale() +
  theme(legend.key.size = unit(0.5, 'cm'))
tree_ddkk_kk_plot <- scaleClade(tree_ddkk_kk_plot, node=822, scale=.05)
tree_ddkk_kk_plot <- viewClade(tree_ddkk_kk_plot, node=822, xmax_adjust = 1)


tree_ddkk_majordk_plot <- ggtree(tree_root, right=TRUE) %<+% ddkk_df +
  geom_point2(aes(fill=major.D.K, color=is.na(major.D.K)),
              # color="black",
              shape=21, size=4,
              na.rm=TRUE) + 
  # geom_tippoint(aes(color=D.D), size = 4, na.rm = TRUE) + 
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA) +
  geom_treescale() +
  theme(legend.key.size = unit(0.5, 'cm'))
tree_ddkk_majordk_plot <- scaleClade(tree_ddkk_majordk_plot, node=822, scale=.05)
tree_ddkk_majordk_plot <- viewClade(tree_ddkk_majordk_plot, node=822, xmax_adjust = 1)
tree_ddkk_majordk_plot


tree_ddkk_minordk_plot <- ggtree(tree_root, right=TRUE) %<+% ddkk_df +
  geom_point2(aes(fill=minor.D.K, color=is.na(minor.D.K)),
              # color="black",
              shape=21, size=4,
              na.rm=TRUE) + 
  # geom_tippoint(aes(color=D.D), size = 4, na.rm = TRUE) + 
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA) +
  geom_treescale() +
  theme(legend.key.size = unit(0.5, 'cm'))
tree_ddkk_minordk_plot <- scaleClade(tree_ddkk_minordk_plot, node=822, scale=.05)
tree_ddkk_minordk_plot <- viewClade(tree_ddkk_minordk_plot, node=822, xmax_adjust = 1)
tree_ddkk_minordk_plot


tree_dkhh_dk_plot <- ggtree(tree_root, right=TRUE) %<+% dkhh_df +
  geom_point2(aes(fill=D.K, color=is.na(D.K)),
              # color="black",
              shape=21, size=4,
              na.rm=TRUE) + 
  # geom_tippoint(aes(color=D.D), size = 4, na.rm = TRUE) + 
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA) +
  geom_treescale() +
  theme(legend.key.size = unit(0.5, 'cm'))
tree_dkhh_dk_plot <- scaleClade(tree_dkhh_dk_plot, node=822, scale=.05)
tree_dkhh_dk_plot <- viewClade(tree_dkhh_dk_plot, node=822, xmax_adjust = 1)
tree_dkhh_dk_plot


tree_dkhh_dkhh_plot <- ggtree(tree_root, right=TRUE) %<+% dkhh_df +
  geom_point2(aes(fill=DK.HH, color=is.na(DK.HH)),
              # color="black",
              shape=21, size=4,
              na.rm=TRUE) + 
  # geom_tippoint(aes(color=D.D), size = 4, na.rm = TRUE) + 
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA) +
  geom_treescale() +
  theme(legend.key.size = unit(0.5, 'cm'))
tree_dkhh_dkhh_plot <- scaleClade(tree_dkhh_dkhh_plot, node=822, scale=.05)
tree_dkhh_dkhh_plot <- viewClade(tree_dkhh_dkhh_plot, node=822, xmax_adjust = 1)
tree_dkhh_dkhh_plot


tree_dkhh_hh_plot <- ggtree(tree_root, right=TRUE) %<+% dkhh_df +
  geom_point2(aes(fill=HH, color=is.na(HH)),
              # color="black",
              shape=21, size=4,
              na.rm=TRUE) + 
  # geom_tippoint(aes(color=D.D), size = 4, na.rm = TRUE) + 
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  scale_fill_viridis(option="G",
                     direction=-1,
                     # end=0.75,
                     na.value=NA) +
  geom_treescale() +
  theme(legend.key.size = unit(0.5, 'cm'))
tree_dkhh_hh_plot <- scaleClade(tree_dkhh_hh_plot, node=822, scale=.05)
tree_dkhh_hh_plot <- viewClade(tree_dkhh_hh_plot, node=822, xmax_adjust = 1)
tree_dkhh_hh_plot

#####
# Highlight targeted nodes

tree_nodes <- ggtree(tree_root, right=TRUE) %<+% dkhh_df +
  geom_point2(aes(fill=is.na(HH), color=is.na(HH)),
              # color="black",
              shape=21, size=4,
              na.rm=TRUE) + 
  # geom_tippoint(aes(color=D.D), size = 4, na.rm = TRUE) + 
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  # scale_fill_manual(values = c("yellow"),
  #                    na.value=NA) +
  scale_fill_manual(values = c("TRUE" = "transparent", "FALSE" = "#FFC000"),
                     guide = guide_colorbar()) +
  geom_treescale() +
  theme(legend.key.size = unit(0.5, 'cm'))
tree_nodes <- scaleClade(tree_nodes, node=822, scale=.05)
tree_nodes <- viewClade(tree_nodes, node=822, xmax_adjust = 1)
tree_nodes


#####
# Highlight targeted nodes -- ancestor vs extant

tree_nodes_2 <- ggtree(tree_root, right=TRUE) %<+% dkhh_df +
  geom_point2(aes(fill=type, color=is.na(HH)),
              # color="black",
              shape=21, size=4,
              na.rm=TRUE) + 
  # geom_tippoint(aes(color=D.D), size = 4, na.rm = TRUE) + 
  scale_color_manual(values = c("TRUE" = "transparent", "FALSE" = "black"),
                     guide = guide_colorbar()) +
  # scale_fill_manual(values = c("yellow"),
  #                    na.value=NA) +
  scale_fill_manual(values = c("Anc" = "#FFC000",
                               "Nif" = "lightblue",
                               "Vnf" = "lightblue",
                               "Anf" = "lightblue"),
                    na.value=NA) +
  geom_treescale() +
  theme(legend.key.size = unit(0.5, 'cm'))
tree_nodes_2 <- scaleClade(tree_nodes_2, node=822, scale=.05)
tree_nodes_2 <- viewClade(tree_nodes_2, node=822, xmax_adjust = 1)
tree_nodes_2

# Save tree plot
ggsave("gtreefig.taxa.pdf", plot = tree_root_zoom_plot,
       width=9, height=6.5, units="in")
ggsave("gtreefig.ddkk.dd.pdf", plot = tree_ddkk_dd_plot,
       width=9, height=6.5, units="in")
ggsave("gtreefig.ddkk.kk.pdf", plot = tree_ddkk_kk_plot,
       width=9, height=6.5, units="in")
ggsave("gtreefig.ddkk.majordk.pdf", plot = tree_ddkk_majordk_plot,
       width=9, height=6.5, units="in")
ggsave("gtreefig.ddkk.minordk.pdf", plot = tree_ddkk_minordk_plot,
       width=9, height=6.5, units="in")
ggsave("gtreefig.dkhh.dk.pdf", plot = tree_dkhh_dk_plot,
       width=9, height=6.5, units="in")
ggsave("gtreefig.dkhh.dkhh.pdf", plot = tree_dkhh_dkhh_plot,
       width=9, height=6.5, units="in")
ggsave("gtreefig.dkhh.dkhh.pdf", plot = tree_dkhh_dkhh_plot,
       width=9, height=6.5, units="in")
ggsave("gtreefig.dkhh.hh.pdf", plot = tree_dkhh_hh_plot,
       width=9, height=6.5, units="in")
ggsave("gtreefig.nodes.pdf", plot = tree_nodes,
       width=9, height=6.5, units="in")
ggsave("gtreefig.nodes2.pdf", plot = tree_nodes_2,
       width=9, height=6.5, units="in")


