
function AnalyseFilopodialDynamics_submission

    %
    % Input dataset required as exported from Imaris:
    % Column A: Filopodia length, Column C: Filopodia angle 
    % Column F: Time, Column G: TrackID, Column H: ID
    %
    close all;
    clear all;
   
    Length_INDEX = 1;
    Angle_INDEX = 2;
    Time_Index = 3;
    TrackId_Index = 4;
    
    [samples]=getExcelFiles;   % Add in this function all your excel files
    [numberOfSamples, trash] = size(samples);
    
    
    %
    %   ask the user for extension/retraction step that should be ignored.
    %   eg. ignore extension/retraction less than 0.3 
    %
    dlg_title = 'Ignore Step Length';
    prompt = {'Ignore extension/retraction steps <= X'};
    defaultValue = {'0.3'};
    answer = inputdlg(prompt,dlg_title,1,defaultValue);
    ignoreStepLength = str2double(answer);

    
    for s=1:numberOfSamples   
        fprintf('Reading Coordinates of Sample %d: %s\n', s, samples{s}.fileName);
        
        
        excelFileName = [samples{s}.pathName samples{s}.fileName];
 
    
        %
        %   Read data from Excel File
        %   Numeric values are not recognized. Use rawData instead. Every
        %   empty cell is a NaN.
        %   
        
        nubmerOfDataPoints = length(xlsread(excelFileName, samples{s}.Length));
        
        dataMatrix = nan( nubmerOfDataPoints, 4);
        dataMatrix(:,Length_INDEX)  = readExcelDataIncludingEmptyCells(excelFileName, 1, samples{s}.Length );
        dataMatrix(:,Angle_INDEX)   = readExcelDataIncludingEmptyCells(excelFileName, 1, samples{s}.Angle);
        dataMatrix(:,Time_Index)    = readExcelDataIncludingEmptyCells(excelFileName, 1, samples{s}.Time);
        dataMatrix(:,TrackId_Index) = readExcelDataIncludingEmptyCells(excelFileName, 1, samples{s}.TrackID);
        
        
        uniqTrackId = unique(dataMatrix(:,TrackId_Index));
        uniqTrackId_noNans = uniqTrackId( ~isnan(uniqTrackId));
        
        imagingTimeWindow.startTime = 1;
        imagingTimeWindow.endTime = max(dataMatrix(:,Time_Index));
        
        heatmapMatrix = zeros(imagingTimeWindow.endTime, 360);
        
        %%
        %   1. for every trackID that exists.
        %
        usedAngles=[];  % [ angle id startTimePoint endTimePoint; ]
        statistics=[];  % [ trackId angle ... ]
        
        for id=1:length(uniqTrackId_noNans)
            
            
            %
            %   filter data: One track-id at a time
            %
            trackID_Indices = logical(dataMatrix(:,TrackId_Index) == uniqTrackId_noNans(id));
            relevantData = dataMatrix(trackID_Indices,:);
            
            
            %
            %   sort the filter data according to time
            %
            sortedRelevantData = sortrows(relevantData, Time_Index );
            timePoints = sortedRelevantData(:,Time_Index);
            sampleTime.startTime = min(timePoints);
            sampleTime.endTime = max(timePoints);
            
            %
            %   get the angle at time t=1
            %   angle range -180 to 180, therefore transform to 0 to 360
            %
            [tAngle] = transformAngle( uniqTrackId_noNans(id), sortedRelevantData(1,Angle_INDEX) );
            
            %
            %   check whether this angle is used by another Track-ID
            %
            [failure] = checkAngleDuplication(uniqTrackId_noNans(id), tAngle, usedAngles, timePoints);
            
            
            if (failure == false )
                
                %
                %   update heatmapMatrix and usedAngles
                %
                heatmapMatrix(timePoints, tAngle ) = sortedRelevantData(:,Length_INDEX);
                
                
                statistics = [statistics; ...
                                doStatistics( uniqTrackId_noNans(id), ...
                                              tAngle, ...
                                              sortedRelevantData(:,Length_INDEX), ...
                                              imagingTimeWindow, ...
                                              sampleTime, ...
                                              ignoreStepLength)...
                              ];
                
                usedAngles = [usedAngles; tAngle uniqTrackId_noNans(id) timePoints(1) timePoints(end)];
            end
        end
        
        %%
        %   2.for all the nans = no TrackID exists.
        %
        relevantData = dataMatrix(isnan(dataMatrix(:, TrackId_Index)), :);
        [numberOfentries, trash]=size(relevantData);
        fprintf('Length %d\n', length(relevantData));
        fprintf('Size %d %d\n', size(relevantData));
        
        for entry=1:numberOfentries
            
            fakeID = -1 * entry;
            
            %
            %   get the angle at time t=1
            %   angle range -180 to 180, therefore transform to 0 to 360
            %
            [tAngle] = transformAngle( fakeID, relevantData(entry, Angle_INDEX) );
            timePoints = relevantData(entry, Time_Index);
            sampleTime.startTime = min(timePoints);
            sampleTime.endTime = max(timePoints);

            
            %
            %   check whether this angle is used by another Track-ID
            %
            [failure] = checkAngleDuplication(fakeID, tAngle, usedAngles, timePoints);
            
            
            if (failure == false )
                
                %
                %   update heatmapMatrix and usedAngles
                %
                
                heatmapMatrix(timePoints, tAngle ) = relevantData(entry,Length_INDEX);
                
                statistics = [statistics; ...
                                doStatistics( fakeID, ...
                                              tAngle, ...
                                              relevantData(entry,Length_INDEX), ...
                                              imagingTimeWindow, ...
                                              sampleTime, ...
                                              ignoreStepLength )...
                             ];

                
                usedAngles = [usedAngles; tAngle fakeID timePoints(1) timePoints(end)];
            end
        end
        
        
        %
        %   display heatmap
        %
        figureX = figure(1);
        imagesc(-180:1:180,imagingTimeWindow.startTime:imagingTimeWindow.endTime, heatmapMatrix);
        colormap('Jet');
        colorbar();
        caxis([0 7]); %Anything above 7 will appear as the maximum. Edit the range as desired.
        set(gca,'YDir','normal');
        
        title( 'Length (um) vs Time & Angle', 'FontSize', 16, 'FontWeight','bold', 'FontName','Arial');
        xlabel('Angle','FontSize', 16, 'FontWeight','bold', 'FontName','Arial');
        ylabel('Time', 'FontSize', 16, 'FontWeight','bold', 'FontName','Arial');
        
        tifFileName = [samples{s}.pathName 'Heatmap-' samples{s}.fileName(1:end-5)];
        saveas(figureX,  tifFileName ,'tif');
    
        
        %
        %   write to Excel Sheet the statistical results
        %
        excelStatisticFileName = [samples{s}.pathName 'Statistic-' samples{s}.fileName(1:end-5) '.txt'];
        fileID = fopen(excelStatisticFileName, 'w');   % 'w' open file for writing; discard existing contents
        
        fprintf(fileID,['TrackID\t Angle\t'...
                       'Extension Events\tExtension Speed (mean)\tExtension Speed (std)\t' ...
                       'Retraction Events\tRetraction Speed (mean)\tRetraction Speed (std)\t' ...
                       'Total Events\tTotal Speed (mean)\tTotal Speed (std)\t' ...
                       'Filter = %5.2f\t' ...
                       'Filtered Extension Events\t Filtered Extension Speed (mean)\tFiltered Extension Speed (std)\t' ...
                       'Filtered Retraction Events\tFiltered Retraction Speed (mean)\tFiltered Retraction Speed (std)\t' ...
                       'Filtered Total Events\tFiltered Total Speed (mean)\tFiltered Total Speed (std)\t' ...
                       ...
                       'Length (mean)\t Length (std)\t' ...
                       'Final Length\t'...
                       'Start Time\tEnd Time\n'], ignoreStepLength);
                    
        [datasets, trash] = size(statistics);
        for rows=1:datasets
            fprintf(fileID, '%d\t', statistics(rows,1));
            fprintf(fileID, '%d\t', statistics(rows,2));
            fprintf(fileID, '%5.6f\t', statistics(rows,3:11));
            fprintf(fileID, '\t');
            fprintf(fileID, '%5.6f\t', statistics(rows,12:end));
            fprintf(fileID, '\n');
        end
         
        fclose(fileID);
    
    
    end

end


function [samples]=getExcelFiles


    %
    %   Define the the number of samples for analysis
    %
    NUMBER_OF_SAMPLES =2;
    samples = cell(NUMBER_OF_SAMPLES,1);
    index=1;
    
    %%
    %   Add here your other samples. Make sure your filename does
    %   not have any blanks. Also change the variable NUMBER_OF_SAMPLES to 
    %   the actual sample number.
    %   Please do not forget to increment the variable index.
    %
    
     %Examples:                  
    samples{index}.pathName = 'C:\your directory';
    samples{index}.fileName = 'yourdataset1.xlsx';
    samples{index}.excelSheet = 1;  
    samples{index}.Length  = 'A3:A755';    %Start and end row numbers for the values.
    samples{index}.Angle   = 'C3:C755';    
    samples{index}.Time    = 'F3:F755';    
    samples{index}.TrackID = 'G3:G755';    
    
    index=index+1;
    samples{index}.pathName ='C:\your directory';
    samples{index}.fileName = 'your-dataset2.xlsx';
    samples{index}.excelSheet = 1;  
    samples{index}.Length  = 'A3:A631';    
    samples{index}.Angle   = 'C3:C631';    
    samples{index}.Time    = 'F3:F631';    
    samples{index}.TrackID = 'G3:G631';    

end



function [excelData] = readExcelDataIncludingEmptyCells(excelFileName, sheet, range )

    dummyNumber = -9999999999999;
    
    [qnum, qtext, rawData] = xlsread(excelFileName, sheet, range);

    %fprintf('%d) Length of qnum: %d, Length of qtxt: %d, Length of rawdata %d\n', p, length(qnum), length(qtext), length(rawData));

    if (length(qnum) == length(rawData))
        excelData=qnum; % everything is fine, NaN in the middle can be handled !!!
    else
        % NaN and cell2mat does not work. Therefore I have created
        % this workaround. I replace NaNs with a negative number
        % convert everything to a double and replace afterwards the
        % negative number of Nan. 
        [trash, columns] = size(rawData);
        for i=1:length(rawData)
            if ((strcmp(rawData(i,1), 'NaN') == 1) || strcmp(rawData(i,1), ''))
                for j=1:columns
                    rawData{i,j} = dummyNumber;
                end
            end
        end;

        excelData = cell2mat(rawData);
        excelData(logical( excelData==dummyNumber)) = NaN;  % change every negative number to NaNs
    end

end


function [tAngle] = transformAngle( trackId, angle )

    %
    %   get the angle at time t=1
    %   angle range -180 to 180, therefore transform to 0 to 360
    %
    tAngle = round(angle) + 180;
    
    
    fprintf('TrackId; %s; Original Angle; %5.2f; transformed Angle; %5.2f;\n',...
                        num2str(trackId), angle, tAngle);

    
end 


function [failure] = checkAngleDuplication( trackID, angle, usedAngles, timePoints)

    failure = false;
    if (~isempty(usedAngles))
        angleAccepted = usedAngles(:,1) == angle; % logical area

        numberOfDuplicates = sum(angleAccepted);
        if (numberOfDuplicates > 0)
            
            %
            %   Angle was used before!
            %   get vectors of start-times and end -times
            %
            usedAngleTrackIDs = usedAngles(angleAccepted,2);
            usedAngleVector = usedAngles(angleAccepted,1);
            startTimePoints = usedAngles(angleAccepted,3); 
            endTimePoints = usedAngles(angleAccepted,4);   
            
            result = ((logical( timePoints(1) < startTimePoints ) & logical( timePoints(end) < startTimePoints)) | ...
                (logical( timePoints(1) > endTimePoints ) & logical( timePoints(end) > endTimePoints)) );
          
                 
            for i=1:length(usedAngleTrackIDs)  
                fprintf('TrackID; %d; time; (%d-%d);\n', ...
                    usedAngleTrackIDs(i), ...  % Track ID
                    startTimePoints(i),...   % start time Point
                    endTimePoints(i));  % end time pint
                
            end

            if (result == false)
                
                fprintf('---->; TrackId: %d; time; (%d-%d); all extend with t-angle; %d; TrackID: %d ; not used\n', ...
                    trackID, ...
                    timePoints(1), ...
                    timePoints(end), ...
                    usedAngleVector(1), ...    % transformed angle used by both filopdia
                    trackID);
                
                failure = true;
            else
                
             
                fprintf('---->;TrackId: %d; time; (%d-%d); all extend with t-angle; %d; TackID: %d ; intregrated\n', ...
                    trackID, ...
                    timePoints(1), ...
                    timePoints(end), ...
                    usedAngleVector(1), ...    % transformed angle used by both filopdia
                    trackID);
               
                
            end
          
           
        end
    end
end     


function results = doStatistics( trackId, transformedAngle, lengthData, timeWindow, sampleTime, ignoreStepLength)



    correctedLengthData = lengthData;  
    
    %
    % if sample starts extending after first time point then count the first
    % event as extension
    %
    if (sampleTime.startTime > timeWindow.startTime )
        % add '0' at the front of correctedLengthData
        correctedLengthData = [0; correctedLengthData];
    end

    %
    % if sample stops before last time point then count it as a
    % retraction event
    %
    if (sampleTime.endTime < timeWindow.endTime )
         % add '0' at the end of correctedLengthData
         correctedLengthData = [correctedLengthData; 0];
    end
    
    
    %%
    %   1.get the change of filopodial length for each time step
    %
    lengthPerTimeStep = correctedLengthData(2:end)-correctedLengthData(1:end-1);  % Length(t+1)-Length(t)
    finalLength = correctedLengthData(end);    
    
    speedPerTimeStep  = lengthPerTimeStep;  
    retractionSpeedPerTimeStep = lengthPerTimeStep( logical(lengthPerTimeStep<0));
    extensionSpeedPerTimeStep = lengthPerTimeStep( logical(lengthPerTimeStep>0));
    
    
    %%
    %   2. ignore all steps that are smaller equal <ignoreStepLength>
    %   do not ignore the first step the first step!
    %   do not ignore the last step
    %
    ignoreIndices = [logical(abs(lengthPerTimeStep) <= ignoreStepLength )];
    ignoreIndices(1) = 0;
    ignoreIndices(end) = 0;   % last step cannot be ignored!
    
    filteredLengthPerTimeStep = lengthPerTimeStep( ~ignoreIndices );
    
    filteredSpeedPerTimeStep  = filteredLengthPerTimeStep;  
    filteredRetractionSpeedPerTimeStep = filteredLengthPerTimeStep( logical(filteredLengthPerTimeStep<0));
    filteredExtensionSpeedPerTimeStep = filteredLengthPerTimeStep( logical(filteredLengthPerTimeStep>0));
    

    results = [trackId ...
                  transformedAngle-180 ...                % angle
                  length(extensionSpeedPerTimeStep) ...   % number of events
                  mean(extensionSpeedPerTimeStep) ...     % average speed
                  std(extensionSpeedPerTimeStep) ...      % std speed
                    ...
                  length(retractionSpeedPerTimeStep) ...    % number of events
                  mean(abs(retractionSpeedPerTimeStep)) ...      % average speed
                  std(abs(retractionSpeedPerTimeStep)) ...
                    ...
                  length(speedPerTimeStep) ...              % number of events
                  mean(abs(speedPerTimeStep)) ...           % average speed
                  std(abs(speedPerTimeStep)) ...            % std speed
                  ...
                  length(filteredExtensionSpeedPerTimeStep) ...              % number of events
                  mean(abs(filteredExtensionSpeedPerTimeStep)) ...           % average speed
                  std(abs(filteredExtensionSpeedPerTimeStep)) ...            % std speed
                  ...
                  length(filteredRetractionSpeedPerTimeStep) ...              % number of events
                  mean(abs(filteredRetractionSpeedPerTimeStep)) ...           % average speed
                  std(abs(filteredRetractionSpeedPerTimeStep)) ...            % std speed
                  ...
                  length(filteredSpeedPerTimeStep) ...              % number of events
                  mean(abs(filteredSpeedPerTimeStep)) ...           % average speed
                  std(abs(filteredSpeedPerTimeStep)) ...            % std speed
                  ...
                  mean(lengthData) ...
                  std(lengthData) ...
                  ...
                  finalLength ...                           % final Length
                  sampleTime.startTime ...
                  sampleTime.endTime ...
                  ]; 

end





