% Cascade of neural processing orchestrates cognitive control in human frontal cortex
% Hanlin Tang, Hsiang-Yu Yu, Chien-Chen Chou, Nathan E. Crone, Joseph R. Madsen, William S. Anderson, Gabriel Kreiman
%
% This script plots Figure 2, panel A
% Tested with MATLAB 7.14 (R2012a)


% load the figure 2 source data
x = load('figure2_sourceData.mat');

% **Behavioral data** [1 x NUMTRIALS] vectors
% WORD_IDX - which word was displayed (1 = red, 2 = green, 3 = blue)
% COLOR_IDX - which color was displayed (same convention as above)
% CONGRUENT - whether the trial was congruent (1 = congruent, 0 = incongruent).
% STROOP -    whether the trial was stroop task (=1) or reading task (=0)
% ONSET - onset of speech response in milliseconds relative to onset of 
%         word. NaN means no clear response time recorded (e.g. noisy mic).
% ANSWER - response (1 = red, 2 = green, 3 = red)
%          blue). NaN means either no clear answer was provided or multiple answers
%           were provided (e.g. a self-correction trial, as described in Figure 6).
% CORRECT - was the subject correct
% BAD_TRIALS - trials to exclude from analysis (answer = NaN or onset = NaN)
%
% **Field potential data** (in gamma power z-scored)
% POWER_RESPONSE_ALIGNED [nsamples x ntrials] - zscored gamma power aligned
% to response onset.
% TIME_RESPONSE_ALIGNED [1 x nsamples] - time of each recorded sample,
% aligned to response onset.
% 
% POWER_STIMULUS_ALIGNED [nsamples x ntrials] - zscored gamma power aligned
% to stimulus onset.
% TIME_STIMULUS_ALIGNED [1 x nsamples] - time of each recorded sample,
% aligned to stimulus onset.

% graphics settings
color_congruent = [0 0 0]; % black for congruent trials
color_incongruent = [0.58 0.29 0]; % brown for incongruent trials
alpha = 0.3; % transparency of error bars

% compute mean and sem for each condition
idx = x.congruent & x.stroop & x.correct & ~x.bad_trials; % correct trials + congruent trials + stroop task
average_congruent = mean(x.power_stimulus_aligned(:,idx),2);
sem_congruent = std(x.power_stimulus_aligned(:,idx),[],2)/sqrt(sum(idx)); 

idx = ~x.congruent & x.stroop & x.correct & ~x.bad_trials; % correct trials + incongruent trials + stroop task
average_incongruent = mean(x.power_stimulus_aligned(:,idx),2);
sem_incongruent = std(x.power_stimulus_aligned(:,idx),[],2)/sqrt(sum(idx)); 

% plot data (requires the accompanying function shadedErrorBar
figure(1); clf; set(gcf,'Color','w');
shadedErrorBar(x.time_stimulus_aligned,average_congruent,sem_congruent,{'-','Color',color_congruent,'LineWidth',1},alpha); hold on;
shadedErrorBar(x.time_stimulus_aligned,average_incongruent,sem_incongruent,{'-','Color',color_incongruent,'LineWidth',1},alpha);
       
% figure settings
ylim([-0.25 2.5]);
xlim([-0.5 3]);
set(gca,'XTick',0:3);
set(gca,'YTick',0:3);
xlabel('Time (s)');
ylabel('\gamma-power (z-scored)');
box off;
set(gca,'TickDir','out'); 


