%
% Script generating figures for the paper:
%   Tikhonov M (2016) "Community-level cohesion without cooperation".
%
function RUN_ME(override)
% Run with no arguments to generate figures using pre-calculated simulation data files
%
% To calculate everything from scratch, either delete all *.mat files from
% the folder before running this function, or set the override flag to "true":  
%       RUN_ME(true)
% !! This will OVERWRITE all saved data files! !!
%
if nargin<1
    override = false;
end

params = initialize;
data = generateData(params, override);

generate_Fig2(params, data);
generate_Fig3(params, data);
generate_Fig4(params, data);
generate_Fig5(params, data);

end

function params = initialize
params.saveFigs = true;
params.individualFitnessOffset = 0;

params.epsilonSweep.epsilonRange = exp(linspace(log(1e-1), log(1e-5),17));
params.epsilonSweep.N = 10;
params.epsilonSweep.seeds = 10;
params.epsilonSweep.tries = 10;
params.epsilonSweep.K = 100;
params.epsilonSweep.dThresh = 1e-4; % will be multiplied by epsilon
params.epsilonSweep.abdPresenceThresh = 1e-4;

params.fixedEpsilon.epsilon = 1e-3;
params.fixedEpsilon.N = 10;
params.fixedEpsilon.seed = 3; %1 
params.fixedEpsilon.offset = 0; %1 

params.fixedEpsilon.dynamics.selectK = 100;
params.fixedEpsilon.dynamics.selectOutOf = 2^params.fixedEpsilon.N - 1;
params.fixedEpsilon.dynamics.seedOffset = 8; % showcases diversity of trajectories
params.fixedEpsilon.dynamics.tries = 10;
params.fixedEpsilon.dynamics.replicates = 10;
params.fixedEpsilon.dynamics.minAbd = 1e-3;
params.fixedEpsilon.dynamics.maxT = 1e6;

params.fixedEpsilon.subsets.chooseK = 4;
params.fixedEpsilon.subsets.chooseOutOf = 50; 
params.fixedEpsilon.subsets.choiceSeed = 0;
params.fixedEpsilon.subsets.dThresh = 1e-5;
params.fixedEpsilon.subsets.abdPresenceThresh = 0.001;

params.fixedEpsilon.competition.quadrantN = 11;
params.fixedEpsilon.competition.pairsN = 500;
params.fixedEpsilon.competition.pairsN_AllOnAll = 10000;
% For figure 5:
params.otherEpsilon = params.fixedEpsilon;
params.otherEpsilon.epsilon = 0.1;
params.otherEpsilon.seed = 3; %1  

W1 = 15;    % one-column and two-column figures
W2 = 30;

params.labelFontSize = 26;
params.titleFontSize = 14;
params.axesFontSize = 12;

params.Fig4.quadrantColors = [0 1 1;
                              0 0 1;
                              1 0 1;
                              1 0 0];
params.Fig4.colorBoth = [1 1 0];
params.Fig4.colorNeither = [0 0 0];
params.Fig4.colorNA = 0.*[1 1 1];
params.Fig4.quadrantLabelsCoordX = [-1, 0.6];
params.Fig4.quadrantLabelsCoordY = [-105, -30];
params.Fig4.quadLabelFontSize = 20;
params.Fig4.W = W2;
params.Fig4.H = 8.8;
params.Fig4.offAx = 1.7;
params.Fig4.offAy = 1.7;
params.Fig4.wA = 6; % 8
params.Fig4.labelOffY = 0.4;
params.Fig4.labelAOffX = -1.5;
params.Fig4.labelBOffX = -1.6;
params.Fig4.offBx = 2;
params.Fig4.offBy = 1.3;
params.Fig4.wB = 12;
params.Fig4.hB = 2.6; % 3.5
params.Fig4.binN = 8;
params.Fig4.labelDOffX = -1.3;
params.Fig4.hD = 5.6;
params.Fig4.wD = 5.6;
params.Fig4.offDx = 2;
params.Fig4.offDy = 1.4;
params.Fig4.insetX = 0.5;
params.Fig4.insetH = 1;
params.Fig4.insetY = params.Fig4.hD-params.Fig4.insetH-0.1;

params.Fig5.quadrantColors = params.Fig4.quadrantColors;
params.Fig5.colorBoth = params.Fig4.colorBoth;
params.Fig5.colorNeither = params.Fig4.colorNeither;
params.Fig5.colorNA = params.Fig4.colorNA;

params.Fig5.quadrantLabelsCoordX = [-1 1];
params.Fig5.quadrantLabelsCoordY = [-1.9 0.3];
params.Fig5.quadLabelFontSize = params.Fig4.quadLabelFontSize;
params.Fig5.W = 12;
params.Fig5.H = 10.1;
params.Fig5.offAx = 2;
params.Fig5.offAy = 1.3;
params.Fig5.wA = 8; 

params.Fig3.W = 14.5;
params.Fig3.H = 9.5;
params.Fig3.offx = 2.2;
params.Fig3.offy = 1.5;
params.Fig3.w = 7;
params.Fig3.dx = 2;
params.Fig3.w2 = 2.8;
params.Fig3.h2 = 2.8;
params.Fig3.labelAOffX = -1.8;
params.Fig3.labelBOffX = -1.5;
params.Fig3.labelOffY = -0.5;

params.Fig3_sup.W = 14.5;
params.Fig3_sup.H = 10.5;
params.Fig3_sup.offx = 2.7;
params.Fig3_sup.offy = 1.5;
params.Fig3_sup.w = 8;
params.Fig3_sup.w2 = 10;
params.Fig3_sup.w2split = 0.5;
params.Fig3_sup.h2 = 3.4;
params.Fig3_sup.labelBOffX = -2.5;
params.Fig3_sup.labelOffY = 0;

params.Fig2.W = 15.5;
params.Fig2.H = 7.3;
params.Fig2.wA = 5.5;
params.Fig2.hA = 5.5;
params.Fig2.whB = 5.5;
params.Fig2.offAx = 2;
params.Fig2.offBx = 2.2;
params.Fig2.offAy = 1.3;
params.Fig2.offBy = 1.4;
params.Fig2.labelOffY = 0.2;
params.Fig2.labelAOffX = -1.9;
params.Fig2.labelBOffX = -1.5;
params.Fig2.AAxisRangeX = [0 53];
end

function generate_Fig3(params, data)
d = data.fixedEpsilon.convergenceDynamics;
p = params.Fig3;
%%
clf;
set(gcf, 'PaperPositionMode','Manual', 'PaperUnits','Centimeters',...
    'PaperSize', [p.W p.H], 'PaperPosition',[0 0 p.W p.H],...
    'Units','Centimeters','Position',[0 0 p.W p.H]); 

axes('Units','Centimeters',...
    'Position',[p.offx, p.offy, p.w, p.w]);
hold all
col = lines(10);
col(9,:) = [1 0 0];
lw = ones(1,10);
lw(9) = 2.5;
kinkTime9 = 282; % the location of the kink in the 9th trajectory
annotation(gcf,'arrow',[0.3260 0.3309], [0.7856 0.7792]);
for t=1:length(d)
    dInd = (d{t}.ind)/params.fixedEpsilon.epsilon;
    dComm = (d{t}.comm)/params.fixedEpsilon.epsilon;
    plot(dInd, dComm, '.-','LineWidth',lw(t),'Color',col(t,:))
%     Optional check:
%     visualize the kink location to check it is correct
%     if t==9
%         plot(dInd(kinkTime9), dComm(kinkTime9), 'k.', 'MarkerSize', 40)
%     end
end
xOff = (d{1}.ind(end))/params.fixedEpsilon.epsilon;
yOff = (d{1}.comm(end))/params.fixedEpsilon.epsilon;
axis([xOff xOff yOff yOff]+0.06*[-0.7 0.43 -0.34 0.03]);

set(gca,'XTick',2.36:0.02:2.42,'YTick',2.43:0.01:2.45);
xlabel('\langle f \rangle / \epsilon (mean intrinsic performance)');    
ylabel('F/\epsilon (community fitness)');    
adjustSizes(gca,1,params.axesFontSize);
%title('Convergence to equilibrium', 'FontSize', params.titleFontSize);
text(p.labelAOffX,p.w+p.labelOffY,'A', 'Units', 'Centimeters', 'FontSize', params.labelFontSize, 'VerticalAlignment','bottom');
ar = get(gca,'DataAspectRatio');
box on
w=3.5;
axes('Units','Centimeters',...
     'Position',[p.offx+0.6, p.offy+3.1, 1,w]);
hold all
for t=length(d):-1:1
    dInd = (d{t}.ind)/params.fixedEpsilon.epsilon;
    dComm = (d{t}.comm)/params.fixedEpsilon.epsilon;
    plot(dInd, dComm, 'k-','LineWidth',1, 'Color', col(t,:))
end
                
set(gca,'DataAspectRatioMode','manual',...
    'PlotBoxAspectRatioMode','auto',...
    'XLimMode','manual','YLimMode','manual','DataAspectRatio',ar);
axis([2 2.5 2 2.5])
set(gca,'XTick',2.5,'YTick',[]);
text(1.87, 1.95,'2.0');
text(1.87, 2.54,'2.5');
box on
adjustSizes(gca,1,params.axesFontSize-2);
xlabel('\langle f \rangle / \epsilon');    
text(1.87,2.25,'F/\epsilon','Rotation',90,'HorizontalAlignment','center','VerticalAlignment','middle');

annotation(gcf,'arrow',[0.224 0.240], [0.558 0.635],'HeadLength',8,'HeadWidth',8);
annotation(gcf,'arrow',[0.415 0.431], [0.206 0.282]);

% Panel B
xRange = [1e2,1e5];
axes('Units','Centimeters',...
    'Position',[p.offx+p.w+p.dx, p.offy+p.w-p.h2, p.w2, p.h2]);
% Identify species that eventually go extinct; we will plot these traces in
% a different color
c = data.fixedEpsilon.convergenceDynamics;
extinct = c{9}.sol.y(:,end)<1e-4;
semilogx(c{9}.sol.x, c{9}.sol.y(extinct,:), 'r-');
hold on; 
semilogx(c{9}.sol.x, c{9}.sol.y(~extinct,:), 'k-');
% Optional check:
% plot(c{9}.t(kinkTime9)*[1 1], [0 50], 'k--','LineWidth',1); % plot location of kink
annotation(gcf,'arrow',0.9069*[1 1], [0.92 0.90]);
axis([xRange 0 50])
set(gca,'YTick',[0,50],'XTick',xRange)
box on
text(sqrt(prod(xRange)),-3,'time /\tau_0','HorizontalAlignment','center','VerticalAlignment','top', 'FontSize', params.axesFontSize)
text(0.8*xRange(1),25,'abundance','Rotation',90,'HorizontalAlignment','center','VerticalAlignment','bottom', 'FontSize', params.axesFontSize-1);
adjustSizes(gca,1,params.axesFontSize);
text(p.labelBOffX,p.h2+p.labelOffY,'B', 'Units', 'Centimeters', 'FontSize', params.labelFontSize, 'VerticalAlignment','bottom');

% panel C
enz = data.fixedEpsilon.prop.enzymesInSpecies;
axes('Units','Centimeters',...
    'Position',[p.offx+p.w+p.dx, p.offy, p.w2, p.h2]);
hold all
for t=1:10
    totalEnz = enz'*c{t}.sol.y;
    h = 100./totalEnz;
    semilogx(c{t}.t,mean(h),'-','Color',col(t,:),'LineWidth',1);
end
epsilon = params.fixedEpsilon.epsilon;
axRange = [xRange 1-3*epsilon 1];
axis(axRange);
set(gca,'XScale','log','XTick',xRange,'YTick', [1+(-3:0)*epsilon], ...
    'YTickLabel',{'1-3\epsilon', '', '', '1'}, 'TickLength', [0.03,0.025])
text(sqrt(prod(xRange)),axRange(3)-epsilon/4,'time /\tau_0','HorizontalAlignment','center','VerticalAlignment','top', 'FontSize', params.axesFontSize)
text(0.4*xRange(1),1-1.5*epsilon,'resource','Rotation',90,'HorizontalAlignment','center','VerticalAlignment','bottom', 'FontSize', params.axesFontSize);
text(0.8*xRange(1),1-1.5*epsilon,'availability','Rotation',90,'HorizontalAlignment','center','VerticalAlignment','bottom', 'FontSize', params.axesFontSize-1);
box on
adjustSizes(gca,1,params.axesFontSize);
text(p.labelBOffX,p.h2+p.labelOffY,'C', 'Units', 'Centimeters', 'FontSize', params.labelFontSize, 'VerticalAlignment','bottom');
%%
if params.saveFigs
    fname = sprintf('Figure3%s', '');
    saveas(gcf,[fname '.png'])
    saveas(gcf,[fname '.pdf'])
    saveas(gcf,[fname '.fig'])
else
    pause;
end
%%
p = params.Fig3_sup;
% Make the sup. figure.
clf;
set(gcf, 'PaperPositionMode','Manual', 'PaperUnits','Centimeters',...
    'PaperSize', [p.W p.H], 'PaperPosition',[0 0 p.W p.H],...
    'Units','Centimeters','Position',[0 0 p.W p.H]); 

xRange = [1e-2,1e5];
axes('Units','Centimeters',...
    'Position',[p.offx, p.offy+p.w-p.h2, p.w2, p.h2]);
c = data.fixedEpsilon.convergenceDynamics;
semilogx(c{9}.sol.x, c{9}.sol.y, 'k-');
axis([xRange 0 100])
box on
ylabel('abundance');
adjustSizes(gca,1,params.axesFontSize);
text(p.labelBOffX,p.h2+p.labelOffY,'A', 'Units', 'Centimeters', 'FontSize', params.labelFontSize, 'VerticalAlignment','bottom');

enz = data.fixedEpsilon.prop.enzymesInSpecies;
axes('Units','Centimeters',...
    'Position',[p.offx, p.offy, p.w2, p.h2]);
t=9;
totalEnz = enz'*c{t}.sol.y;
h = 100./totalEnz;
loglog(c{t}.t,h,'k-','LineWidth',1);
epsilon = params.fixedEpsilon.epsilon;
axRange = [xRange 1e-2 1.5];
axis(axRange);
set(gca,'XScale','log');
xlabel('time /\tau_0')
ylabel({'resource','availability'});
box on
adjustSizes(gca,1,params.axesFontSize);
text(p.labelBOffX,p.h2+p.labelOffY,'B', 'Units', 'Centimeters', 'FontSize', params.labelFontSize, 'VerticalAlignment','bottom');

axes('Units','Centimeters',...
    'Position',[p.offx+p.w2*p.w2split, p.offy, p.w2*(1-p.w2split), p.h2],...
    'YAxisLocation','right','XScale','log','box','on','XTick',[1e2,1e4],'XTickLabel',{});
hold on
axRange(1)=exp(log(axRange(1))*(1-p.w2split)+log(axRange(2))*p.w2split);
axRange(3:4)=[1-8*epsilon,1+4*epsilon];
axis(axRange);
semilogx(c{t}.t,h,'k-','LineWidth',1);
adjustSizes(gca,1,params.axesFontSize);
%%
if params.saveFigs
    fname = sprintf('FigureS1%s', '');
    saveas(gcf,[fname '.png'])
    saveas(gcf,[fname '.pdf'])
    saveas(gcf,[fname '.fig'])
else
    pause;
end

%%
end

function generate_Fig4(params, data)
d = data.fixedEpsilon.subsetCompetition;
p = params.Fig4;
%%
clf;
set(gcf, 'PaperPositionMode','Manual', 'PaperUnits','Centimeters',...
    'PaperSize', [p.W p.H], 'PaperPosition',[0 0 p.W p.H],...
    'Units','Centimeters','Position',[0 0 p.W p.H]); 

%%% PANEL A %%%
% It appears that MatLab has a bug whereby plotting a patch in panel D
% makes visible points outside plotting area in panel A. To correct this,
% remove points manually.
axes('Units','Centimeters',...
    'Position',[p.offAx, p.offAy, p.wA, p.wA]);
axRange = [-2 2 -170 10];
hold on;
x = d.x/params.fixedEpsilon.epsilon;
y = d.y/params.fixedEpsilon.epsilon;
sel = x>axRange(1) & x<axRange(2) & y>axRange(3) & y<axRange(4);
x = x(sel);
y = y(sel);
noQ = ~any(d.q(:,sel));
plot(x(noQ), y(noQ),'k.')
for i=1:4
    plot(x(d.q(i,sel)), y(d.q(i,sel)), '.','Color', p.quadrantColors(i,:))
end
xlabel('\langle f \rangle / \epsilon (mean individual performance)');    
ylabel('F/\epsilon (community fitness)');
axis(axRange);
set(gca,'YTick',-150:50:0);
axis square
adjustSizes(gca,1,12);
title(sprintf('%d-species communities',...
    params.fixedEpsilon.subsets.chooseK), 'FontSize', params.titleFontSize);
text(p.labelAOffX,p.wA+p.labelOffY,'A', 'Units', 'Centimeters', 'FontSize', params.labelFontSize);
cx = p.quadrantLabelsCoordX;
cy = p.quadrantLabelsCoordY;
%
text(cx(1),cy(1),'I', 'FontSize', p.quadLabelFontSize,'FontName','Times','HorizontalAlignment','center');
text(cx(1),cy(2),'II', 'FontSize', p.quadLabelFontSize,'FontName','Times','HorizontalAlignment','center','Color','w');
text(cx(2),cy(2),'III', 'FontSize', p.quadLabelFontSize,'FontName','Times','HorizontalAlignment','center');
text(cx(2),cy(1),'IV', 'FontSize', p.quadLabelFontSize,'FontName','Times','HorizontalAlignment','center');
%
%%% PANEL B %%%
axes('Units','Centimeters',...
    'Position',[p.offAx+p.wA+p.offBx, p.offAy+p.wA-p.hB, p.wB, p.hB]);
if ~isempty(data.fixedEpsilon.subsetCompetition.competition31)
    tbl = data.fixedEpsilon.subsetCompetition.competition31.summaryTable;
    image(tbl);
    colormap([p.quadrantColors(3,:);
              p.quadrantColors(1,:);
              p.colorBoth;
              p.colorNeither;
              p.colorNA
              p.quadrantColors(2,:);
              p.quadrantColors(4,:);
              p.colorBoth;
              p.colorNeither;
              p.colorNA]);
    set(gca,'YTick',[1 size(tbl,1)],'YTickLabel',{'high','low'},'XTick',[1 params.fixedEpsilon.competition.pairsN])
    adjustSizes(gca,1,params.axesFontSize);
    text((params.fixedEpsilon.competition.pairsN+1)/2,-0.7,'Elimination assay: competing quadrant I vs. III', 'FontSize', params.titleFontSize,'HorizontalAlignment','center');
    text(-0.4,p.hB/2,'\it{f}_\sigma', 'Units', 'Centimeters', ...
         'FontSize', params.axesFontSize,'HorizontalAlignment','center');
    text(p.labelBOffX,p.hB+p.labelOffY,'B', 'Units', 'Centimeters', 'FontSize', params.labelFontSize);
end

%%% PANEL C %%%
axes('Units','Centimeters',...
    'Position',[p.offAx+p.wA+p.offBx, p.offBy, p.wB, p.hB]);
if ~isempty(data.fixedEpsilon.subsetCompetition.competition24)
    tbl = data.fixedEpsilon.subsetCompetition.competition24.summaryTable;
    image(tbl+5);
    set(gca,'YTick',[1 size(tbl,1)],'YTickLabel',{'high','low'},'XTick',[1 params.fixedEpsilon.competition.pairsN])
    adjustSizes(gca,1,params.axesFontSize);
    text((params.fixedEpsilon.competition.pairsN+1)/2,-0.7,'Quadrant II vs. IV', 'FontSize', params.titleFontSize,'HorizontalAlignment','center');
    text(-0.4,p.hB/2,'\it{f}_\sigma', 'Units', 'Centimeters', ...
         'FontSize', params.axesFontSize,'HorizontalAlignment','center');
    text(p.labelBOffX,p.hB+p.labelOffY,'C', 'Units', 'Centimeters', 'FontSize', params.labelFontSize);
    text(p.wB/2,-0.3, 'Competition trials (ordered by dominant color)', ...
        'Units','Centimeters','HorizontalAlignment','center','FontSize', params.axesFontSize);
end
%

allOnAll = data.fixedEpsilon.allOnAllCompetition;
% Get two datapoints out of each comparison
sim = [allOnAll.similarity1; allOnAll.similarity2];
comm = [allOnAll.commFitDiff; -allOnAll.commFitDiff];
ind = [allOnAll.indFitDiff; -allOnAll.indFitDiff];
commSc = comm/max(abs(comm));
indSc = ind/max(abs(ind));

%
%%% PANEL D %%%
binBdry = linspace(-1,1,p.binN);
binCtr = binBdry(1:end-1)+diff(binBdry)/2;
binBdry(end)=Inf;
[~,commInd] = histc(commSc, binBdry);
binMeanC = accumarray(commInd,sim,[p.binN-1, 1],@mean);
binStdC = accumarray(commInd,sim,[p.binN-1, 1],@std);
upperC = (binMeanC+binStdC)';
lowerC = (binMeanC-binStdC)';

[~,indInd] = histc(indSc, binBdry);
binMeanI = accumarray(indInd,sim,[p.binN-1, 1],@mean);
binStdI = accumarray(indInd,sim,[p.binN-1, 1],@std);
upperI = (binMeanI+binStdI)';
lowerI = (binMeanI-binStdI)';

axes('Units','Centimeters',...
    'Position',[p.offAx+p.wA+p.offBx+p.wB+p.offDx, p.offDy, p.wD, p.hD]);
text(p.labelDOffX,p.wA+p.labelOffY-p.offDy+p.offAy,'D', 'Units', 'Centimeters', 'FontSize', params.labelFontSize);



hold on
h1 = plot(binCtr,binMeanC,'r-','LineWidth',2);
patch([binCtr, binCtr(end:-1:1)],[upperC, lowerC(end:-1:1)],'r','FaceAlpha',0.5,'EdgeColor','none');

h2 = plot(binCtr,binMeanI,'k-','LineWidth',2);
patch([binCtr, binCtr(end:-1:1)],[upperI, lowerI(end:-1:1)],'k','FaceAlpha',0.5,'EdgeColor','none');

axis([-1 1 0 1.05])
set(gca,'YTick',[0 1]);
xlabel('Fitness difference of \it{C}\rm_1 and \it{C}\rm_2')

text(-0.4,p.hD/2,{'Similarity of \it{C}\rm_1 and \it{C}'}, 'Units', 'Centimeters', ...
        'FontSize', params.axesFontSize,'HorizontalAlignment','center','Rotation',90);
adjustSizes(gca,1,params.axesFontSize);
title({'Predicting the outcome','of community coalescence'}, 'FontSize', params.titleFontSize);

hLeg = legend([h1 h2],{'community','individual'},'Location','South');
hLeg.Units = 'centimeters';
legend('boxoff');
hLeg.Position(1) = hLeg.Position(1)+1;

text('String','Fitness measure:', 'Units', 'Centimeters', 'Position',[3.9 1.65], ...
        'FontSize', params.axesFontSize,'HorizontalAlignment','center');

cartoon = imread('Cartoon_Fig4_H.bmp','bmp');

axParent = get(gca,'Position');
axes('Units','Centimeters',...
    'Position',[axParent(1)+p.insetX, axParent(2)+p.insetY, p.insetH/size(cartoon,1)*size(cartoon,2), p.insetH]);
imshow(cartoon)
axis image

%%
if params.saveFigs
    fname = sprintf('Figure4%s', '');
    saveas(gcf,[fname '.png'])
    saveas(gcf,[fname '.pdf'])
    saveas(gcf,[fname '.fig'])
else
    pause;
end
end


function generate_Fig5(params, data)
subsets = data.otherEpsilon.subsets;
p = params.Fig5;
%%
% Assign quadrants
x = subsets(:,end-1);
y = subsets(:,end);
xRank = getRank(x);
yRank = getRank(y);

quadrantN = params.otherEpsilon.competition.quadrantN;
xLim = linspace(min(xRank),max(xRank),quadrantN);
yLim = linspace(min(yRank),max(yRank),quadrantN);
q1 = xRank<xLim(2) & yRank<yLim(2);
q2 = xRank<xLim(2) & yRank>yLim(quadrantN-2);
q3 = xRank>xLim(quadrantN-2) & yRank>yLim(quadrantN-2);
q4 = xRank>xLim(quadrantN-2) & yRank<yLim(2);
noQ = ~(q1|q2|q3|q4);

d.x = x;
d.y = y;
d.q = [q1(:), q2(:), q3(:), q4(:)]';

%
clf;
set(gcf, 'PaperPositionMode','Manual', 'PaperUnits','Centimeters',...
    'PaperSize', [p.W p.H], 'PaperPosition',[0 0 p.W p.H],...
    'Units','Centimeters','Position',[0 0 p.W p.H]); 

axes('Units','Centimeters',...
    'Position',[p.offAx, p.offAy, p.wA, p.wA]);
hold on;
x = d.x/params.otherEpsilon.epsilon;
y = d.y/params.otherEpsilon.epsilon;
plot(x(noQ), y(noQ),'k.')
for i=1:4
    plot(x(d.q(i,:)), y(d.q(i,:)), '.','Color', p.quadrantColors(i,:))
end
xlabel('\langle f \rangle / \epsilon (mean individual performance)');    
ylabel('F/\epsilon (community fitness)');
axis([-2 2 -150 0]);
axis square
axis auto
adjustSizes(gca,1,12);

line1x = [min(x(d.q(1,:))), max(x(d.q(1,:))), max(x(d.q(1,:)))];
line1y = [min(y(d.q(3,:))), min(y(d.q(3,:))), max(y(d.q(3,:)))];
line2x = [min(x(d.q(3,:))), min(x(d.q(3,:))), max(x(d.q(3,:)))];
line2y = [min(y(d.q(1,:))), max(y(d.q(1,:))), max(y(d.q(1,:)))];
line(line1x,line1y,'LineWidth',2,'Color',p.quadrantColors(2,:));
line(line2x,line2y,'LineWidth',2,'Color',p.quadrantColors(4,:));

cx = p.quadrantLabelsCoordX;
cy = p.quadrantLabelsCoordY;
text(cx(1),cy(1),'I', 'FontSize', p.quadLabelFontSize,'FontName','Times','HorizontalAlignment','center');
text(cx(1),cy(2),'II', 'FontSize', p.quadLabelFontSize,'FontName','Times','HorizontalAlignment','center');
text(cx(2),cy(2),'III', 'FontSize', p.quadLabelFontSize,'FontName','Times','HorizontalAlignment','center');
text(cx(2),cy(1),'IV', 'FontSize', p.quadLabelFontSize,'FontName','Times','HorizontalAlignment','center');
%%
if params.saveFigs
    fname = sprintf('Figure5%s', '');
    saveas(gcf,[fname '.png'])
    saveas(gcf,[fname '.pdf'])
    saveas(gcf,[fname '.fig'])
else
    pause;
end
end


function data = generateData(params, override)
% We need: 
%   1) Sweep through epsilons, and equilibrate a lot of random subsets for
%      each - this goes into Fig 2B
%   2) For one particular epsilon and one particular cost realization:
%          *) Find true final equilibrium (Fig 1B, Fig 2A)
%          *) Get full dynamical trajectories for a bunch of random initial
%             conditions (Fig 3)
%          *) Generate random k-subsets and compete them with each other (Fig 4)

    % Sweep through epsilon range
    fprintf('Equilibrating random subsets from communities at a range of epsilon...\n');
    data.epsilonSweep = getEpsilonSweepData(params.epsilonSweep,override);
    fprintf('Done.\n');

    
    % Work with fixed epsilon
    fprintf('Generating the unique cost structure for all subsequent calculations...\n');
    data.fixedEpsilon.prop = generateCostStructure(params.fixedEpsilon);
    fprintf('Done.\n');

    fprintf('Finding the global community optimum.\n');
    data.fixedEpsilon.trueEquilibrium = getTrueEquilibrium(data.fixedEpsilon.prop);
    fprintf('Done.\n');

    fprintf('Calculating dynamical trajectories from random initial conditions...\n')
    data.fixedEpsilon.convergenceDynamics = getConvergenceDynamics(params.fixedEpsilon, data.fixedEpsilon.prop, override);
    fprintf('Done.\n');

    fprintf('Generating sub-communities for competition assays...\n')
    data.fixedEpsilon.subsets = generateSubsets(params.fixedEpsilon, data.fixedEpsilon.prop, override);
    fprintf('Done.\n');

    fprintf('Running competition assays...\n');
    data.fixedEpsilon.subsetCompetition = ...
        runSubsetCompetition(params.fixedEpsilon, data.fixedEpsilon, override);
    fprintf('Done.\n');
    
    data.fixedEpsilon.allOnAllCompetition = processAllOnAllCompetition(params,data);
        
    % Figure 5
    fprintf('Repeating part of that using a differnt epsilon (Fig. 5).');
    fprintf('Generating the other cost structure...\n');
    data.otherEpsilon.prop = generateCostStructure(params.otherEpsilon);
    fprintf('Done.\n');

    fprintf('Generating sub-communities for competition assays...\n')
    data.otherEpsilon.subsets = generateSubsets(params.otherEpsilon, data.otherEpsilon.prop, override);
    fprintf('Done.\n');
end

function subsetCompetition = runSubsetCompetition(p, data, override)
    K = p.subsets.chooseOutOf;
    k = p.subsets.chooseK;
    fname = sprintf('ksubsets_3pools_eps=%g_seed=%d_choose_%d_out_of_random_%d_choiceSeed=%d_quadN=%d_pairsN=%d.mat',...
        p.epsilon, p.seed, k, K, p.subsets.choiceSeed, p.competition.quadrantN,p.competition.pairsN);

    if exist(fname,'file') && ~override
        subsetCompetition = load(fname);
        subsetCompetition = subsetCompetition.subsetCompetition;
        fprintf('\tLoaded from file.\n');
    else
        % Define competition pools
        subsets = data.subsets;

        x = subsets(:,end-1);
        y = subsets(:,end);
        xRank = getRank(x);
        yRank = getRank(y);

        quadrantN = p.competition.quadrantN;
        xLim = linspace(min(xRank),max(xRank),quadrantN);
        yLim = linspace(min(yRank),max(yRank),quadrantN);
        q1 = xRank<xLim(2) & yRank<yLim(2);
        q2 = xRank<xLim(2) & yRank>yLim(quadrantN-2);
        q3 = xRank>xLim(quadrantN-2) & yRank>yLim(quadrantN-2);
        q4 = xRank>xLim(quadrantN-2) & yRank<yLim(2);
        allQ = true(size(q1));
        
        s.x = x;
        s.y = y;
        s.q = [q1(:), q2(:), q3(:), q4(:)]';

        % Compete communities against each other
        fprintf('Two pools of interest against each other...\n');
        s.competition24 = competePools(q2,q4, data, p.competition.pairsN);
        fprintf('Two control pools against each other...\n');
        s.competition31 = competePools(q3,q1, data, p.competition.pairsN);
        fprintf('Fully random pairs...\n');
        s.competitionAll = competePools(allQ,allQ, data, p.competition.pairsN_AllOnAll);

        subsetCompetition = s;
        save(fname,'subsetCompetition')
    end
end

function competitionResults = competePools(q1,q2, data, tryN)    
    competitionPool1 = find(q1);
    competitionPool2 = find(q2);
    if isempty(competitionPool1) || isempty(competitionPool2)
        warning('Empty competition pool. Adjust parameters.');
        competitionResults = [];
        return;
    end
    % choose competitors from pools
    rng(0);
    i1 = competitionPool1(randi(length(competitionPool1),[tryN,1]));
    i2 = competitionPool2(randi(length(competitionPool2),[tryN,1]));

    outcome(tryN) = compete(data.subsets, [i1(end), i2(end)], data.prop);
    for p=1:tryN-1
        if mod(p,10)==0
            fprintf('\t%d/%d...\n',p,tryN);
        end
        outcome(p) = compete(data.subsets, [i1(p), i2(p)], data.prop);
    end
    fprintf('\t%d/%d.\n',tryN,tryN);    
    
    summaryTable = NaN(size(data.subsets,2)-2,tryN);
    for p=1:tryN
        from1 = outcome(p).group(1).abd>0;
        from2 = outcome(p).group(2).abd>0;
        survive = outcome(p).merge.abd>0.001;
        didCompete = from1 | from2;
        fitness = data.prop.cpe(outcome(p).types);
        % If there were fewer than 8 competitors, the ones that were absent
        % from the start should not be counted as types that went extinct.
        % For such competitions there are fewer than 8 rows, but the matrix
        % we will be displaying has 8 rows nonetheless. So the color values
        % we get here should be vertically centered in that table. To
        % achieve this, the easy hack is to set the fitness of the first
        % absent species to Inf (then it's always at the top, and always
        % black). And the second absetn species, if there is one, should
        % get fitness -Inf, so it's always at the bottom (and also black)
        %
        % These adjustments don't actually have any perceptible effect
        % visually, it's just out of perfectionism...
        
        dead = find(~didCompete);
        if length(dead)>=1
            fitness(dead(1))=Inf;
        end
        if length(dead)>=2
            fitness(dead(2))=-Inf;
        end
        [~,ord] = sort(fitness);
        result = zeros(size(survive));
        % we should assign a color to species that go extinct - this means
        % they had to be present to begin with! 
        color = didCompete & ~survive; 
        % Color types by provenance
        result(from1 & color) = 1;
        result(from2 & color) = 2;
        result(from1 & from2 & color) = 3;
        result(~color) = 4;

        summaryTable(1:length(result),p) = result(ord);
    end
    summaryTable(isnan(summaryTable)) = 5;
    [~, ord] = sort(sum(summaryTable==1)-sum(summaryTable==2));
    summaryTable = summaryTable(:,ord);
    
    competitionResults.outcome = outcome;
    competitionResults.summaryTable = summaryTable;
end

function subsets = generateSubsets(p, prop, override)
K = p.subsets.chooseOutOf;
k = p.subsets.chooseK;
fname = sprintf('ksubsets_eps=%g_seed=%d_choose_%d_out_of_random_%d_choiceSeed=%d.mat',...
    p.epsilon, p.seed, k, K, p.subsets.choiceSeed);

if exist(fname,'file') && ~override
    subsets = load(fname);
    subsets = subsets.subsets;
    fprintf('\tLoaded from file.\n')
else
    % Construct all k-species subsets
    rng(p.subsets.choiceSeed);
    Ksubset = randsample(length(prop.cost),K);

    C = nchoosek(Ksubset, k);

    % C is the table listing community membership. Complement it with:
    %       k new columns listing abundances of the types
    %       2 new columns listing individual + community fitness values
    % data:
    % abd1...abdk  indFitness commFitness
    data = NaN(size(C,1),k+2);    
    for i=1:size(C,1)
        if mod(i,10)==0
            fprintf('\t%d/%d...\n',i,size(C,1));
        end
        % First check if this organism group together has all pathways
        % If not, skip this group
        if all(sum(prop.enzymesInSpecies(C(i,:),:)))
            abd = zeros(size(prop.cost));
            abd(C(i,:)) = 1;
            abd = equilibrateCommunity(abd, prop, p.subsets.dThresh);
            selectAbd = abd(C(i,:));
            if any(selectAbd<p.subsets.abdPresenceThresh)
                selectAbd(selectAbd<p.subsets.abdPresenceThresh)=0;
                % this is not a "true" k-species community
                % (at least one of the types went extinct)
            end
            data(i,1:k) = selectAbd;
            [data(i,end), data(i,end-1)] = getCommunityFitness(abd,prop);
        end % else: this row in "data" stays NaN
    end
    fprintf('%d/%d...\n',size(C,1),size(C,1));
    accept = ~isnan(data(:,1));
    fprintf('%d accepted combinations out of %d (%.1f%%)\n',sum(accept),length(accept),100*sum(accept)/length(accept));
    subsets = [C(accept,:) data(accept, :)];
    save(fname,'subsets');
end
end

function convergenceDynamics = getConvergenceDynamics(p, prop, override)
fname = sprintf('convergenceDynamics_eps=%g_seed=%d_tries=%d+%d_choose_%d_of_%d.mat',...
    p.epsilon,p.seed,p.dynamics.seedOffset,p.dynamics.tries,p.dynamics.selectK,p.dynamics.selectOutOf);
if exist(fname,'file') && ~override   
    convergenceDynamics = load(fname);
    convergenceDynamics = convergenceDynamics.convergenceDynamics;
    fprintf('\tLoaded from file.\n')
else
    info = cell(1, p.dynamics.replicates);
    for t=1:p.dynamics.replicates
        fprintf('\t%d/%d...\n',t,p.dynamics.replicates);
        rng(t+p.dynamics.seedOffset);
        abd = 10.^(random('Uniform', -5,2,size(prop.cost)));
        info{t} = fullDynamicalSimulation(abd, prop, p.dynamics.maxT);
    end
    convergenceDynamics = info;
    save(fname,'convergenceDynamics');
end
end

function trueEquilibrium = getTrueEquilibrium(prop)
    abd0 = findEquilibriumSmart(prop,[],10);
    [comm0, ind0] = getCommunityFitness(abd0,prop);
    trueEquilibrium.abd = abd0;
    trueEquilibrium.communityFitness = comm0;
    trueEquilibrium.individualFitness = ind0;
end

function prop = generateCostStructure(p)
    prop = init(p.N);
    prop.offset = p.offset*ones(size(prop.offset));
    rng(p.seed);
    prop.cpe = 1 + p.epsilon*random('Normal',0,1,size(prop.cost));
    prop.enzCount = sum(prop.enzymesInSpecies,2);
    prop.cost = prop.cpe .* prop.enzCount;
end

function d = getEpsilonSweepData(p,override)
    epsList = p.epsilonRange;
    prop = init(p.N);
    enzCount = sum(prop.enzymesInSpecies,2);
    rankTable_sweep = NaN(p.N, length(epsList),p.seeds*p.tries);
    abdTable_sweep = NaN(p.N, length(epsList),p.seeds*p.tries);
    redFlag_sweep = false(length(epsList),p.seeds*p.tries);

    for epsInd = 1:length(epsList)
        fprintf('%d/%d...\n',epsInd, length(epsList));
        epsilon = epsList(epsInd);
        % For a fixed epsilon:
        fname = sprintf('epsSweepData_eps=%g_seeds=%d_tries=%d.mat',epsilon,p.seeds, p.tries);
        if exist(fname,'file') && ~override
            fprintf('\tLoaded from file.\n');
            dSweep = load(fname);
            rankTable = dSweep.rankTable;
            redFlag = dSweep.redFlag;
            abdTable = dSweep.abdTable;
        else
            % calculate & save
            rankTable = NaN(p.N, p.seeds*p.tries);
            abdTable = NaN(p.N, p.seeds*p.tries);
            redFlag = false(1,p.seeds*p.tries);
            lastEntryNo = 0;
            for seed = 1:p.seeds
                fprintf('\tSeed %d/%d...\n',seed, p.seeds);
                rng(seed);
                cpe = 1 + epsilon*random('Normal',0,1,size(prop.cost));
                prop.cost = cpe .* enzCount;
                % Defined the universe; now select and equilibrate subsets
                for t=1:p.tries
                    lastEntryNo = lastEntryNo + 1;
                    rng(t);
                    abd = zeros(size(prop.cost));
                    subset = randsample(length(prop.cost),p.K);
                    cpeRank = getRank(cpe(subset));
                    abd(subset)=1;
                    [abd, warn] = equilibrateCommunity(abd,prop,p.dThresh*epsilon);
                    % ensure there are no more than N non-zero
                    abdSrt = sort(abd,'descend');
                    maxAbdAfterN = max(abd(abd<=abdSrt(p.N+1)));
                    if maxAbdAfterN>p.abdPresenceThresh || warn
                        % did not equilibrate properly
                        redFlag(lastEntryNo)=true;
                    end
                        
                    abd(abd<=abdSrt(p.N+1)) = 0;
                    
                    % logical array of length = length(subset)
                    abdSubset = abd(subset);
                    survivors = abdSubset>p.abdPresenceThresh;
                    rankOfSurvivors = cpeRank(survivors);
                    abdOfSurvivors = abdSubset(survivors);
                    rankTable(1:length(rankOfSurvivors), lastEntryNo)=rankOfSurvivors;
                    abdTable(1:length(rankOfSurvivors), lastEntryNo)=abdOfSurvivors;
                end
            end
            fprintf('Failure rate: %d%%\n', 100*sum(redFlag)/length(redFlag));
            save(fname,'rankTable','redFlag','abdTable');
        end
        fprintf('Done.\n');
        rankTable_sweep(:,epsInd,:) = rankTable;
        abdTable_sweep(:,epsInd,:) = abdTable;
        redFlag_sweep(epsInd,:) = redFlag;
    end
    d.rankTable = rankTable_sweep;
    d.abdTable = abdTable_sweep;
    d.redFlag = redFlag_sweep;
end

function generate_Fig2(params, data)
dSweep = data.epsilonSweep;
d = data.fixedEpsilon;
p = params.Fig2;
%%
clf;

set(gcf, 'PaperPositionMode','Manual', 'PaperUnits','Centimeters',...
    'PaperSize', [p.W p.H], 'PaperPosition',[0 0 p.W p.H],...
    'Units','Centimeters','Position',[0 0 p.W p.H]); 

%%% PANEL A %%%

axes('Units','Centimeters',...
    'Position',[p.offAx, p.offAy, p.wA, p.hA]);
fitnessRank = getRank(d.prop.cpe);
sel = d.trueEquilibrium.abd>0.1;
abdVec = d.trueEquilibrium.abd(sel);
[~,ord] = sort(abdVec,'ascend');
% invert first two for aesthetic reasons (so labels don't overlap)
ord([end-1, end]) = ord([end, end-1]);
sel = find(sel); sel = sel(ord);
barh(d.trueEquilibrium.abd(sel),'k')

YTickLabel = cellstr(d.prop.speciesStr(sel,:));
set(gca,'XTick',0:10:50,'YTick',1:length(sel),'YTickLabel',YTickLabel,'XLim',p.AAxisRangeX)

for i=1:length(sel)
    text(d.trueEquilibrium.abd(sel(i))+1,i,sprintf('#%d',fitnessRank(sel(i))),...
        'HorizontalAlignment','left','Color','k','FontSize',params.axesFontSize)
end
xlabel('Abundance');
text(p.labelAOffX,p.hA+p.labelOffY,'A', 'Units', 'Centimeters', 'FontSize', params.labelFontSize);
adjustSizes(gca,1,params.axesFontSize);

%%% Panel B %%%
axes('Units','Centimeters',...
    'Position',[p.offAx+p.wA+p.offBx, p.offBy, p.whB, p.whB]);
%
rnkTbl = dSweep.rankTable;
abdTbl = dSweep.abdTable;

repRedFlag = repmat(shiftdim(dSweep.redFlag,-1),[size(rnkTbl,1),1,1]);
rnkTbl(repRedFlag)=NaN;
abdTbl(repRedFlag)=NaN;
count = squeeze(sum(~isnan(rnkTbl)))';
count(count==0)=NaN; % those were red-flagged
med = squeeze(nanmedian(rnkTbl))'; % here the red-flagged entries are NaN already
% weighted median
wMed = NaN(size(med));
% abdTable_sweep = NaN(p.N, length(epsList),p.seeds*p.tries);
for eps=1:size(abdTbl,2)
    for rep=1:size(abdTbl,3)
        if ~dSweep.redFlag(eps,rep)
            rnk = rnkTbl(:,eps,rep);
            abd = abdTbl(:,eps,rep);
            sel = ~isnan(rnk);
            wMed(rep, eps) = weightedMedian(rnk(sel),abd(sel));
        end
    end
end

hold off
box on
hold all;
h1 = semilogx(params.epsilonSweep.epsilonRange, nanmean(med),'r-','LineWidth',1.5); %#ok<*UDIM>
fprintf('Standard deviation of the median rank estimate: %.2f\n',nanstd(med(1:round(end/2))));
h11 = semilogx(params.epsilonSweep.epsilonRange, nanmean(wMed),'b--','LineWidth',1.5);
fprintf('Standard deviation of the wMed   rank estimate: %.2f\n',nanstd(wMed(1:round(end/2))));
axis([7e-6 0.15 0 10.9]);
axis square
set(gca,'XScale','log', 'XTick',10.^(-6:-1))
text(p.labelBOffX,-p.offBy + p.offAy + p.hA+p.labelOffY,'B', 'Units', 'Centimeters', 'FontSize', params.labelFontSize);
h = legend([h1 h11], {'Median perform. rank','Weighted median rank'},'Location','South', 'FontSize', params.axesFontSize-1);
legend boxoff                                     
h.Position = [0.63 0.23 0.34 0.16];
xlabel('Cost scatter \epsilon')
adjustSizes(gca,1,params.axesFontSize);

%%
if params.saveFigs
    fname = sprintf('Figure2%s', '');
    saveas(gcf,[fname '.png'])
    saveas(gcf,[fname '.pdf'])
    saveas(gcf,[fname '.fig'])
else
    pause;
end
end



function summary = processAllOnAllCompetition(params,data)
%%
% Last two columns of data.fixedEpsilon.subsets: community and mean
% individual fitness of a given subset 
% also fields x and y of data.fixedEpsilon.subsetCompetition
% 
% Determine the competing subcommunities by regenerating their
% pseudorandom indices from the same seed (=0):
s = data.fixedEpsilon.subsetCompetition;
tryN = params.fixedEpsilon.competition.pairsN_AllOnAll;
rng(0);
i1 = randi(length(s.x),[tryN,1]);
i2 = randi(length(s.x),[tryN,1]);

% verify:
for i=1:length(i1)
    types1 = union(data.fixedEpsilon.subsets(i1(i),1:4), data.fixedEpsilon.subsets(i2(i),1:4));
    types2 = s.competitionAll.outcome(i).types;
    assert(all(types1==types2),'Internal inconsistency.');
end
fprintf('No mistakes.\n');
%%
similarityToFitterParent = zeros(size(i2));
similarityToOtherParent = zeros(size(i2));
for i=1:length(i2)
    outcome = s.competitionAll.outcome(i);
    abd1 = outcome.group(1).abd;
    abd2 = outcome.group(2).abd;
    abdM = outcome.merge.abd;
    similarityToFitterParent(i) = abd1 * abdM'/(norm(abd1)*norm(abdM));    
    similarityToOtherParent(i) = abd2 * abdM'/(norm(abd2)*norm(abdM));    
end

% compete.m always places more fit community (in community fitness sense)
% first.
% To match "parent 1" and "parent 2" to "fitterParent" and "otherParent",
% check the sign of the community fitness difference between 1 and 2.
summary.commFitDiff = abs(data.fixedEpsilon.subsetCompetition.y(i1)-data.fixedEpsilon.subsetCompetition.y(i2));
summary.indFitDiff = data.fixedEpsilon.subsetCompetition.x(i1)-data.fixedEpsilon.subsetCompetition.x(i2);
firstIsFitter = summary.commFitDiff>0;
similarity1 = zeros([tryN,1]);
similarity2 = zeros([tryN,1]);
similarity1(firstIsFitter) = similarityToFitterParent(firstIsFitter);
similarity2(firstIsFitter) = similarityToOtherParent(firstIsFitter);
similarity1(~firstIsFitter) = similarityToOtherParent(~firstIsFitter);
similarity2(~firstIsFitter) = similarityToFitterParent(~firstIsFitter);

summary.similarity1 = similarity1;
summary.similarity2 = similarity2;
end
