function [abd0, pool, warnings] = findEquilibriumSmart(prop, abd0, pool, verbatim)
% Same as findEquilibriumFast, but use the built-in ODE solvers which are
% much more efficient.
% Input: 
%   abd0 = best guess of final equilibrium, if available, otherwise [].
%   pool = boolean array of types that are likely to be found.
%          OR a single number K, then K lowest-cost types are included
% Output:
%   abd0 = the "true" equilibrium found.
%   pool = the input pool, updated to include all species in abd0

SPECIES_NUMBER_LIMIT = 300;

minAbundance = 1e-3; 
dThresh = 1e-5;
dThreshStringent = 1e-7;

if nargin<4
    verbatim = false;
end

pool5 = verifyPool(5, prop); % include 5 lowest-cost anyway; can speed up calcualtions in high-noise regimes
pool = verifyPool(pool, prop) | pool5;

if verbatim
    fprintf('\tEvolving a total of %d species.\n',sum(pool));
end

% Starting point: abd0 + minAbundance of each of the species in the pool
if isempty(abd0)
    abd0 = zeros(size(pool));
end
abd0 = abd0 + minAbundance * double(pool);

invadersExist = true;
p=0;
while invadersExist
    abd0 = equilibrateCommunity(abd0, prop, dThresh);
    if verbatim
        fprintf('.');
    end
    invaders = findInvaders(abd0,prop,dThresh);
    if ~any(invaders) && (sum(abd0>0)>prop.N)
        % This might be the true equilibirum, but we need to make sure.
        % For this, zero out all types leaving at most N
        fprintf('Putative equilibrium found...\n');
        abd0 = equilibrateCommunity(abd0, prop, dThreshStringent);
        %dThreshStringent = dThreshStringent/2;

        abdSrt = sort(abd0,'descend');
        thresh = max([abdSrt(prop.N+1), 0]);
        abd0(abd0<=thresh)=0; % now at most N entries in abd0 are non-zero and positive
        abd0 = equilibrateCommunity(abd0, prop, dThresh);
        invaders = findInvaders(abd0,prop,dThresh);
    end
        
    if ~any(invaders)
        % Ok, now we're sure.
        fprintf('The simulated subset is at non-invadeable equilibrium.\n');
        invadersExist = false;
    else
        pool = abd0>0 | invaders;
        if sum(pool)>SPECIES_NUMBER_LIMIT
            % revise the pool
            fprintf('Refusing to simulate %d types; pruning.\n',sum(pool));
            invaders = find(invaders);
            pool = false(size(pool));
            pool(abd0>minAbundance) = true;
            abd0(abd0<=minAbundance) = 0;
            remain = SPECIES_NUMBER_LIMIT - sum(pool);
            if length(invaders)>remain
                invaders = randsample(invaders, remain);
            end
            pool(invaders) = true;
        end
        abd0 = abd0 + minAbundance * double(pool);
    end
    p=p+1;
    if p==30
        fprintf('Infinite loop? Aborting equilibration.\n');
        break;
    end
end

if sum(abd0>0)>prop.N
    warning('"Equilibrium state" contains more than N types!');
    warnings = true;
else
    warnings = false;
end
if verbatim
    fprintf('\n\tFound final state with %d species:\n',sum(abd0>0));
    disp(prop.speciesStr(abd0>0,:));
end
pool = pool | abd0>0;

end

function pool = verifyPool(pool, prop)
if length(pool)==1
    NN = pool;
    % Construct pool as the union, over all enzymes, of the NN lowest-cost 
    % types containing them 
    NN = min(NN, length(prop.cost));
    enzCount = sum(prop.enzymesInSpecies,2);
    cpe = prop.cost./enzCount;
    [~, ind] = sort(cpe);
    pool = false(size(prop.cost));
    for e=1:size(prop.enzymesInSpecies,2)
        pool(find(prop.enzymesInSpecies(ind,e),NN))=true;
    end
    includeInd = ind(pool);
    pool = false(size(prop.cost));
    pool(includeInd) = true;    
elseif islogical(pool)
    % do nothing
end
end