#!/usr/bin/env python
# encoding: utf-8
"""
example_analysis.py

This script is provided as Supplementary Online Material
to the paper:
Eco-HAB - FULLY AUTOMATED AND ECOLOGICALLY RELEVANT ASSESSMENT 
OF SOCIAL IMPAIRMENTS IN MOUSE MODELS OF AUTISM

Copyright 2015 Szymon Łęski

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
"""


import EcoHab
import numpy as np
import matplotlib.pyplot as plt
from matplotlib.dates import epoch2num
import locale
from ExperimentConfigFile import ExperimentConfigFile


datasets = ['BALB VPA data cohort 1']
smells = {'BALB VPA data cohort 1': {'soc': 3, 'nsoc': 1}}
# social smell between antennas 6&7; non-social between 2&3
            
if __name__ == '__main__':
    # Analysis in 12, 2, and 1-hour bins
    binsizes = [12 * 3600., 2 * 3600., 1 * 3600.]
    bintitles = ['12', '2', '1']

    for path in datasets:
        # Load raw data
        ehd = EcoHab.EcoHabData(path)
        # Convert raw data to 'sessions'
        ehs = EcoHab.EcoHabSessions(ehd)
        
        # Load experimental phases
        cf = ExperimentConfigFile(path)
        tstart, tend = cf.gettime('ALL')
    
        def time2title(tt):
            return cf(epoch2num(tt + 300.))

        # A loop over consecutive time bins
        for binsize, btit in zip(binsizes, bintitles):
            tt = tstart
            tts = []
            results_soc = dict([(mm, []) for mm in ehd.mice])
            results_nsoc = dict([(mm, []) for mm in ehd.mice])
            results_soc_t = dict([(mm, []) for mm in ehd.mice])
            results_nsoc_t = dict([(mm, []) for mm in ehd.mice])
    
            while tt < tend:
                ehs.unmask_data()
                # Mask the data to the current time bin
                ehs.mask_data(tt, tt + binsize)
                # Loop over mice
                for mm in ehd.mice:
                    # Retrieve compartment numbers and durations of sessions
                    adds = ehs.getaddresses(mm)
                    durs = ehs.getdurations(mm)
                    # Calculate the results
                    results_nsoc[mm].append(adds.count(smells[path]['nsoc']))
                    results_soc[mm].append(adds.count(smells[path]['soc']))
                    results_nsoc_t[mm].append(sum([d for d, ad in zip(durs, adds)
                                  if ad == smells[path]['nsoc']]))
                    results_soc_t[mm].append(sum([d for d, ad in zip(durs, adds)
                                  if ad == smells[path]['soc']]))
                tts.append(tt)
                tt += binsize
    
            # Save results to a csv file
            f = open('res_%s_%s.csv' %(path, btit), 'w')
            header = '\t'.join(['Mouse',] + map(time2title, tts))

            f.write('Total time with social smell (box %d), seconds\n' %smells[path]['soc'])
            f.write(header + '\n')
            for mm in ehd.mice: 
                f.write('%s' %(mm))
                for idx in range(len(tts)):
                    try:                
                        f.write('\t%s' %locale.format("%f", results_soc_t[mm][idx]))
                    except KeyError:
                        f.write('\t')
                f.write('\n')
            f.write('\n\n') 

            f.write('Total time with non-social smell (box %d), seconds\n' %smells[path]['nsoc'])
            f.write(header + '\n')
            for mm in ehd.mice: 
                f.write('%s' %(mm))
                for idx in range(len(tts)):
                    try:                
                        f.write('\t%s' %locale.format("%f", results_nsoc_t[mm][idx]))
                    except KeyError:
                        f.write('\t')
                f.write('\n')
            f.write('\n\n') 

            f.write('Number of visits to social smell (box %d)\n' %smells[path]['soc'])
            f.write(header + '\n')
            for mm in ehd.mice: 
                f.write('%s' %(mm))
                for idx in range(len(tts)):
                    try:                
                        f.write('\t%s' %locale.format("%d", results_soc[mm][idx]))
                    except KeyError:
                        f.write('\t')
                f.write('\n')
            f.write('\n\n') 

            f.write('Number of visits to non-social smell (box %d)\n' %smells[path]['nsoc'])
            f.write(header + '\n')
            for mm in ehd.mice: 
                f.write('%s' %(mm))
                for idx in range(len(tts)):
                    try:                
                        f.write('\t%s' %locale.format("%d", results_nsoc[mm][idx]))
                    except KeyError:
                        f.write('\t')
                f.write('\n')
            f.write('\n\n') 
            f.close()