function [ai,chan,dactype,trigger_value,trigger_condition] = ttl_in(varargin)
%This little function will recognize a ttl pulse input through a DAC object
%and use this data to control the flow of the program.
%Synatax: output = ttl_in('dac',[],'dactype','','num',[],'trigger_value',[],'trigger_condtion','')
%Input: dac = if this parameter is set, the function will not open its
%       own dac object, nor will it close the dac object upon exit, this
%       is designed so that if a function is to use the same dac object
%       multiple types, then we will not have to open the object multiple
%       times.
%       chan = the channel handle.
%       dactype = the type of DAC object.  Default = winsound
%       num = the number of TTL pulses before the function returns
%       positive.  Default = 1
%       trigger_value = the value on which the trigger activates. NOTE:
%       No Default value is set for this variable, you must enter a trigger
%       value
%       trigger_condtion = how the trigger is activated.  Possible
%       Conditions are: rising, falling, leaving & entering.  Default =
%       Leaving.
%Output: All the TTL parameters.


%Parse the input parse it.
[ai,chan,dactype,num,trigger_value,trigger_condition,close] = parse(varargin);

%Set defaults and initialize values
output = 0;

%Open a DAC object if one is not open already
if isempty(ai)
    ai = analoginput(dactype);       %Open DAC
    chan = addchannel(ai,1);            %Open a Channel
    maxrate = daqhwinfo(ai,{'MaxSampleRate'});   %Get max sample rate
    set(ai,'SampleRate',maxrate{1,1});        %Set the DAC to sampel at the max
    set(ai,'SamplesPerTrigger',1);       %In the case of a ttl, the data is irrelavent
end

%Set Trigger properties
set(ai,'TriggerType','Software');
set(ai,'TriggerChannel',chan)
set(ai,'TriggerCondition',trigger_condition);
set(ai,'TriggerConditionValue',trigger_value);
%set(ai,'TriggerFcn','brk = 1');       %The callback function here is just a break out function.
%set(ai,'TriggerRepeat',num-1);        %The sets specific number of times the trigger can be executed.  The allows one Start call to operate through the function.

%Start the DAC engine
start(ai)
%Wait for TTL
for i = 1:num
    while 1
        if get(ai,'SamplesAvailable')~=0
            flushdata(ai)
            %getdata(ai);
            break
        end
    end
end

%Clean up - optional (default does not close)
if close == 1
    delete(ai)
    clear ai
end


%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [ai,chan,dactype,num,trigger_value,trigger_condition,close] = parse(input)
%set defaults
ai = [];
chan = [];
dactype = 'winsound';
num = 1;
trigger_value = [-2.8 2.8];
trigger_condition = 'Leaving';
close = 0;

%Parse the input
if ~isempty(input)
    a = size(input,2);
    for i = 1:2:a
        value = input{1,i+1};
        switch input{1,i}
            case 'dac'
                if isa(value,'analoginput')
                    ai = value;
                else
                    error(['Error: You must enter the object handle of a DAC for dac']);
                end
            case 'chan'
                if isa(value,'aichannel')
                    chan = value;
                else
                    error(['Error: You must enter the object handle of a DAC for dac']);
                end

            case 'dactype'
                if ischar(value)
                    dactype = value;
                else
                    error(['Error: You must enter a string for property dactype']);
                end
            case 'num'
                if isnumeric(value)
                    num = value;
                else
                    error(['Error: You must enter a number for the property num']);
                    
                end
            case 'trigger_value'
                if isnumeric(value)
                    trigger_value = value+value.*0.05;
                else
                    error(['Error: You must enter a number for the property baseline']);
                    
                end
            case 'trigger_condition'
                if ischar(value)
                    trigger_condition = value;
                else
                    error(['Error: You must enter a string for property trigger_condition']);
                end
            case 'close'
                if isnumeric(value)
                    close = value;
                else
                    error(['Error: You must enter either 1 or 0 for the close value']);
                end
            otherwise
                error(['Error: You entered jibberish.']);
        end
    end
end

%Get the right trigger values
switch lower(trigger_condition)
    case 'rising'
        trigger_value = max(trigger_value);
    case 'falling'
        trigger_value = min(trigger_value);
end
