function s_AlignGCs_v7(si,Ch,SIv)
% _v7 calculates the aligned gc center coordinates in the scaled and
% registered spaces.
% unlike all previous versions, _v6 does not save the transformed gc
% images. instead, the 4x4 final transfomation matrices are saved. the
% unused (0-valued) entries 4th row, colums 1:3 are used to save the center
% (XYZ) of each gc volume in the scaled stacks.
%%
% directory. reading the previously scaled stacks
if strcmp(Ch,'G')
    scaDir = si.scaG;
    locDir = si.mLocG;
else
    scaDir = si.scaR;
    locDir = si.mLocR;
end
scaList  = dir(cat(2,scaDir,'*.tif'));
if isempty(SIv.trakSpan)
    startIdx = 1; numStacks = numel(scaList);
else
    startIdx = SIv.trakSpan(1); stopIdx = SIv.trakSpan(2);
    numStacks = stopIdx-startIdx+1;
end
shftSeedIdx = SIv.seedIdx-startIdx+1;
% reading seed stack and recording full stack dimensions
[templateStruct, sizZ] = tiffread2([scaDir scaList(SIv.seedIdx).name]);
sizX = templateStruct(1).width; sizY = templateStruct(1).height;
seedStack = zeros(sizY,sizX,sizZ,'uint16');
for j0 = 1:sizZ; seedStack(:,:,j0) = templateStruct(j0).data; end
clear templateStruct

% extracting registered image dimensions. note that the registered stacks
% were created using a Fiji macro. the process involves centered-embedding the
% scaled stacks in a larger super-stack, applying the registration matrix,
% and cropping z dimension of the transformed stack. the dimensions of the
% the super-stack as well as the z limits of the crop operation are
% manually saved in the file read in the line below. these values are used
% in the conversion of the GC track coordinates, generated in the
% registered stacks, to the scaled stack reference.
regDims = s_readFijiXformLims_v2(si);
regX = regDims.xBot-regDims.xTop+1; % regXYZ are the post-crop dimensions of the registered stacks
regY = regDims.yBot-regDims.yTop+1;
regZ = regDims.zBot-regDims.zTop+1;
M_t1 = eye(4); M_t1(1:3,4) = -1*0.5*[regDims.regX; regDims.regY; regDims.regZ]; % the translation matrix uses the embedded dimensions of the registered stacks;
M_t1rev = eye(4); M_t1rev(1:3,4) = 0.5*[regDims.regX; regDims.regY; regDims.regZ];
M_t2 = eye(4); M_t2(1:3,4) = 0.5*[sizX; sizY; sizZ];
M_t2rev = eye(4); M_t2rev(1:3,4) = -1*0.5*[sizX; sizY; sizZ];
% distance from trackGC coordinates to the center of the gc cylinder. set
% as 6 um.
L = round(SIv.gcDims(4)/SIv.xyRes);

% need to read outputs of earlier scripts.
trackGCsTemp = dlmread([si.txt 'TrackGCs.txt'],'\t');
unitGCs = dlmread([si.txt 'UnitGCs.txt'],'\t');
% reshape trackGCs
trackGCs = zeros(3,size(unitGCs,2),numStacks);
for i0 = 1:numStacks
    trackGCs(:,:,i0) = trackGCsTemp(:,1+(i0-1)*size(unitGCs,2):i0*size(unitGCs,2));
end
clear trackGCsTemp
% identify goodGCs, ones that have been tracked through the time series.
goodGCs = find(trackGCs(1,:,1)); 
numGCs = numel(goodGCs);
% find the GC voi centers. i am defining this as L pixels along unitGCs,
% starting from the coordinates in trackGCs.
% the coordinates, originally defined in the registered stacks are
% transformed back to the scaled stacks.
centGCs = zeros(3,length(goodGCs),numStacks);
% in the same loop i will also transform the unit vectors to the scaled
% stack reference frame. see comments below on how these vectors are
% calculated.

% calculate two vectors orthogonal to unitGCs. we will treat unitGCs as the
% z component and set the 'y' component's x value to 0. the values of the y
% and z values of this vector then fall out of the definition of the dot
% product. the third vector, the 'x' component will be found by crossing
% 'z' and 'y'.
unitY = zeros(3,length(goodGCs)); unitX = unitY;

FijiGCRunM = zeros(4,4,length(goodGCs),numStacks); 
FijiGCinitM = zeros(4,4,length(goodGCs),numStacks); 
FijiGCfullM = zeros(4,4,length(goodGCs),numStacks); 

% i will transform the vois to align with the mask and then
% register the masked target voi to the template voi. the transformation of
% the original vois with the running matrices should give me gc volumes that
% are registered through time and in standard orientation (i.e. that of the
% cylMask).
disp('transforming GC centers to the scaled stack coordinates and');
disp('constructing GC vectors in the scaled stack reference frame.');
for i0 = 1:numStacks
     truIdx = i0+startIdx-1;
     stkName = scaList(truIdx).name;
     stkIdx = str2double(stkName(regexp(stkName,'\d')));
     regCents = trackGCs(:,goodGCs,i0) + (L * unitGCs(:,goodGCs));
     XformFiji = dlmread([si.mGloFx 'XFullFiji' num2str(stkIdx,'%03.0f') '.txt'],'\t');
     RotFiji = eye(4); RotFiji(1:3,1:3) = XformFiji(1:3,1:3);
     for i1 = 1:numGCs
         % mapping the GC centers defined in the registered stacks to the
         % scaled stack coordinates
         oldCent = ones(4,1);
         oldCent(1) = regCents(1,i1)+regDims.xTop-1; % correcting for the x crop.
         oldCent(2) = regCents(2,i1)+regDims.yTop-1; % correcting for the y crop.
         oldCent(3) = regCents(3,i1)+regDims.zTop-1; % correcting for the z crop.
         newCent = M_t2*inv(XformFiji)*M_t1*oldCent;
         centGCs(:,i1,i0) = round(newCent(1:3));
         % calculating unit vectors and transforming them back to the
         % scaled stack reference frame
         if i0 == 1
             unitY(2,i1) = 1;
             unitY(3,i1) = -1*(unitGCs(2,goodGCs(i1))./unitGCs(3,goodGCs(i1)));
             unitY(:,i1) = unitY(:,i1)./norm(unitY(:,i1));
             unitX(:,i1) = cross(unitY(:,i1),unitGCs(:,goodGCs(i1)));
         end
         tempU = eye(4);
         tempU(1:3,1:3) = [unitX(:,i1) unitY(:,i1) unitGCs(:,goodGCs(i1))];
         FijiGCinitM(:,:,i1,i0) = inv(inv(RotFiji)*tempU);
         if i0 == shftSeedIdx
             FijiGCRunM(:,:,i1,shftSeedIdx) = eye(4);
             FijiGCfullM(:,:,i1,shftSeedIdx) = FijiGCinitM(:,:,i1,shftSeedIdx);
         end
     end
end

disp('building the GC mask.');
legMask = round(12.5/SIv.xyRes);
dimMask = 2*legMask+1;
radius = round(3/SIv.xyRes);
gcMask = zeros(dimMask,dimMask,dimMask);
gcMask(legMask+1,legMask+1,radius:end-radius+1) = 1;
gcDisk = gcMask(:,:,legMask+1); gcDisk = double(bwdist(gcDisk)<round(8/SIv.xyRes));
gcMask(:,:,legMask+1-L) = gcDisk;
gcMask = double(bwdist(gcMask)<radius);
gcMask = imgaussian(gcMask,round(0.5*radius)-1);
% allocate QC output variables
% preparing the montage stack

Options = struct('Similarity','cc','Registration','Rigid',...
                 'Penalty',1e-5,'MaxRef',2,'Grid',[],...
                 'Spacing',[],'MaskMoving',[],'MaskStatic',[],...
                 'Verbose',0,'Points1',[],'Points2',[],...
                 'PStrength',[],'Interpolation','Linear','Scaling',[1 1 1]);


% define volume dimensions
leg3Dful = round(SIv.gcDims(1)/SIv.xyRes); % half dim of the extracted volume
leg2Dsub = round(SIv.gcDims(2)/SIv.xyRes); % half XY dim of the saved volume
leg2Dmip = round(SIv.gcDims(3)/SIv.xyRes); % half XY dim of the projection
dim3Dful = 2*leg3Dful+1;
dim2Dsub = 2*leg2Dsub+1;
dim2Dmip = 2*leg2Dmip+1;
% build mip mask
mipMsk = zeros(2*leg3Dful+1,2*leg3Dful+1,2*leg3Dful+1);
mipMsk(leg3Dful+1,leg3Dful+1,:) = 1;
mipMsk = bwdist(mipMsk)<leg2Dmip; mipMsk = imgaussian(single(mipMsk),3);

% define temporary and final montage variables.
annoWidth = 9;
stkVMIP_pf = zeros(dim3Dful,annoWidth+dim2Dmip,numGCs,'uint16'); % re-initialize for each GC loop
stkZMIP_pf = zeros(dim3Dful,annoWidth+dim2Dmip,numGCs,'uint16'); % re-initialize for each GC loop
stkYMIP_pf = zeros(dim2Dsub,annoWidth+dim2Dsub,numGCs,'uint16'); % re-initialize for each GC loop
montVMIP = stak2mont(stkVMIP_pf); % this variable is used only to extract the montage dimensions in the next line.
montZMIP = stak2mont(stkZMIP_pf); % this variable is used only to extract the montage dimensions in the next line.
montYMIP = stak2mont(stkYMIP_pf); % this variable is used only to extract the montage dimensions in the next line.

montVMIPStack = zeros(size(montVMIP,1), size(montVMIP,2), numStacks,'uint16'); % output variable
montZMIPStack = zeros(size(montZMIP,1), size(montZMIP,2), numStacks,'uint16'); % output variable
montYMIPStack = zeros(size(montYMIP,1), size(montYMIP,2), numStacks,'uint16'); % output variable
projVal = zeros(numStacks,numGCs); % fractional size of the GCs main axis' projection onto the XY plane. 
projAng = zeros(numStacks,numGCs); % angle between the GCs main axis' and the XY plane, measured in the YZ plane. inverse cosine of projVal.

gcSNR = zeros(numGCs,numStacks);% initiate snr array 

centGCsca = zeros(numGCs,3,numStacks,'uint16'); % GC center coordinates in scaled space; saved as images for convenience.
centGCreg = zeros(numGCs,3,numStacks,'uint16'); % GC center coordinates in registered space; saved as images for convenience.

trkGC3Dsca = zeros(sizY,sizX,sizZ,'uint16'); % GC centers painted on a sca stack
trkGC3Dreg = zeros(regY,regX,regZ,'uint16'); % GC centers painted on a reg stack

algnQC = zeros(numGCs,numStacks); % 2 id alignment is successful, 1 if mapped sca coordinates are out of bounds, 0 if GC slips in registration
algnQC(:,shftSeedIdx) = 2; % starting with good quality seed
disp('creating directories for each GC...')

% in the same loop, process the seedIdx stack GCs
stkName = scaList(SIv.seedIdx).name;
stkIdx = str2double(stkName(regexp(stkName,'\d')));
hf = figure;
for i1 = 1:numGCs
    dirGC = ['GC' num2str(goodGCs(i1),'%03.0f')];
    mkdir([locDir dirGC]);
    
    Mfull = FijiGCfullM(:,:,i1,shftSeedIdx);
    % decomposing the full affine transformation matrix into
    % Mrot*Mtra=Mfull. in this order of multiplication, the rotation
    % indices are independent of initial translation, making the decomp
    % straightforward.
    Mrot = eye(4); Mrot(1:3,1:3) = Mfull(1:3,1:3);
    Mtra = (1/10000)*round(10000*((eye(4)/Mrot)*Mfull)); % the rounding flanked by the scaling up and down cleans up sign errors.
    MrInv = eye(4)/Mrot;
    % in the sca stack frame, zThet is the angle between the y axis and
    % the y component of the unit z vector when the standard GC view is transformed back
    % to the sca stack frame. it is used as a correction factor so that
    % the final images are aligned with the vertical axis.
    zThet = acosd(dot([0;1],MrInv(1:2,3))/norm(MrInv(1:2,3)));
    zRot = eye(4);
    zRot(1,1:2) = [cosd(zThet) -sind(zThet)];
    zRot(2,1:2) = [sind(zThet) cosd(zThet)];
    % once the z rotation is factored out, the only correction required to
    % align the GC along the horizontal axis is a rotation about X, stored
    % in the projAng variable.
    Mcomp = 0.0001*round(10000*zRot*MrInv); % keepin it real
    projVal(shftSeedIdx,i1) = Mcomp(2,3); projAng(shftSeedIdx,i1) = acosd(Mcomp(2,3));
    
    volGC = im2double(extractROI_3D(centGCs(:,i1,shftSeedIdx),[leg3Dful;leg3Dful;leg3Dful],seedStack)); % carve the volume from the full stack
    volGCt = affine_transform(volGC,Fiji2Kroon(zRot*Mtra),3); % transform the volume
    mipMskt = affine_transform(mipMsk,Fiji2Kroon(Mcomp),3);
    
    M0 = FijiGCRunM(:,:,i1,shftSeedIdx);
    M_iTemp = FijiGCinitM(:,:,i1,shftSeedIdx);
    volGCv = affine_transform(volGC,Fiji2Kroon(M0*M_iTemp),3); % saving the '(v)ertical' GC mip in a montage for comparison.
    VMIP = zMIP_v1(rot90_3D(volGCv.*mipMsk,2,3),1,dim3Dful);
    VMIP = VMIP(:,leg3Dful+1-leg2Dmip:leg3Dful+1+leg2Dmip);
    VMIP = im2uint16(VMIP);
    VMIP = imadjust(VMIP,stretchlim(VMIP,0),[]);
    stkVMIP_pf(:,:,i1) = [zeros(dim3Dful,annoWidth,'uint16') VMIP]; % pad with the annotation strip
    
    ZMIP = zMIP_v1(volGCt.*mipMskt,1,dim3Dful); % mask the zmip
    ZMIP = ZMIP(:,leg3Dful+1-leg2Dmip:leg3Dful+1+leg2Dmip); % crop the zmip
    ZMIP = im2uint16(ZMIP);
    % calculate SNR
    zMIP = single(ZMIP);
    zMIPg = imgaussian(zMIP,1);
    kNZ = find(zMIP); % catch, in case we have some zero padded pixels in the mip.
    [~,kMsk] = kmeans_oa_v(round(zMIPg),kNZ,2);
    kGC = find(kMsk(kNZ)==2);
    eNoise = evar(zMIP(kNZ)); % noise variance estimate, relative to normal distribution with std=1
    zMIPw = wiener2(zMIP,[3 3],eNoise*mean(zMIP(kNZ(kGC)))); % 'cleaned' image, using estimated noise variance power
    snr = -10*log10(sum((zMIPw(kNZ(kGC))-zMIP(kNZ(kGC))).^2) / sum(zMIPw(kNZ(kGC)).^2));
    gcSNR(i1,shftSeedIdx) = snr;
    ZMIP = imadjust(ZMIP,stretchlim(ZMIP,0),[]); % adjust the intensity
    stkZMIP_pf(:,:,i1) = [zeros(dim3Dful,annoWidth,'uint16') ZMIP]; % pad with the annotation strip
    
    YMIP = extractROI_3D([leg3Dful+1;leg3Dful+1;leg3Dful+1],[leg2Dsub;leg3Dful;leg2Dsub],volGCt); % the ymip square is smaller than the full volume
    YMIP = zMIP_v1(rot90_3D(YMIP,2,3),1,dim3Dful);
    YMIP = im2uint16(YMIP);
    YMIP = imadjust(YMIP,stretchlim(YMIP,0),[]); % adjust the intensity
    stkYMIP_pf(:,:,i1) = [zeros(dim2Dsub,annoWidth,'uint16') YMIP]; % pad with the annotation strip
    
    M_gc = zRot*Mtra;
    M_gc(4,1:3) = centGCs(:,i1,shftSeedIdx)';
    dlmwrite([locDir dirGC '/' dirGC '-' num2str(stkIdx,'%03.0f') '.txt'],M_gc,'\t');
    % calculating GC center coordinates.
    XformFiji = dlmread([si.mGloFx 'XFullFiji' num2str(stkIdx,'%03.0f') '.txt'],'\t');
    [algnCtrSca1, algnCtrReg1] = sint_mapGCcents(L,XformFiji,regDims,M_gc,projAng(shftSeedIdx,i1),M_t1rev,M_t2rev);
    if any(algnCtrSca1) % mapped GC coordinates fall in sca stack limits
        algnCtrSca1 = uint16(round(algnCtrSca1'));
        algnCtrReg1 = uint16(round(algnCtrReg1'));
        centGCsca(i1,:,shftSeedIdx) = algnCtrSca1;
        centGCreg(i1,:,shftSeedIdx) = algnCtrReg1;
    else
        centGCsca(i1,:,i0) = [0 0 0];
        centGCreg(i1,:,i0) = [0 0 0];
        algnQC(i1,shftSeedIdx) = 1;
    end
       
end
disp('done.');

montVMIPStack(:,:,shftSeedIdx) = stak2mont(stkVMIP_pf);
montZMIPStack(:,:,shftSeedIdx) = stak2mont(stkZMIP_pf);
montYMIPStack(:,:,shftSeedIdx) = stak2mont(stkYMIP_pf);
legSub2 = floor(5.25/SIv.xyRes)+ceil(5.25/SIv.xyRes);
imshow(montZMIPStack(:,:,shftSeedIdx),[]);

% note2self: in this script, i have switched the conventional use of
% template and target nomenclature in registration. here we have the moving
% stack defined as target and the static stack defined as the template.
tLoop = zeros(numStacks,1);
disp('looping up through the series, from seed stack to end of trak span.');
templateStack = seedStack;

for i0 = (shftSeedIdx+1):numStacks
    tic
    truIdx = i0+startIdx-1;
    stkName = scaList(truIdx).name;
    stkIdx = str2double(stkName(regexp(stkName,'\d')));
    XformFiji = dlmread([si.mGloFx 'XFullFiji' num2str(stkIdx,'%03.0f') '.txt'],'\t');
    % read the template and target stacks. note that for the first round,
    % the template stack has been loaded at the top of the script.
    if i0 > shftSeedIdx+1; templateStack = targetStack; end
    [targetStruct, ~] = tiffread2([scaDir stkName]);
    targetStack = zeros(sizY,sizX,sizZ,'uint16');
    for j0 = 1:sizZ
        targetStack(:,:,j0) = targetStruct(j0).data;
    end
    
    % temp variables for the parfor loop.
    stkVMIP_pf = zeros(dim3Dful,annoWidth+dim2Dmip,numGCs,'uint16'); % re-initialize for each GC loop
    stkZMIP_pf = zeros(dim3Dful,annoWidth+dim2Dmip,numGCs,'uint16'); % re-initialize for each GC loop
    stkYMIP_pf = zeros(dim2Dsub,annoWidth+dim2Dsub,numGCs,'uint16'); % re-initialize for each GC loop
    projVal_pf = zeros(1,numGCs); projAng_pf = zeros(1,numGCs);
    FijiGCRunM_pf = FijiGCRunM(:,:,:,i0-1);
    FijiGCinitM_pf = FijiGCinitM(:,:,:,i0-1);
    centGCs_Temp_pf = centGCs(:,:,i0-1);
    centGCs_Targ_pf = centGCs(:,:,i0);
    gcSNR_pf = gcSNR(:,i0);
    algnQC_pre = algnQC(:,i0-1); % 1 id alignment is successful, 0 if mapped sca coordinates are out of bounds, -1 if GC slips in registration
    algnQC_pf = algnQC_pre;
    centGCsca_pf = zeros(numGCs,3,'uint16');
    centGCreg_pf = zeros(numGCs,3,'uint16');
    
    matlabpool open 6
    parfor i1 = 1:numGCs
        
        disp(cat(2,'stk loop = ',num2str(truIdx),', GC loop = ',num2str(i1),', GCid = ',num2str(goodGCs(i1))))
        if algnQC_pre(i1)==2 % commit to the calcs only if the prior iteration was successful
        
            M0 = FijiGCRunM_pf(:,:,i1);
            if norm(M0(1:3,4))<0.5*legMask % making sure that alignment does not walk off into the distance
                
                M_iTemp = FijiGCinitM_pf(:,:,i1);
                % extract the template and target GC vois.
                templateGC = im2double(extractROI_3D(centGCs_Temp_pf(:,i1),[legMask;legMask;legMask],templateStack));
                templateGC = affine_transform(templateGC,Fiji2Kroon(M0*M_iTemp),3);
                templateGCg = imgaussian(templateGC,1).*gcMask;
                subTemp = templateGCg(legMask+1-legSub2:legMask+1+legSub2,legMask+1-legSub2:legMask+1+legSub2,:);
                
                targetGC = im2double(extractROI_3D(centGCs_Targ_pf(:,i1),[legMask;legMask;legMask],targetStack));
                M_iTarg = FijiGCinitM(:,:,i1,i0);
                targetGC = affine_transform(targetGC,Fiji2Kroon(M0*M_iTarg),3);
                targetGCg = imgaussian(targetGC,1).*gcMask;
                subTarg = targetGCg(legMask+1-legSub2:legMask+1+legSub2,legMask+1-legSub2:legMask+1+legSub2,:);
                
                [~,~,~,M1_K] = register_volumes(subTarg,subTemp,Options);
                M1 = Kroon2Fiji(M1_K);
                % registration is complete. now we extract the volume to be
                % saved from the target stack, translate the center as determined by the registration matrix,
                % and perform a turn in the XY plane to align the GC with the
                % vertical axis. this is the saved volume.
                Mfull = M1*M0*M_iTarg;
                Mrot = eye(4); Mrot(1:3,1:3) = Mfull(1:3,1:3);
                Mtra = (1/10000)*round(10000*((eye(4)/Mrot)*Mfull));
                MrInv = eye(4)/Mrot;
                zThet = acosd(dot([0;1],MrInv(1:2,3))/norm(MrInv(1:2,3)));
                zRot = eye(4); zRot(1,1:2) = [cosd(zThet) -sind(zThet)]; zRot(2,1:2) = [sind(zThet) cosd(zThet)];
                Mcomp = 0.0001*round(10000*zRot*MrInv);
                projVal_pf(1,i1) = Mcomp(2,3); projAng_pf(1,i1) = acosd(Mcomp(2,3));
                
                volGC = im2double(extractROI_3D(centGCs_Targ_pf(:,i1),[leg3Dful;leg3Dful;leg3Dful],targetStack)); % carve the volume from the full stack
                volGCt = affine_transform(volGC,Fiji2Kroon(zRot*Mtra),3); % transform the volume
                mipMskt = affine_transform(mipMsk,Fiji2Kroon(Mcomp),3);
                volGCv = affine_transform(volGC,Fiji2Kroon(M1*M0*M_iTarg),3); % saving the '(v)ertical' GC mip in a montage for comparison.
                
                M_gc = zRot*Mtra;
                M_gc(4,1:3) = centGCs_Targ_pf(:,i1)';
                
                [algnCtrSca1, algnCtrReg1] = sint_mapGCcents(L,XformFiji,regDims,M_gc,projAng_pf(1,i1),M_t1rev,M_t2rev);
                
                % this catch does not work when there are 0 values in the raw data 
%                 % using the zmip of the transformed volume as a second catch on out
%                 % of bounds GCs. the central 2*legSub2+1 rows of the image
%                 % should be free of zero values. this is probably a more
%                 % stringent constraint than looking at the mapping of
%                 % aligned GC centers as done in sint_mapGCcents.
%                 tMIP = zMIP_v1(volGCt,1,dim3Dful); % mask the zmip
%                 tMIP = tMIP(leg3Dful+1-legSub2:leg3Dful+1+legSub2,leg3Dful+1-leg2Dmip:leg3Dful+1+leg2Dmip);
%                 if min(double(im2uint16(tMIP(:))))==0
%                     algnCtrSca1 = [0;0;0];algnCtrReg1 = [0;0;0];
%                 end
                
                if any(algnCtrSca1) % mapped GC coordinates fall in sca stack limits
                    algnCtrSca1 = uint16(round(algnCtrSca1'));
                    algnCtrReg1 = uint16(round(algnCtrReg1'));
                    centGCsca_pf(i1,:) = algnCtrSca1;
                    centGCreg_pf(i1,:) = algnCtrReg1;
                    algnQC_pf(i1) = 2;
                    
                    FijiGCRunM(:,:,i1,i0) = M1*M0;
                    FijiGCfullM(:,:,i1,i0) = Mfull;
                    
                    VMIP = zMIP_v1(rot90_3D(volGCv.*mipMsk,2,3),1,dim3Dful);
                    VMIP = VMIP(:,leg3Dful+1-leg2Dmip:leg3Dful+1+leg2Dmip);
                    VMIP = im2uint16(VMIP);
                    VMIP = imadjust(VMIP,stretchlim(VMIP,0),[]);
                    stkVMIP_pf(:,:,i1) = [zeros(dim3Dful,annoWidth,'uint16') VMIP]; % pad with the annotation strip
                    
                    ZMIP = zMIP_v1(volGCt.*mipMskt,1,dim3Dful); % mask the zmip
                    ZMIP = ZMIP(:,leg3Dful+1-leg2Dmip:leg3Dful+1+leg2Dmip);
                    ZMIP = im2uint16(ZMIP);
                    % calculate SNR
                    zMIP = single(ZMIP);
                    zMIPg = imgaussian(zMIP,1);
                    kNZ = find(zMIP); % catch, in case we have some zero padded pixels in the mip.
                    [~,kMsk] = kmeans_oa_v(round(zMIPg),kNZ,2);
                    kGC = find(kMsk(kNZ)==2);
                    eNoise = evar(zMIP(kNZ)); % noise variance estimate, relative to normal distribution with std=1
                    zMIPw = wiener2(zMIP,[3 3],eNoise*mean(zMIP(kNZ(kGC)))); % 'cleaned' image, using estimated noise variance power
                    snr = -10*log10(sum((zMIPw(kNZ(kGC))-zMIP(kNZ(kGC))).^2) / sum(zMIPw(kNZ(kGC)).^2));
                    gcSNR_pf(i1) = snr;
                    
                    ZMIP = imadjust(ZMIP,stretchlim(ZMIP,0),[]); % adjust the intensity
                    stkZMIP_pf(:,:,i1) = [zeros(dim3Dful,annoWidth,'uint16') ZMIP]; % pad with the annotation strip
                    
                    YMIP = extractROI_3D([leg3Dful+1;leg3Dful+1;leg3Dful+1],[leg2Dsub;leg3Dful;leg2Dsub],volGCt); % the ymip square is smaller than the full volume
                    YMIP = zMIP_v1(rot90_3D(YMIP,2,3),1,dim3Dful);
                    YMIP = im2uint16(YMIP);
                    YMIP = imadjust(YMIP,stretchlim(YMIP,0),[]); % adjust the intensity
                    stkYMIP_pf(:,:,i1) = [zeros(dim2Dsub,annoWidth,'uint16') YMIP]; % pad with the annotation strip
                    
                    dirGC = ['GC' num2str(goodGCs(i1),'%03.0f')];
                    fileName = [locDir dirGC '/' dirGC '-' num2str(stkIdx,'%03.0f') '.txt'];
                    if exist(fileName,'file') ~= 0 % delete result of previous run, if it exists.
                        delete(fileName);
                    end
                    dlmwrite([locDir dirGC '/' dirGC '-' num2str(stkIdx,'%03.0f') '.txt'],M_gc,'\t');
                else
                    centGCsca_pf(i1,:) = [0 0 0];
                    centGCreg_pf(i1,:) = [0 0 0];
                    algnQC_pf(i1) = 1;
                end            
            else
                centGCsca_pf(i1,:) = [0 0 0];
                centGCreg_pf(i1,:) = [0 0 0];
                algnQC_pf(i1) = 0; 
            end
        end
    end
    matlabpool close
    projVal(i0,:) = projVal_pf;
    projAng(i0,:) = projAng_pf;
    montVMIPStack(:,:,i0) = stak2mont(stkVMIP_pf);
    montZMIPStack(:,:,i0) = stak2mont(stkZMIP_pf);
    montYMIPStack(:,:,i0) = stak2mont(stkYMIP_pf);
    gcSNR(:,i0) = gcSNR_pf;
    centGCsca(:,:,i0) = centGCsca_pf;
    centGCreg(:,:,i0) = centGCreg_pf;
    algnQC(:,i0) = algnQC_pf;
%     k1 = find(algnQC_pf==1);
%     if ~isempty(k1)
%         algnQC(k1,i0:end) = 1;
%     end
    tLoop(i0) = toc;
    disp(['time taken to process stack loop ' num2str(truIdx) ' was ' num2str(tLoop(i0)) ' seconds.']);
    imshow(montZMIPStack(:,:,i0),[]);
end

disp('looping down through the series, from seed stack to beginning of trak span.');
templateStack = seedStack;
for i0 = (shftSeedIdx-1):-1:1
    tic
    truIdx = i0+startIdx-1;
    stkName = scaList(truIdx).name;
    stkIdx = str2double(stkName(regexp(stkName,'\d')));
    XformFiji = dlmread([si.mGloFx 'XFullFiji' num2str(stkIdx,'%03.0f') '.txt'],'\t');
    % read the template and target stacks. note that for the first round,
    % the template stack has been loaded at the top of the script.
    if i0 < shftSeedIdx-1; templateStack = targetStack; end
    [targetStruct, ~] = tiffread2([scaDir stkName]);
    targetStack = zeros(sizY,sizX,sizZ,'uint16');
    for j0 = 1:sizZ
        targetStack(:,:,j0) = targetStruct(j0).data;
    end
    
    % temp variables for the parfor loop.
    stkVMIP_pf = zeros(dim3Dful,annoWidth+dim2Dmip,numGCs,'uint16'); % re-initialize for each GC loop
    stkZMIP_pf = zeros(dim3Dful,annoWidth+dim2Dmip,numGCs,'uint16'); % re-initialize for each GC loop
    stkYMIP_pf = zeros(dim2Dsub,annoWidth+dim2Dsub,numGCs,'uint16'); % re-initialize for each GC loop
    projVal_pf = zeros(1,numGCs); projAng_pf = zeros(1,numGCs);
    FijiGCRunM_pf = FijiGCRunM(:,:,:,i0+1);
    FijiGCinitM_pf = FijiGCinitM(:,:,:,i0+1);
    centGCs_Temp_pf = centGCs(:,:,i0+1);
    centGCs_Targ_pf = centGCs(:,:,i0);
    gcSNR_pf = gcSNR(:,i0);
    algnQC_pre = algnQC(:,i0+1); % 1 id alignment is successful, 0 if mapped sca coordinates are out of bounds, -1 if GC slips in registration
    algnQC_pf = algnQC_pre;
    centGCsca_pf = zeros(numGCs,3,'uint16');
    centGCreg_pf = zeros(numGCs,3,'uint16');
    matlabpool open 6

    parfor i1 = 1:numGCs
       
        disp(cat(2,'stk loop = ',num2str(i0),', GC loop = ',num2str(i1),', GCid = ',num2str(goodGCs(i1))))
        % new approach to t-matrix multiplication: do it all in Fiji
        % format, convert the final product to Kroon, when needed.
        
        if algnQC_pre(i1)==2 % commit to the calcs only if the prior iteration was successful

            M0 = FijiGCRunM_pf(:,:,i1);
            if norm(M0(1:3,4))<0.5*legMask % making sure that alignment does not walk off into the distance
                M_iTemp = FijiGCinitM_pf(:,:,i1);
                % extract the template and target GC vois.
                templateGC = im2double(extractROI_3D(centGCs_Temp_pf(:,i1),[legMask;legMask;legMask],templateStack));
                templateGC = affine_transform(templateGC,Fiji2Kroon(M0*M_iTemp),3);
                templateGCg = imgaussian(templateGC,1).*gcMask;
                subTemp = templateGCg(legMask+1-legSub2:legMask+1+legSub2,legMask+1-legSub2:legMask+1+legSub2,:);
                
                targetGC = im2double(extractROI_3D(centGCs_Targ_pf(:,i1),[legMask;legMask;legMask],targetStack));
                M_iTarg = FijiGCinitM(:,:,i1,i0);
                targetGC = affine_transform(targetGC,Fiji2Kroon(M0*M_iTarg),3);
                targetGCg = imgaussian(targetGC,1).*gcMask;
                subTarg = targetGCg(legMask+1-legSub2:legMask+1+legSub2,legMask+1-legSub2:legMask+1+legSub2,:);
                
                [~,~,~,M1_K] = register_volumes(subTarg,subTemp,Options);
                M1 = Kroon2Fiji(M1_K);
                % registration is complete. now we extract the volume to be
                % saved from the target stack, translate the center as determined by the registration matrix,
                % and perform a turn in the XY plane to align the GC with the
                % vertical axis. this is the saved volume.
                Mfull = M1*M0*M_iTarg;
                Mrot = eye(4); Mrot(1:3,1:3) = Mfull(1:3,1:3);
                Mtra = (1/10000)*round(10000*((eye(4)/Mrot)*Mfull));
                MrInv = eye(4)/Mrot;
                zThet = acosd(dot([0;1],MrInv(1:2,3))/norm(MrInv(1:2,3)));
                zRot = eye(4); zRot(1,1:2) = [cosd(zThet) -sind(zThet)]; zRot(2,1:2) = [sind(zThet) cosd(zThet)];
                Mcomp = 0.0001*round(10000*zRot*MrInv);
                projVal_pf(1,i1) = Mcomp(2,3); projAng_pf(1,i1) = acosd(Mcomp(2,3));
                
                volGC = im2double(extractROI_3D(centGCs_Targ_pf(:,i1),[leg3Dful;leg3Dful;leg3Dful],targetStack)); % carve the volume from the full stack
                volGCt = affine_transform(volGC,Fiji2Kroon(zRot*Mtra),3); % transform the volume
                mipMskt = affine_transform(mipMsk,Fiji2Kroon(Mcomp),3);
                volGCv = affine_transform(volGC,Fiji2Kroon(M1*M0*M_iTarg),3); % saving the '(v)ertical' GC mip in a montage for comparison.
                
                M_gc = zRot*Mtra;
                M_gc(4,1:3) = centGCs_Targ_pf(:,i1)';
                
                [algnCtrSca1, algnCtrReg1] = sint_mapGCcents(L,XformFiji,regDims,M_gc,projAng_pf(1,i1),M_t1rev,M_t2rev);
                % this catch does not work when there are 0 values in the raw data 
%                 % using the zmip of the transformed volume as a second catch on out
%                 % of bounds GCs. the central 2*legSub2+1 rows of the image
%                 % should be free of zero values. this is probably a more
%                 % stringent constraint than looking at the mapping of
%                 % aligned GC centers as done in sint_mapGCcents.
%                 tMIP = zMIP_v1(volGCt,1,dim3Dful); % mask the zmip
%                 tMIP = tMIP(leg3Dful+1-legSub2:leg3Dful+1+legSub2,leg3Dful+1-leg2Dmip:leg3Dful+1+leg2Dmip);
%                 if min(double(im2uint16(tMIP(:))))==0
%                     algnCtrSca1 = [0;0;0];algnCtrReg1 = [0;0;0];
%                 end

                if any(algnCtrSca1) % mapped GC coordinates fall in sca stack limits
                    algnCtrSca1 = uint16(round(algnCtrSca1'));
                    algnCtrReg1 = uint16(round(algnCtrReg1'));
                    centGCsca_pf(i1,:) = algnCtrSca1;
                    centGCreg_pf(i1,:) = algnCtrReg1;
                    algnQC_pf(i1) = 2;
                    
                    FijiGCRunM(:,:,i1,i0) = M1*M0;
                    FijiGCfullM(:,:,i1,i0) = Mfull;
                    
                    VMIP = zMIP_v1(rot90_3D(volGCv.*mipMsk,2,3),1,dim3Dful);
                    VMIP = VMIP(:,leg3Dful+1-leg2Dmip:leg3Dful+1+leg2Dmip);
                    VMIP = im2uint16(VMIP);
                    VMIP = imadjust(VMIP,stretchlim(VMIP,0),[]);
                    stkVMIP_pf(:,:,i1) = [zeros(dim3Dful,annoWidth,'uint16') VMIP]; % pad with the annotation strip
                    
                    ZMIP = zMIP_v1(volGCt.*mipMskt,1,dim3Dful); % mask the zmip
                    ZMIP = ZMIP(:,leg3Dful+1-leg2Dmip:leg3Dful+1+leg2Dmip);
                    ZMIP = im2uint16(ZMIP);
                    % calculate SNR
                    zMIP = single(ZMIP);
                    zMIPg = imgaussian(zMIP,1);
                    kNZ = find(zMIP); % catch, in case we have some zero padded pixels in the mip.
                    [~,kMsk] = kmeans_oa_v(round(zMIPg),kNZ,2);
                    kGC = find(kMsk(kNZ)==2);
                    eNoise = evar(zMIP(kNZ)); % noise variance estimate, relative to normal distribution with std=1
                    zMIPw = wiener2(zMIP,[3 3],eNoise*mean(zMIP(kNZ(kGC)))); % 'cleaned' image, using estimated noise variance power
                    snr = -10*log10(sum((zMIPw(kNZ(kGC))-zMIP(kNZ(kGC))).^2) / sum(zMIPw(kNZ(kGC)).^2));
                    gcSNR_pf(i1) = snr;
                    
                    ZMIP = imadjust(ZMIP,stretchlim(ZMIP,0),[]); % adjust the intensity
                    stkZMIP_pf(:,:,i1) = [zeros(dim3Dful,annoWidth,'uint16') ZMIP]; % pad with the annotation strip
                    
                    YMIP = extractROI_3D([leg3Dful+1;leg3Dful+1;leg3Dful+1],[leg2Dsub;leg3Dful;leg2Dsub],volGCt); % the ymip square is smaller than the full volume
                    YMIP = zMIP_v1(rot90_3D(YMIP,2,3),1,dim3Dful);
                    YMIP = im2uint16(YMIP);
                    YMIP = imadjust(YMIP,stretchlim(YMIP,0),[]); % adjust the intensity
                    stkYMIP_pf(:,:,i1) = [zeros(dim2Dsub,annoWidth,'uint16') YMIP]; % pad with the annotation strip
                    
                    dirGC = ['GC' num2str(goodGCs(i1),'%03.0f')];
                    fileName = [locDir dirGC '/' dirGC '-' num2str(stkIdx,'%03.0f') '.txt'];
                    if exist(fileName,'file') ~= 0 % delete result of previous run, if it exists.
                        delete(fileName);
                    end
                    dlmwrite([locDir dirGC '/' dirGC '-' num2str(stkIdx,'%03.0f') '.txt'],M_gc,'\t');
                else
                    centGCsca_pf(i1,:) = [0 0 0];
                    centGCreg_pf(i1,:) = [0 0 0];
                    algnQC_pf(i1) = 1;
                end
            else
                centGCsca_pf(i1,:) = [0 0 0];
                centGCreg_pf(i1,:) = [0 0 0];
                algnQC_pf(i1) = 0;
            end
        end
    end
    matlabpool close
    projVal(i0,:) = projVal_pf;
    projAng(i0,:) = projAng_pf;
    montVMIPStack(:,:,i0) = stak2mont(stkVMIP_pf);
    montZMIPStack(:,:,i0) = stak2mont(stkZMIP_pf);
    montYMIPStack(:,:,i0) = stak2mont(stkYMIP_pf);
    gcSNR(:,i0) = gcSNR_pf;
    centGCsca(:,:,i0) = centGCsca_pf;
    centGCreg(:,:,i0) = centGCreg_pf;
    algnQC(:,i0) = algnQC_pf;
%     k1 = find(algnQC_pf==1);
%     if ~isempty(k1)
%         algnQC(k1,1:i0) = 1;
%     end
    tLoop(i0) = toc;
    disp(['time taken to process stack loop ' num2str(truIdx) ' was ' num2str(tLoop(i0)) ' seconds.']);
    imshow(montZMIPStack(:,:,i0),[]);
end
close(hf)

% preparing the annotation text
hf = figure('color','w');
image(ones(dim3Dful,annoWidth+dim2Dmip));

set(gca,'units','pixels','position',[5 5 (annoWidth+dim2Dmip)-1 (dim3Dful)-1],'visible','off');
montZAnno = zeros(size(montZMIP,1), size(montZMIP,2), numStacks);
montYAnno = zeros(size(montYMIP,1), size(montYMIP,2), numStacks);
gcDepth = zeros(numStacks,numGCs);
for i0 = 1:numStacks
    annoZStk = zeros(dim3Dful,annoWidth+dim2Dmip);
    annoYStk = zeros(dim2Dsub,annoWidth+dim2Dsub);
    for i1 = 1:numGCs
        if algnQC(i1,i0)==2
            gcDepth(i0,i1) = SIv.voxAR(3)*SIv.xyRes*((double(centGCsca(i1,3,i0))/SIv.voxAR(3))+SIv.ScaZ_stkLims(1)-1); % micron depth of GC center from raw stack top.
            ht = text('units','pixels','color','k','position',[5 2],'rotation',90,'fontsize',10,'string',...
                [num2str(goodGCs(i1),'%03.0f') ' pVal=' num2str(projVal(i0,i1),'%03.2f') ' Depth=' num2str(gcDepth(i0,i1),'%3.0f')]);
        elseif algnQC(i1,i0)==1
            ht = text('units','pixels','color','k','position',[5 2],'rotation',90,'fontsize',10,'string',...
                [num2str(goodGCs(i1),'%03.0f') ' OOB']);
        else
            ht = text('units','pixels','color','k','position',[5 2],'rotation',90,'fontsize',10,'string',...
                [num2str(goodGCs(i1),'%03.0f') ' slip']);
        end
        tim = getframe(gca);
        tim2 = tim.cdata;
        kText = find(tim2(:,:,3)==0);
        textMask = zeros(dim3Dful,annoWidth+dim2Dmip);
        textMask(kText) = 1;
        annoZStk(:,:,i1) = textMask;
        annoYStk(:,1:annoWidth+dim2Dmip,i1) = textMask(dim3Dful-dim2Dsub+1:end,1:annoWidth+dim2Dmip);
        delete(ht)
    end
    montZAnno(:,:,i0) = stak2mont(annoZStk);
    montYAnno(:,:,i0) = stak2mont(annoYStk);
end
close(hf)
kTextZ = find(montZAnno);
kTextY = find(montYAnno);

fileNameV = [si.img 'AlignGCVMIP-' Ch '.tif'];
    if exist(fileNameV,'file') ~= 0 % delete result of previous run, if it exists.
        delete(fileNameV);
    end
fileNameZ = [si.img 'AlignGCZMIP-' Ch '.tif'];
    if exist(fileNameZ,'file') ~= 0 % delete result of previous run, if it exists.
        delete(fileNameZ);
    end
fileNameZ2 = [si.img 'AlignGCZMIP2-' Ch '.tif'];
    if exist(fileNameZ2,'file') ~= 0 % delete result of previous run, if it exists.
        delete(fileNameZ2);
    end
fileNameY = [si.img 'AlignGCYMIP-' Ch '.tif'];
    if exist(fileNameY,'file') ~= 0 % delete result of previous run, if it exists.
        delete(fileNameY);
    end
fileNameS = [si.img 'centGCsca-' Ch '.tif'];
    if exist(fileNameS,'file') ~= 0 % delete result of previous run, if it exists.
        delete(fileNameS);
    end
fileNameR = [si.img 'centGCreg-' Ch '.tif'];
    if exist(fileNameR,'file') ~= 0 % delete result of previous run, if it exists.
        delete(fileNameR);
    end
montVMIPStack_anno = montVMIPStack; montVMIPStack_anno(kTextZ) = 2^15;
montZMIPStack_anno = montZMIPStack; montZMIPStack_anno(kTextZ) = 2^15;
montYMIPStack_anno = montYMIPStack; montYMIPStack_anno(kTextY) = 2^15;

k2 = find((sum(algnQC,2)/(2*numStacks))==1);

for i0 = 1:numStacks
    imwrite(uint16(montVMIPStack_anno(:,:,i0)),fileNameV,'tiff','Compression','none','WriteMode','append');
    imwrite(uint16(montZMIPStack_anno(:,:,i0)),fileNameZ,'tiff','Compression','none','WriteMode','append');
    imwrite(uint16(montYMIPStack_anno(:,:,i0)),fileNameY,'tiff','Compression','none','WriteMode','append');
    
    ZMIPstk = mont2stak(montZMIPStack_anno(:,:,i0),[dim3Dful,annoWidth+dim2Dmip,numGCs]);
    ZMIPmont = stak2mont(ZMIPstk(:,:,k2));
    imwrite(uint16(ZMIPmont),fileNameZ2,'tiff','Compression','none','WriteMode','append');
    
    imwrite(centGCsca(:,:,i0),fileNameS,'tiff','Compression','none','WriteMode','append');
    imwrite(centGCreg(:,:,i0),fileNameR,'tiff','Compression','none','WriteMode','append');
end

imwrite(uint8(algnQC),[si.img 'algnQC-' Ch '.tif'],'tiff','Compression','none'); % overwrites by default


for i0 = 1:numStacks
    for i1 = 1:numGCs
        if algnQC(i1,i0) == 2
            s = double(centGCsca(i1,:,i0));
            trkGC3Dsca(s(2),s(1),s(3)) = goodGCs(i1);
            r = double(centGCsca(i1,:,i0));
            if any(r)
                trkGC3Dreg(r(2),r(1),r(3)) = goodGCs(i1);
            end
        end
    end
end

fileNameSI = [si.img 'trkGC3Dsca-' Ch '.tif'];
    if exist(fileNameSI,'file') ~= 0 % delete result of previous run, if it exists.
        delete(fileNameSI);
    end
for i0 = 1:sizZ
    imwrite(trkGC3Dsca(:,:,i0),fileNameSI,'tiff','Compression','none','WriteMode','append');    
end
fileNameRI = [si.img 'trkGC3Dreg-' Ch '.tif'];
    if exist(fileNameRI,'file') ~= 0 % delete result of previous run, if it exists.
        delete(fileNameRI);
    end
for i0 = 1:regZ
    imwrite(trkGC3Dreg(:,:,i0),fileNameRI,'tiff','Compression','none','WriteMode','append');    
end


dlmwrite([si.txt 'FijiAlignGCs.txt'],FijiGCfullM,'\t');
dlmwrite([si.txt 'GoodGCs.txt'],goodGCs,'\t');
dlmwrite([si.txt 'CentGCs.txt'],centGCs,'\t');
dlmwrite([si.txt 'projVal.txt'],projVal,'\t');
dlmwrite([si.txt 'projAng.txt'],projAng,'\t');
dlmwrite([si.txt 'gcSNR-' Ch '.txt'],gcSNR,'\t');
dlmwrite([si.txt 'gcDepth.txt'],gcDepth,'\t');

