function slm = slmfit(x,y,varargin)
% slmfit: GUI tool to estimate a spline function (Shape Langauge Model) from data
% usage 1: slm = slmfit(x,y);
% usage 3: slm = slmengine(x,y,prop1,val1,prop2,val2,...);
%
% Note: slmfit is a GUI tool for fitting a shape language model
%   to data. The user supplies information about the expected curve
%   shape. slmengine is used for the modeling.
%
% Note: The optimization toolbox (lsqlin) is required for most fits.
%
% arguments: (input)
%  x,y     - vectors of data used to fit the model
%
%  prescription - structure generated by slmset to control the
%            initial fit parameters to start the GUI.
%
%            Alternatively, one can simply supply property/value
%            pairs directly to slmfit.
%
% Arguments: (output)
%  slm     - a shape prescriptive model structure, stored in a
%            Hermite form.
%
%            slmeval can be used to evaluate the resulting model
%            at any set of points, or any derivative of the model.
%            
%            Alternatively, the result may be returned in pp form,
%            evaluable by ppval. You may also use slm2pp to convert
%            an slm form into a pp form.



% which form does the shape prescription take? (if any)
if nargin<2
  error(SLMFIT:improperarguments,'Must supply a pair of vectors to fit a curve in the form: y = f(x)')
elseif (nargin > 2) && ~iseven(nargin)
  error(SLMFIT:improperarguments,'Additional arguments must be in pairs, as properties/values')
elseif nargin==2
  % use a default for the shape prescription.
  prescription = slmset('plot','off');
elseif (nargin>=3)
  % prescription supplied directly or as property/value pairs,
  % or as a prescription, modified by a few set of pairs.
  % slmset will resolve all of these cases.
  prescription = slmset(varargin{:});
end

% plot is inappropriate for the gui in case it was specified
if strcmp(prescription.Plot,'on')
  % override the Plot option
  prescription.Plot = 'off';
end

% check the data for size, turning it into column vectors
x=x(:);
y=y(:);
n = length(x);
if n~=length(y)
  error 'x and y must be the same size'
end

% set up the gui
params.BackColor = [.6 .7 .7]; % GUI background color
params.ButtonColor = [.9 .9 .9];
params.MaxStack = 50; % maximum stack size allowed
params.SLMStack = cell(params.MaxStack,1);

InitializeSLMGUI

% Aggregate any parameters for the gui, plot, etc., in one
% place for reference by all nested functions
params.X = x;
params.Y = y;
params.n = n;

% Plot may be 'function', 'residuals', 'dy', 'dy2', 'dy3'
params.Plot = 'function';
if n <= 100
  params.DataMarker = 'o';
else
  params.DataMarker = '.';
end
params.MarkerColor = 'r';
params.LineStyle = '-';
params.LineColor = 'b';
% number of points to evaluate the spline at along the curve
params.Nev = 501;
% some handles to help selectdata to work properly
params.Handles.Data = [];
params.Handles.Curve = [];
% maintain the result in Hermite form, for ease of plotting
% the derivatives later if necessary. We will convert it to
% pp form at the very end if requested.
if strcmpi(prescription.Result,'pp')
  params.Result = 'pp';
  prescription.Result = 'slm';
else
  params.Result = 'slm';
end

% fit the initial model
slm = slmengine(x,y,prescription);

% set up the stack for prescriptions and models to allow undos
% Note that the slm form returned also contains the prescription
% that was used to build it. The character string is used to
% refer to these particular settings in a listbox.
params.SLMStack = {slm 'Initial model'};

% update the gui to reflect the fit statistics, and plot the model
ReportSLMStats(slm)
PlotSLM(slm)

% Initial fit is done and fully reported. Wait for further
% actions so that we can return the final model.
uiwait

% ...

% ... and resume.

% get the final model from the stack
finalmodel = params.SLMStack{end};
slm = finalmodel{1};

% Do we need to convert to pp form?
if strcmpi(params.Result,'pp')
  slm = slm2pp(slm);
end

% close down the figure window
delete(params.Handles.Figure)

% all done now.

% ===================================================
%    nested functions (internal processing)
% ===================================================
function InitializeSLMGUI
  % Build the GUI window & set up the callbacks
  
  % open up a new figure window
  params.Handles.Figure = figure('Color',params.BackColor, ...
    'Units','normalized', ...
    'Position',[.1 .1 .8 .8], ...
    'Name','Shape Language Modeling Tool', ...
    'CloseRequestFcn',@(s,e) CloseSLM(s,e,'X'));
  
  % set the axes
  params.Handles.Axes = axes('Parent',params.Handles.Figure, ...
    'Units','normalized', ...
    'CameraUpVector',[0 1 0], ...
    'Color',[1 1 1], ...
    'Position',[.05 .16 .70 .8], ...
    'Tag','Axes1', ...
    'XColor',[0 0 0], ...
    'YColor',[0 0 0], ...
    'ZColor',[0 0 0]);
  h2 = text('Parent',params.Handles.Axes, ...
    'Color',[0 0 0], ...
    'Units','normalized', ...
    'FontName','helvetica', ...
    'HandleVisibility','off', ...
    'HorizontalAlignment','center', ...
    'Position',[0.496 -0.053 0], ...
    'Tag','Axes1Text4', ...
    'VerticalAlignment','cap');
  set(get(h2,'Parent'),'XLabel',h2);
  h2 = text('Parent',params.Handles.Axes, ...
    'Color',[0 0 0], ...
    'Units','normalized', ...
    'FontName','helvetica', ...
    'HandleVisibility','off', ...
    'HorizontalAlignment','center', ...
    'Position',[-0.06 0.495], ...
    'Rotation',90, ...
    'Tag','Axes1Text3', ...
    'VerticalAlignment','baseline');
  set(get(h2,'Parent'),'YLabel',h2);
  h2 = text('Parent',params.Handles.Axes, ...
    'Color',[0 0 0], ...
    'Units','normalized', ...
    'FontName','helvetica', ...
    'HandleVisibility','off', ...
    'HorizontalAlignment','center', ...
    'Position',[0.497 1.013], ...
    'Tag','Axes1Text1', ...
    'VerticalAlignment','bottom');
  set(get(h2,'Parent'),'Title',h2);
  
  % done button
  params.Handles.Done = uicontrol('Parent',params.Handles.Figure, ...
    'Units','normalized', ...
    'BackgroundColor',params.ButtonColor, ...
    'Position',[0.55 .05 .10 .05], ...
    'String','Done', ...
    'HorizontalAlignment','left', ...
    'Style','PushButton', ...
    'CallBack',@(s,e) CloseSLM(s,e,'button'), ...
    'TooltipString','Terminate and return the fitted spline');
  
  % box to report statistics 
  params.Handles.Stats = uicontrol('Parent',params.Handles.Figure, ...
    'Units','normalized', ...
    'BackgroundColor',[1 1 1], ...
    'Position',[0.76 0.01 .23 .13], ...
    'String','', ...
    'HorizontalAlignment','left', ...
    'Style','listbox', ...
    'Fontname','Monaco', ...
    'fontsize',8, ...
    'TooltipString','Statistics on the spline fit and information about the spline itself');
  
  % undo/revert/redo button
  
  
  
  
  % undo listbox
  
  
  
  
  % menus: knots/breaks
  
  
  
  
  % menus: value
  
  
  
  
  % menus: slope
  
  
  
  
  % menus: shape
  
  
  
  
  % menus: Fit
  
  
  
  
  % menus: plot
  
  
  
  
  
  
  
  
  
end % (nested) InitializeGUI

% ===================================================
%    nested functions (internal processing)
% ===================================================
function ReportSLMStats(slm)
  % computes statistics on the model, the fit, the shape
  % of the function as fit, and stuff that info into the
  % stats box.
  
  % some of the parameters can be obtained from slmpar
  funrange = [slmpar(slm,'minfun'),slmpar(slm,'maxfun')];
  fprange = [slmpar(slm,'minslope'),slmpar(slm,'maxslope')];
  funint = slmpar(slm,'integral');
  
  if slm.degree == 3
    dy2dx2 = slmeval(slm.knots,slm,2);
    fpprange = [min(dy2dx2),max(dy2dx2)];
  else
    fpprange = [0,0];
  end
  
  % residual errors: yhat - y
  resids = slmeval(params.X,slm,0) - params.Y;
  errorrange = [max(resids),min(resids)];
  
  % Standard deviation of the residuals
  stdresids = std(resids);
  
  % R^2, adjusted R^2 are computed by slmengine
  outinfo = cell(8,1);
  outinfo{1} = ['Function min & max:       ',num2str(funrange)];
  outinfo{2} = ['Slope min & max:          ',num2str(fprange)];
  outinfo{3} = ['2nd deriv min & max:      ',num2str(fpprange)];
  outinfo{4} = ['Integral:                 ',num2str(funint)];
  
  outinfo{5} = ['Errors min & max:         ',num2str(slm.stats.ErrorRange)];
  outinfo{6} = ['25 & 75% error quartiles: ',num2str(slm.stats.Quartiles)];
  outinfo{7} = ['RMSE:                     ',num2str(slm.stats.RMSE)];
  outinfo{8} = ['std of residuals:         ',num2str(stdresids)];
  outinfo{9} = ['R-squared:                ',num2str(slm.stats.R2)];
  outinfo{10} = ['Adjusted R-squared:       ',num2str(slm.stats.R2Adj)];
  
  % Stuff the statistics box with this information.
  set(params.Handles.Stats,'String',outinfo)
  
end % (nested) ReportSLMStats

% ===================================================
%    nested functions (internal processing)
% ===================================================
function PlotSLM(slm)
  % 
  
  
  
  
end % (nested) PlotSLM

% ===================================================
%    nested functions (callback processing)
% ===================================================
function CloseSLM(src,evnt,howcalled) %#ok
  % close request has been issued.
  if strcmp(howcalled,'button')
    % don't quibble here
    uiresume
  end
  
  % do we ask if they mean it?
  if (nargin < 3) || isempty(howcalled) || strcmp(howcalled,'X')
    terminate = questdlg('Do you wish to terminate the spline fit process?','Finish?','Yes','No','Yes');
  else
    terminate = 'Yes';
  end
  if strcmp(terminate,'Yes')
    % allow the resume to happen, returning the final spline
    uiresume
  end
end % (nested) qwerty

% ===================================================
%    nested functions (callback processing)
% ===================================================
function qwerty(src,evnt) %#ok
  
  
  
  
  
  
  
  
  
  
end % (nested) qwerty

end % Mainline end 

