function [Ireg,O_trans,Spacing,M,Bx,By,Bz,Fx,Fy,Fz] = register_volumes(Imoving,Istatic,Options)
% This function register_volumes is the most easy way to register two
% 3D images both rigid and nonrigidly.
%
% Features:
% - It can be used with 3D images from different type of scans or modalities.
% - It uses both a rigid transform and a nonrigid b-spline grid transform.
% - It uses grid refinement
% - It can be used with images of different sizes.
% - The function will automaticaly detect if the images can be registered
% with the sum of squared pixel distance (SSD), or when mutual information 
% must be used as image similarity measure.
%
% Note: First Compile the c-files with compile_c_files <enter>
%
% [Ireg,Grid,Spacing,M,Bx,By,Bz,Fx,Fy,Fz] = register_volumes(Imoving,Istatic,Options);
%
% Inputs,
%   Imoving : The image which will be registerd
%   Istatic : The image on which Imoving will be registered
%   Options : Registration options, see help below
%
% Outputs,
%   Ireg : The registered moving image
%   Grid: The b-spline controlpoints, can be used to transform another
%         3D image in the same way: I=bspline_transform(Grid,I,Spacing); 
%   Spacing: The uniform b-spline knot spacing 
%	M : The affine transformation matrix
%   Bx, By, Bz : The backwards transformation fields of the pixels in 
%       x,y and z direction seen from the  static image to the moving image.
%   Fx, Fy, Fz : The (approximated) forward transformation fields of the pixels in 
%       x, y and z direction seen from the moving image to the static image.
%       (See the function backwards2forwards)
%
% Options,
%   Options.Similarity: Similarity measure (error) used can be set to:
%               sd : Squared pixel distance
%               mi : Normalized (Local) mutual information 
%               d, gd, gc, cc, pi, ld : see image_difference.m.
%   Options.Registration: 
%               Rigid    : Translation, Rotation
%               Affine   : Translation, Rotation, Shear, Resize
%               NonRigid : B-spline grid based local registration
%               Both     : Nonrigid and Affine (Default)
%   Options.Penalty: Thin sheet of metal smoothness penalty, default 1e-5,
%               if set to zero the registration will take a shorter time, but
%               will give a more distorted transformation field.
%   Options.Interpolation: Linear (default) or Cubic, the final result is
%               always cubic interpolated.
%   Options.MaxRef : Maximum number of grid refinements steps (default 2).
%   Options.Grid: Initial B-spline controlpoints grid, if not defined is initalized
%               with an uniform grid. (Used for in example while registering
%               a number of movie frames)
%   Options.Spacing: Spacing of initial B-spline grid in pixels 1x3 [sx sy sz]
%                    sx, sy and sz must be powers of 2, to allow grid
%                    refinement.
%   Options.MaskMoving: Image which is transformed in the same way as Imoving and
%               is multiplied with the individual pixel errors
%               before calculation of the te total (mean) similarity
%               error. In case of measuring the mutual information similariy 
%               the zero mask pixels will be simply discarded.
%   Options.MaskStatic: Also a Mask but is used  for Istatic
%   Options.Verbose: Display Debug information 0,1 or 2 
%
% Corresponding points / landmarks Options,
%   Options.Points1: List N x 3 of landmarks x,y,z in Imoving image
%   Options.Points2: List N x 3 of landmarks x,y,z in Istatic image, in which
%                     every row correspond to the same row with landmarks
%                     in Points1.
%   Options.PStrength: List Nx1 with the error strength used between the
%                     corresponding points, (lower the strenght of the landmarks
%                     if less sure of point correspondence).
% Example,
%   % add needed function paths
%   functiondir=which('register_volumes.m');
%   addpath([functiondir(1:end-length('register_volumes.m')) '/low_level_examples'])
%
%   % Get the volume data
%   [Imoving,Istatic]=get_example_data;
%
%   % Register the images
%   Ireg = register_volumes(Imoving,Istatic);
%
%   % Show the results
%   showcs3(Imoving);
%   showcs3(Istatic);
%   showcs3(Ireg);
%
% Function is written by D.Kroon University of Twente (January 2010)

% add all needed function paths
add_function_paths;

% Disable warning
warning('off', 'MATLAB:maxNumCompThreads:Deprecated')

% Check for presence of needed functions
check_mexfiles();

% Process inputs
defaultoptions=struct('Similarity',[],'Registration','Both','Penalty',1e-5,'MaxRef',2,'Grid',[],'Spacing',[],'MaskMoving',[],'MaskStatic',[],'Verbose',2,'Points1',[],'Points2',[],'PStrength',[],'Interpolation','Linear','Scaling',[1 1 1]);
if(~exist('Options','var')), 
    Options=defaultoptions; 
else
    tags = fieldnames(defaultoptions);
    for i=1:length(tags)
         if(~isfield(Options,tags{i})),  Options.(tags{i})=defaultoptions.(tags{i}); end
    end
    if(length(tags)~=length(fieldnames(Options))), 
        warning('register_volumes:unknownoption','unknown options found');
    end
end

% Set parameters
type=Options.Similarity;
O_trans=Options.Grid; Spacing=Options.Spacing;
MASKmoving=Options.MaskMoving; MASKstatic=Options.MaskStatic;
Points1=Options.Points1; Points2=Options.Points2; PStrength=Options.PStrength;

% Start time measurement
if(Options.Verbose>0), tic; end

% Convert the input images to double(or keep single) with range 0..1
[Iclass,Imin,Imax,Imoving,Istatic]=images2doublesingle(Imoving,Istatic);

% Resize the moving image to fit the static image
[Istatic,Imoving,MASKmoving]=images2samesize(Istatic,Imoving,MASKmoving);

% Detect if the mutual information or pixel distance can be used as 
% similarity measure. By comparing the histograms.
if(isempty(type)), type=check_image_modalities(Imoving,Istatic,Options); end

% Register the moving image affine to the static image
if(~strcmpi(Options.Registration(1),'N'))
    M=affine_registration(O_trans,Spacing,Options,Imoving,Istatic,MASKmoving,MASKstatic,type,Points1,Points2,PStrength);
else M=[];
end

% Make the initial b-spline registration grid
[O_trans,Spacing,MaxItt]=Make_Initial_Grid(O_trans,Spacing,Options,Imoving,M);

% Register the moving image nonrigid to the static image
if(strcmpi(Options.Registration(1),'N')||strcmpi(Options.Registration(1),'B'))
    [O_trans,Spacing]=nonrigid_registration(O_trans,Spacing,Options,Imoving,Istatic,MASKmoving,MASKstatic,type,Points1,Points2,PStrength,MaxItt);
end

% Transform the input image with the found optimal grid.
if ( nargout<5 )
    Ireg=bspline_transform(O_trans,Imoving,Spacing,3);
else
    [Ireg,Bx,By,Bz]=bspline_transform(O_trans,Imoving,Spacing,3);
end

% Make the forward transformation fields from the backwards
if ( nargout>7 ), [Fx,Fy,Fz]=backwards2forwards(Bx,By,Bz);end

% Back to old image range
Ireg=Back2OldRange(Ireg,Iclass,Imin,Imax);

% End time measurement
if(Options.Verbose>0), toc, end

function add_function_paths()
% add all needed function paths
try
    functionname='register_volumes.m';
    functiondir=which(functionname);
    functiondir=functiondir(1:end-length(functionname));
    addpath([functiondir '/functions'])
	addpath([functiondir '/functions_affine'])
	addpath([functiondir '/functions_nonrigid'])
catch
    me
    disp(me.message);
end

function check_mexfiles()
if(exist('bspline_transform_3d_double','file')~=3)
    error('bspline_transform_3d_double mex function not found, compile the c-file');
end
if(exist('bspline_transform_3d_single','file')~=3)
    error('bspline_transform_3d_single mex function not found, compile the c-file');
end
if(exist('affine_transform_3d_double','file')~=3)
    error('affine_transform_3d_double mex function not found, compile the c-file');
end
if(exist('affine_transform_3d_single','file')~=3)
    error('affine_transform_3d_single mex function not found, compile the c-file');
end
if(exist('mutual_histogram_double','file')~=3)
    error('mutual_histogram_double mex function not found, compile the c-file');
end
if(exist('mutual_histogram_single','file')~=3)
    error('mutual_histogram_single mex function not found, compile the c-file');
end

function type=check_image_modalities(Imoving,Istatic,Options)
% Detect if the mutual information or pixel distance can be used as 
% similarity measure. By comparing the histograms.
Hmoving= hist(Imoving(:),60)./numel(Imoving);
Hstatic = hist(Istatic(:),60)./numel(Istatic);
if(sum(log(abs(Hmoving-Hstatic)+1))>0.5), 
    type='mi'; 
    if(Options.Verbose>0), disp('Multi Modalities, Mutual information is used'); drawnow; end
else
    type='sd';
    if(Options.Verbose>0), disp('Same Modalities, Pixel Distance is used'); drawnow; end
end

function Ireg=Back2OldRange(Ireg,Iclass,Imin,Imax)
% Back to old image range
Ireg=Ireg*(Imax-Imin)+Imin;

% Set the class of output to input class
if(strcmpi(Iclass,'uint8')), Ireg=uint8(Ireg); end
if(strcmpi(Iclass,'uint16')), Ireg=uint16(Ireg); end
if(strcmpi(Iclass,'uint32')), Ireg=uint32(Ireg); end
if(strcmpi(Iclass,'int8')), Ireg=int8(Ireg); end
if(strcmpi(Iclass,'int16')), Ireg=int16(Ireg); end
if(strcmpi(Iclass,'int32')), Ireg=int32(Ireg); end

function [O_trans,Spacing,MaxItt]=Make_Initial_Grid(O_trans,Spacing,Options,Imoving,M)
if(isempty(O_trans)),
    if(isempty(Options.Spacing))
        % Calculate max refinements steps
        MaxItt=min(floor(log2(size(Imoving)/4)));

        % set b-spline grid spacing in x and y direction
        Spacing=[2^MaxItt 2^MaxItt 2^MaxItt];
    else
        % set b-spline grid spacing in x,y and z direction
        Spacing=round(Options.Spacing);
        t=Spacing; MaxItt=0; while((nnz(mod(t,2))==0)&&(nnz(t<8)==0)), MaxItt=MaxItt+1; t=t/2; end
    end
    % Make the Initial b-spline registration grid
    O_trans=make_init_grid(Spacing,size(Imoving),M);
else
    MaxItt=0;
    TestSpacing=Spacing;
    while(mod(TestSpacing,2)==0), TestSpacing=TestSpacing/2; MaxItt=MaxItt+1; end

    % Calculate center of the image
    mean=size(Imoving)/2;
    % Make center of the image coordinates 0,0
    xd=O_trans(:,:,:,1)-mean(1); yd=O_trans(:,:,:,2)-mean(2); zd=O_trans(:,:,:,3)-mean(3);
    % Calculate the rigid transformed coordinates
    O_trans(:,:,:,1) = mean(1) + M(1,1) * xd + M(1,2) *yd + M(1,3) *zd + M(1,4)* 1;
    O_trans(:,:,:,2) = mean(2) + M(2,1) * xd + M(2,2) *yd + M(2,3) *zd + M(2,4)* 1;
    O_trans(:,:,:,3) = mean(3) + M(3,1) * xd + M(3,2) *yd + M(3,3) *zd + M(3,4)* 1;
end

% Limit refinements steps to user input
if(Options.MaxRef<MaxItt), MaxItt=Options.MaxRef; end

function [O_trans,Spacing]=nonrigid_registration(O_trans,Spacing,Options,Imoving,Istatic,MASKmoving,MASKstatic,type,Points1,Points2,PStrength,MaxItt)
% Non-rigid b-spline grid registration
if(Options.Verbose>0), disp('Start non-rigid b-spline grid registration'); drawnow; end

if (Options.Verbose>0), disp(['Current Grid size : ' num2str(size(O_trans,1)) 'x' num2str(size(O_trans,2)) 'x' num2str(size(O_trans,3)) ]); drawnow; end

% set registration options.
options.type=type;
options.penaltypercentage=Options.Penalty;
options.interpolation=Options.Interpolation;
options.scaling=Options.Scaling;
options.verbose=false;
% Enable forward instead of central gradient incase of error measure is pixel distance
if(strcmpi(type,'sd')), options.centralgrad=false; end

% Reshape O_trans from a matrix to a vector.
sizes=size(O_trans); O_trans=O_trans(:);

% Make smooth images for fast registration without local minimums
Hsize=round(0.1667*(size(Istatic,1)/size(O_trans,1)+size(Istatic,2)/size(O_trans,2)+size(Istatic,3)/size(O_trans,3)));
ISmoving=imgaussian(Imoving,Hsize/5,[Hsize Hsize Hsize]);
ISstatic=imgaussian(Istatic,Hsize/5,[Hsize Hsize Hsize]);
resize_per=2^(MaxItt-1);

% Resize the mask to the image size used in the registration
if(~isempty(MASKmoving)), MASKmovingsmall=imresize3d(MASKmoving,1/resize_per);  else MASKmovingsmall=[]; end
if(~isempty(MASKstatic)), MASKstaticsmall=imresize3d(MASKstatic,1/resize_per);  else MASKstaticsmall=[]; end


% Optimizer parameters
optim=struct('GradObj','on','GoalsExactAchieve',0,'StoreN',10,'HessUpdate','lbfgs','Display','off','MaxIter',100,'DiffMinChange',0.03,'DiffMaxChange',1,'MaxFunEvals',1000,'TolX',0.05,'TolFun',1e-8);
if(Options.Verbose>0), optim.Display='iter'; end

% Start the b-spline nonrigid registration optimizer
Spacing_small=Spacing/resize_per;
ISmoving_small=imresize3d(ISmoving,1/resize_per,[],'linear');
ISstatic_small=imresize3d(ISstatic,1/resize_per,[],'linear');
Points1_small=Points1/resize_per;
Points2_small=Points2/resize_per;
PStrength_small=PStrength;
O_trans =  resize_per*fminlbfgs(@(x)bspline_registration_gradient(x,sizes,Spacing_small,ISmoving_small,ISstatic_small,options,MASKmovingsmall,MASKstaticsmall,Points1_small,Points2_small,PStrength_small),O_trans/resize_per,optim);

% Reshape O_trans from a vector to a matrix
O_trans=reshape(O_trans,sizes);

for refine_itt=1:MaxItt
    if (Options.Verbose>0), disp('Registration Refinement'); drawnow; end

    % Refine the b-spline grid
    [O_trans,Spacing]=refine_grid(O_trans,Spacing,size(Imoving)); 

    % Make smooth images for fast registration without local minimums
    Hsize=round(0.1667*(size(ISmoving,1)/size(O_trans,1)+size(ISstatic,2)/size(O_trans,2)+size(ISstatic,3)/size(O_trans,3)));
    ISmoving=imgaussian(Imoving,Hsize/5,[Hsize Hsize Hsize]);
    ISstatic=imgaussian(Istatic,Hsize/5,[Hsize Hsize Hsize]);
    resize_per=2^(MaxItt-1-refine_itt);

    % No smoothing in last registration step
    if(refine_itt==MaxItt), ISmoving=Imoving; ISstatic=Istatic; optim.TolX=0.05; resize_per=1; end

    if (Options.Verbose>0), disp(['Current Grid size : ' num2str(size(O_trans,1)) 'x' num2str(size(O_trans,2)) 'x' num2str(size(O_trans,3)) ]); drawnow; end

    % Reshape O_trans from a matrix to a vector.
    sizes=size(O_trans); O_trans=O_trans(:);

    % Resize the mask to the image size used in the registration
    if(~isempty(MASKmoving)), MASKmovingsmall=imresize3d(MASKmoving,1/resize_per);  else MASKmovingsmall=[]; end
    if(~isempty(MASKstatic)), MASKstaticsmall=imresize3d(MASKstatic,1/resize_per);  else MASKstaticsmall=[]; end

    % Start the b-spline nonrigid registration optimizer
    Spacing_small=Spacing/resize_per;
    ISmoving_small=imresize3d(ISmoving,1/resize_per,[],'linear');
    ISstatic_small=imresize3d(ISstatic,1/resize_per,[],'linear');
    Points1_small=Points1/resize_per;
    Points2_small=Points2/resize_per;
    PStrength_small=PStrength;
    O_trans =  resize_per*fminlbfgs(@(x)bspline_registration_gradient(x,sizes,Spacing_small,ISmoving_small,ISstatic_small,options,MASKmovingsmall,MASKstaticsmall,Points1_small,Points2_small,PStrength_small),O_trans/resize_per,optim);

    % Reshape O_trans from a vector to a matrix
    O_trans=reshape(O_trans,sizes);
end


function M=affine_registration(O_trans,Spacing,Options,Imoving,Istatic,MASKmoving,MASKstatic,type,Points1,Points2,PStrength)
% Smooth both images for faster registration
ISmoving=imgaussian(Imoving,2.5,[10 10 10]);
ISstatic=imgaussian(Istatic,2.5,[10 10 10]);

% Affine register the smoothed images to get the registration parameters
if(strcmpi(Options.Registration(1),'R'))
    if(Options.Verbose>0), disp('Start Rigid registration'); drawnow; end
    % Parameter scaling of the Translation and Rotation
    scale=[1 1 1 1 1 1];
    % Set initial rigid parameters
    x=[0 0 0 0 0 0];
elseif(strcmpi(Options.Registration(1),'A'))
    if(Options.Verbose>0), disp('Start Affine registration'); drawnow; end
    % Parameter scaling of the Translation, Rotation, Resize and Shear
    scale=[1 1 1 0.01 0.01 0.01 0.01 0.01 0.01 0.01 0.01 0.01 0.01 0.01 0.01];
    % Set initial rigid parameters
    x=[0 0 0 0 0 0 100 100 100 0 0 0 0 0 0];
elseif(strcmpi(Options.Registration(1),'B'))
    if(Options.Verbose>0), disp('Start Affine part of Non-Rigid registration'); drawnow; end
    % Parameter scaling of the Translation, Rotation, Resize and Shear
    scale=[1 1 1 0.01 0.01 0.01 0.01 0.01 0.01 0.01 0.01 0.01 0.01 0.01 0.01];
    % Set initial rigid parameters
    x=[0 0 0 0 0 0 100 100 100 0 0 0 0 0 0];
else
     warning('register_volumes:unknownoption','unknown registration method');    
end

if(Options.Interpolation(1)=='L'), interpolation_mode=0; else interpolation_mode=2; end

% Register Affine with 3 scale spaces
for refine_itt=1:3
    if(refine_itt==1)
        ITmoving=imresize3d(ISmoving,0.25);
        ITstatic=imresize3d(ISstatic,0.25);
        if(~isempty(MASKmoving)), ITMASKmoving = imresize3d(MASKmoving,0.25); else ITMASKmoving=[]; end
        if(~isempty(MASKstatic)), ITMASKstatic = imresize3d(MASKstatic,0.25); else ITMASKstatic=[]; end
        Points1t=Points1*0.25; Points2t=Points2*0.25; PStrengtht=PStrength;
    elseif(refine_itt==2)
        x(1:3)=x(1:3)*2;
        ITmoving=imresize3d(ISmoving,0.5);
        ITstatic=imresize3d(ISstatic,0.5);
        if(~isempty(MASKmoving)), ITMASKmoving = imresize3d(MASKmoving,0.5); else ITMASKmoving=[]; end
        if(~isempty(MASKstatic)), ITMASKstatic = imresize3d(MASKstatic,0.5); else ITMASKstatic=[]; end
        Points1t=Points1*0.5; Points2t=Points2*0.5; PStrengtht=PStrength;
    elseif(refine_itt==3)
        x(1:3)=x(1:3)*2;
        ITmoving=Imoving; 
        ITstatic=Istatic;
        ITMASKmoving = MASKmoving;
        ITMASKstatic = MASKstatic;
        Points1t=Points1; Points2t=Points2; PStrengtht=PStrength;
    end
    % Minimizer parameters
    % Use struct because expanded optimset is part of the Optimization Toolbox.
    % OA note, 2012.02.10: original TolFun value is 1e-6.
    optim=struct('GradObj','on','GoalsExactAchieve',1,'Display','off','StoreN',10,'HessUpdate','lbfgs','MaxIter',100,'MaxFunEvals',1000,'TolFun',1e-8,'DiffMinChange',1e-3);
    if(Options.Verbose>0), optim.Display='iter'; end
	x=fminlbfgs(@(x)affine_registration_error(x,scale,ITmoving,ITstatic,type,O_trans,Spacing,ITMASKmoving,ITMASKstatic,Points1t,Points2t,PStrengtht,interpolation_mode),x,optim);            
end

% Scale the translation, resize and rotation parameters to the real values
x=x.*scale;

if(strcmpi(Options.Registration(1),'R'))
    % Make the rigid transformation matrix
    M=make_transformation_matrix(x(1:3),x(4:6));
else
    % Make the affine transformation matrix
    M=make_transformation_matrix(x(1:3),x(4:6),x(7:9),x(10:15));   
end

function [Iclass,Imin,Imax,Imoving,Istatic]=images2doublesingle(Imoving,Istatic)
% Store the class of the inputs
Iclass=class(Imoving);

% Convert uint8, uint32 etc. to single.
if(~strcmpi(Iclass,'single')&&~strcmpi(Iclass,'double'))
    Imoving=single(Imoving);
	Istatic=single(Istatic);
end
Imin=min(min(Istatic(:)),min(Imoving(:))); Imax=max(max(Istatic(:)),max(Istatic(:)));
Imoving=(Imoving-Imin)/(Imax-Imin);
Istatic=(Istatic-Imin)/(Imax-Imin);

function [Istatic,Imoving,MASKmoving]=images2samesize(Istatic,Imoving,MASKmoving)
% Resize the moving image to fit the static image
if(sum(size(Istatic)-size(Imoving))~=0)
    Imoving=imresize3d(Imoving,[],size(Istatic),'cubic');
	if(~isempty(MASKmoving))
		MASKmoving=imresize3d(MASKmoving,[],size(Istatic),'cubic');
	end
end
