####### MAIN SCRIPT --- POST-DISTURBANCE CARBON RECOVERY IN AMAZONIAN FORESTS ######

library(rstan)
rstan_options(auto_write = TRUE)
options(mc.cores = parallel::detectCores())

library(raster)
library(truncnorm)

#### OPENING DATA FROM THE DRYAD REPOSITORY ####
## temporary URL : http://datadryad.org/review?doi=doi:10.5061/dryad.rc279 ##
## files must be converted to csv format

## Primary metadata : cumulative ACS changes
prim_data <- read.csv("Primary metadata.csv")

## Sites' description 
site_info <- read.csv("sites_clim_soil.csv")
#!!! change precipitation value for La Chonta 
site_info$precipitation[site_info$site=="la_chonta"] <- 1580

## Plots information
plot_info <- read.csv("Plot info.csv")

# ## create new dataframe with all measurements 

# total number of measurements 
N <- dim(unique(prim_data[,c("site","plot","t")]))[1]

Adata <- prim_data[1:N,c("site","plot","t")]
Adata$cSg <- prim_data$cum_change[prim_data$stock=="cSg"]
Adata$cSl <- prim_data$cum_change[prim_data$stock=="cSl"]
Adata$cRr <- prim_data$cum_change[prim_data$stock=="cRr"]
Adata$cRg <- prim_data$cum_change[prim_data$stock=="cRg"]
Adata$cRl <- prim_data$cum_change[prim_data$stock=="cRl"]
Adata <- merge(Adata, site_info)
Adata <- merge(Adata, plot_info)


#### CREATING VARIABLES FOR INFERENCE ####

# number of sites
S <- length(unique(Adata$site))

# number of plots
P <- length(unique(paste(Adata$site,Adata$plot)))

# N->S index (size N) :  1 observation -> nb of corresponding site 
ns <- as.numeric(Adata$site)

# N->P index (size N) : 1 observation -> nb of corresponding plot 
np <- as.numeric(as.factor(paste(Adata$site,Adata$plot)))

# P->S index (size P) : 1 plot -> nb of corresponding site 
ps <- sapply(1:P, function(p){ns[np==p][1]})

# time vector
# 1 observation -> time since min. ACS
tvec <- Adata$t

## ACS changes
# survivors' cumulative ACS growth in MgC
cSg <- Adata$cSg
# survivors' cumulative ACS loss in MgC
cSl <- Adata$cSl
# new recruits' ACS
cRr <- Adata$cRr
# recruits' cumulative ACS growth in MgC
cRg <- Adata$cRg
# recruits' cumulative ACS loss in MgC
cRl <- Adata$cRl

# acsmin : disturbance post-minimum ACS (size N)
acsmin <- Adata$acsmin
# observations weight : proportionnal to plot area and sum(weight) = 1
weight <- Adata$plot_size/sum(Adata$plot_size)
  
# covar : matrix of 5 covariates (columns) per plot (rows), centred and scaled for all observations
acs0 <- tapply(Adata$acs_i, Adata$site, mean)[ns]
dacs <- ((Adata$acs_i-acs0)/acs0)*100
covar <- data.frame(acs0, dacs, Adata[,c("precipitation","seasonality","bulk_density")])
covar <- apply(covar, 2, scale)[!duplicated(covar),]
# loss : 6th covariate = % of ACS loss per plot
loss <- c( unique( scale( ( (Adata$acs_i-Adata$acsmin)/Adata$acs_i ))))
  
#### INFERENCE ####
### survivors ###
## opens .stan file that contains the whole model, written in stan language
surv <- stan(file="survivors_model.stan", chains=6, iter=5000)
traceplot(surv, pars=c("alphasg0", "betasg0", "betasl0", "lambdasgloss", "lambdaslloss", "lambdasg", "lambdasl"))

### recruits ###
recr <- stan(file="recruits_model.stan", chains=6, iter=5000)
traceplot(recr, pars=c("alpharr0","alpharg0","eta","betarr0", "betarg0", "betarl0", "lambdarrloss", "lambdargloss", "lambdarlloss", "lambdarr", "lambdarg", "lambdarl"))



#########       FIGURES       #########


#### FIG 1 -- concept grah + data ####

# max likelihood parameters for survivors
smaxlp <- which.max(rstan::extract(surv)$lp__)
max_alphasg0 <- rstan::extract(surv)$alphasg0[smaxlp]
max_betasg0 <- rstan::extract(surv)$betasg0[smaxlp]
max_alphasl0 <- max_alphasg0+median(plot_info$acsmin)
max_betasl0 <- rstan::extract(surv)$betasl0[smaxlp]

# max likelihood parameters for recruits
rmaxlp <- which.max(rstan::extract(recr)$lp__)
max_eta <- rstan::extract(recr)$eta[rmaxlp]
max_alpharr0 <- rstan::extract(recr)$alpharr0[rmaxlp]
max_betarr0 <- rstan::extract(recr)$betarr0[rmaxlp]
max_alpharg0 <- rstan::extract(recr)$alpharg0[rmaxlp]
max_betarg0 <- rstan::extract(recr)$betarg0[rmaxlp]
max_alpharl0 <- rstan::extract(recr)$alpharl0[rmaxlp]
max_betarl0 <- rstan::extract(recr)$betarl0[rmaxlp]

# annual ACS changes matrix (columns = 5 ACS changes, rows = N observations)
annual_changes <- do.call(rbind,sapply(1:P,function(p) {
  time <- tvec[np==p]
  cflux <- cbind(cSg,cSl,cRr,cRg,cRl)[np==p,]
  interval <- diff(c(0,sort(time)))
  if (length(time)>1) return(rbind(cflux[1,],apply(cflux,2,diff))/interval)
  else return(cflux/interval)
}))
names(annual_changes) <- c("Sg","Sl","Rr","Rg","Rl")
  
### Graph : Figure 1 
par(mfrow=c(1,1),mar=c(4.5,4.5,0.5,0.5))
## dots : annual fluxes (green = survivors, red & orange = recruits)
plot(annual_changes[,1]~tvec, col="#66CD0020", pch=16, ylim=c(-2.5,4), xlim=c(1,300), 
     log="x", ylab="ACS change (Mg C/ha/yr)", xlab="t (yr)")
points(annual_changes[,2]~tvec, col="#00640020", pch=16)
points(annual_changes[,3]~tvec, col="#FF8C0020", pch=16)
points(annual_changes[,4]~tvec, col="#B8860B20", pch=16)
points(annual_changes[,5]~tvec, col="#A52A2A20", pch=16)
## curves : dashed lines from 0 to 250 yrs, solid lines from 0 to 30 yrs
curve(max_alphasg0*max_betasg0*(exp(-max_betasg0*x)), add=T, col="#66CD00", lwd=3, lty=2)
curve(max_alphasg0*max_betasg0*(exp(-max_betasg0*x)), add=T, col="#66CD00", lwd=3, xlim=c(1,30))
curve(-max_alphasl0*max_betasl0*(exp(-max_betasl0*x)), add=T, col="#006400", lwd=3, lty=2)
curve(-max_alphasl0*max_betasl0*(exp(-max_betasl0*x)), add=T, col="#006400", lwd=3, xlim=c(1,30))
curve(max_alpharg0*(1-exp(-max_betarg0*x)), add=T, col="#FF8C00", lwd=3, lty=2)
curve(max_alpharg0*(1-exp(-max_betarg0*x)), add=T, col="#FF8C00", lwd=3, xlim=c(1,30))
curve(-(max_alpharg0+max_alpharr0)*(1-exp(-max_betarl0*x)), add=T, col="#B8860B", lwd=3, lty=2)
curve(-(max_alpharg0+max_alpharr0)*(1-exp(-max_betarl0*x)), add=T, col="#B8860B", lwd=3, xlim=c(1,30))
curve(max_alpharr0*(1+max_eta*exp(-max_betarr0*x)), add=T, col="#A52A2A", lwd=3, lty=2)
curve(max_alpharr0*(1+max_eta*exp(-max_betarr0*x)), add=T, col="#A52A2A", lwd=3, xlim=c(1,30))
abline(0,0)
legend("topright", c("Survivors' ACS growth", "Survivors' ACS loss","Recruits' ACS growth", "Recruits' ACS loss", "New recruits' ACS"), col=c("#66CD00","#006400","#FF8C00","#B8860B","#A52A2A"), lwd=3, bg="white")
text(x=rep(300,3), y=c(max_alpharr0,max_alpharg0,-(max_alpharg0+max_alpharr0)), pos=3, labels=c(expression(alpha^Rr, alpha^Rg, alpha^Rl)), col=c("#A52A2A","#FF8C00","#B8860B"))
arrows(x0=1,x1=30,y0=-2.5,length=0.1,code=3)
text(x=5,y=-2.5,pos=3,labels="Calibration interval")
arrows(x0=35,x1=300,y0=-2.5,length=0.1,code=3)
text(x=100,y=-2.5,pos=3,labels="Theoretical behavior")



#### FIG 2 -- lambda plot ####

# graphs titles 
titleSg <- expression(paste(beta[0]^Sg==0.021 )) 
titleSl <- expression(paste(beta[0]^{Sl}==0.007 )) 
titleRr <- expression(paste(beta[0]^Rr==0.101 )) 
titleRg <- expression(paste(beta[0]^Rg==0.042 )) 
titleRl <- expression(paste(beta[0]^Rl==0.012 )) 
labs <- rev(c("loss", "acs0", "dacs", "prec","seas","bd"))

(plotsg <- plot(surv, pars=c("lambdasgloss","lambdasg")) + ggtitle(titleSg) +
  scale_y_discrete(labels=labs) + 
  geom_vline(xintercept = 0, lwd=1.5) )
(plotsl <- plot(surv, pars=c("lambdaslloss","lambdasl")) + ggtitle(titleSl) +
  scale_y_discrete(labels=element_blank()) +
  geom_vline(xintercept = 0, lwd=1.5) )
(plotrr <- plot(recr, pars=c("lambdarrloss", "lambdarr")) + ggtitle(titleRr) + 
  scale_y_discrete(labels=element_blank()) +
  geom_vline(xintercept = 0, lwd=1.5) )
(plotrg <- plot(recr, pars=c("lambdargloss","lambdarg"), col="grey", fill="blue") + ggtitle(titleRg) + 
  scale_y_discrete(labels=element_blank()) +
  geom_vline(xintercept = 0, lwd=1.5))
(plotrl <- plot(recr, pars=c("lambdarlloss","lambdarl")) + ggtitle(titleRl) + 
  scale_y_discrete(labels=element_blank()) +
  geom_vline(xintercept = 0, lwd=1.5))

#### FIG 2 - Supp Figure 1 -- obs-fit graphs ####
## surv ##
## parameters ##
alphasgp <- rstan::extract(surv)$alphasgp
betasgp <- matrix(rep(rstan::extract(surv)$betasg0, P), ncol=P, byrow=F) + rstan::extract(surv)$lambdasgloss%*%t(loss) + rstan::extract(surv)$lambdasg%*%t(covar)
betaslp <- matrix(rep(rstan::extract(surv)$betasl0, P), ncol=P, byrow=F) + rstan::extract(surv)$lambdaslloss%*%t(loss) + rstan::extract(surv)$lambdasl%*%t(covar)
## fitted values ###
fittedsg <- t(sapply(1:N, function(i){
  (alphasgp[,np[i]])*(1-exp(-(betasgp[,np[i]])*tvec[i]))
}))
fittedsl <- t(sapply(1:N, function(i){
  -(alphasgp[,np[i]]+acsmin[i])*(1-exp(-(betaslp[,np[i]])*tvec[i]))
}))

## recr ##
### extract parameters chains ###
eta <- rstan::extract(recr)$eta
alpharrs <- rstan::extract(recr)$alpharrs 
alphargs <- rstan::extract(recr)$alphargs 
betarrp <- rstan::extract(recr)$betarrp
betargp <- rstan::extract(recr)$betargp
betarlp <- rstan::extract(recr)$betarlp
### fitted values ###
## Rr
fittedrr <- t(sapply(1:N, function(i){
  (alpharrs[,ns[i]])*(tvec[i] + eta*(1-exp(-(betarrp[,np[i]])*tvec[i]))/(betarrp[,np[i]]))
}))
## Rg
fittedrg <- t(sapply(1:N, function(i){
  (alphargs[,ns[i]])*(tvec[i] - (1-exp(-(betargp[,np[i]])*tvec[i]))/(betargp[,np[i]]))
}))
## Nl
fittedrl <- t(sapply(1:N, function(i){
  -(alpharrs[,ns[i]]+alphargs[,ns[i]])*(tvec[i] - (1-exp(-(betarlp[,np[i]])*tvec[i]))/(betarlp[,np[i]]))
}))

### graphs obs vs fit
par(mfrow=c(2,3), mar=c(3,3,0,0), oma=c(2,2,0.5,0.5))
colors <- rgb(red=1, green=0, blue=0, alpha=(weight+0.001)/max(weight+0.001)) 
## Sg
plot(rowMeans(fittedsg)~cSg, col=colors, pch=16, xlim=c(0,120), ylim=c(0,120))
legend("topleft", "cSg", bty = "n")
abline(a=0,b=1)
## Sl
plot(rowMeans(fittedsl)~cSl, col=colors, pch=16, xlim=c(-80,0), ylim=c(-80,0))
legend("topleft", "cSl", bty = "n")
abline(a=0,b=1)
plot.new()
## Rr
plot((rowMeans(fittedrr))~cRr, col=colors, pch=16, xlim=c(0,35), ylim=c(0,35))
legend("topleft", "cRr", bty = "n")
abline(a=0,b=1)
## Rg
plot(rowMeans(fittedrg)~cRg, col=colors, pch=16, xlim=c(0,30), ylim=c(0,30))
legend("topleft", "cRg", bty = "n")
abline(a=0,b=1)
## Rl
plot(rowMeans(fittedrl)~cRl, col=colors, pch=16, xlim=c(-20,0), ylim=c(-20,0))
legend("topleft", "cRl", bty = "n")
abline(a=0,b=1)
mtext("Fitted values", side=2, outer=T)
mtext("Observed values", side=1, outer=T)



###########################################
###############     MAPS    ############### 
###########################################


## info = raster brick, maps of 4 spatial covariates (centred and standardized): 
# 1. ACS; 2. precipitation; 3. seasonality; 4. bulk density
# it might take a while (a few hundreds MB to download)!
# before sourcing, open the "extract_maps.R" file and read it through to see 
# instructions (2 maps to download, marked with asterisks *** in the script)
source("extract_maps.R")


## extract 200 sets of parameters to create 200 maps

# randomly choose 200 runs
runs <- sample(1:length(rstan::extract(surv)$alphasg0),200)
## how much would be a 40% acs loss (standardized)?
n_loss <- ((Adata$acs_i-Adata$acsmin)/Adata$acs_i)*100 # vector of % of acs loss in TmFO plots
acsloss <- (40-mean(n_loss))/sd(n_loss)

# create 5 matrices (one per ACS change) of parameters values (columns) for the 200 runs (rows) 
## Sg ##
# lambda values (4 spatial covariates + loss)
lambda <- rstan::extract(surv)$lambdasg[runs,-2]
lloss <- rstan::extract(surv)$lambdasgloss[runs]
# beta0 values
beta0 <- rstan::extract(surv)$betasg0[runs]
# alpha_j values ~ N(alpha0, sd_alpha)
alpha0 <- rstan::extract(surv)$alphasg0[runs]
sd_a0 <- sqrt(rstan::extract(surv)$var_alpha[runs])
alphaj <- rtruncnorm(length(runs), a=0, mean=alpha0,sd=sd_a0) ## truncnorm : positive values
par_sg <- cbind(lambda, lloss, beta0, alphaj)

## Sl ##
# lambda values
lambda <- rstan::extract(surv)$lambdasl[runs,-2]
lloss <- rstan::extract(surv)$lambdaslloss[runs]
# beta0 [ Rm : alphaSl_j = - (alphaSg_j + acsmin ) ]
beta0 <- rstan::extract(surv)$betasl0[runs]
par_sl <- cbind(lambda, lloss, beta0, alphaj)

## Rr ##
lambda <- rstan::extract(recr)$lambdarr[runs,-2]
lloss <- rstan::extract(recr)$lambdarrloss[runs]
beta0 <- rstan::extract(recr)$betarr0[runs]
alpha0 <- rstan::extract(recr)$alpharr0[runs]
sd_a0 <- sqrt(rstan::extract(recr)$var_alpha[runs,1])
alpharri <- rtruncnorm(length(runs),mean=alpha0,sd=sd_a0,a=0) ## positive values
eta <- rstan::extract(recr)$eta[runs]
par_rr <- cbind(lambda, lloss, beta0, alpharri, eta)

## Rg ##
lambda <- rstan::extract(recr)$lambdarg[runs,-2]
lloss <- rstan::extract(recr)$lambdargloss[runs]
beta0 <- rstan::extract(recr)$betarg0[runs]
alpha0 <- rstan::extract(recr)$alpharg0[runs]
sd_a0 <- sqrt(rstan::extract(recr)$var_alpha[runs,2])
alphargi <- rtruncnorm(length(runs),mean=alpha0,sd=sd_a0,a=0)
par_rg <- cbind(lambda, lloss, beta0, alphargi)

## Rl ##
lambda <- rstan::extract(recr)$lambdarl[runs,-2]
lloss <- rstan::extract(recr)$lambdarlloss[runs]
beta0 <- rstan::extract(recr)$betarl0[runs]
alpharli <- -(alpharri + alphargi)
par_rl <- cbind(lambda, lloss, beta0, alpharli)


### create maps of 5 ACS changes over period=10 yr ###
period <- 10

csg_map <- brick(apply(par_sg, 1, function(coef){
  effect <- function(x) {
    a0 <- coef[7]
    beta <- coef[1]*x[[1]]+coef[2]*x[[2]]+coef[3]*x[[3]]+coef[4]*x[[4]] + coef[5]*acsloss + coef[6]
    csg <- a0*(1-exp(-beta*period))
    return(csg)
  }
  calc(var_sd, fun=effect)
}))

csl_map <- brick(apply(par_sl, 1, function(coef){
  effect <- function(x) {
    acs0 <- (x[[1]]*sd(Adata$acs_i) + (mean(Adata$acs_i)))
    a0 <- -(coef[7]+(1-0.4)*acs0)
    beta <- coef[1]*x[[1]]+coef[2]*x[[2]]+coef[3]*x[[3]]+coef[4]*x[[4]] + coef[5]*acsloss + coef[6]
    ### alpha0 for survivors mortality = alpha0sg + acsmin, with acsmin=(1-loss)*acs0
    csl <- a0*(1-exp(-beta*period))
    return(csl)
  }
  calc(var_sd, fun=effect)
}))

crr_map <- brick(apply(par_rr, 1, function(coef){
  effect <- function(x) {
    a0 <- coef[7]
    beta <- coef[1]*x[[1]]+coef[2]*x[[2]]+coef[3]*x[[3]]+coef[4]*x[[4]] + coef[5]*acsloss + coef[6]
    crr <- a0*(period+coef[8]*(1-exp(-beta*period))/beta)
    return(crr)
  }
  calc(var_sd, fun=effect)
}))

crg_map <- brick(apply(par_rg, 1, function(coef){
  effect <- function(x) {
    a0 <- coef[7]
    beta <- coef[1]*x[[1]]+coef[2]*x[[2]]+coef[3]*x[[3]]+coef[4]*x[[4]] + coef[5]*acsloss + coef[6]
    crg <- a0*(period-(1-exp(-beta*period))/beta)
    return(crg)
  }
  calc(var_sd, fun=effect)
}))

crl_map <- brick(apply(par_rl, 1, function(coef){
  effect <- function(x) {
    a0 <- coef[7]
    beta <- coef[1]*x[[1]]+coef[2]*x[[2]]+coef[3]*x[[3]]+coef[4]*x[[4]] + coef[5]*acsloss + coef[6]
    crl <- a0*(period-(1-exp(-beta*period))/beta)
    return(crl)
  }
  calc(var_sd, fun=effect)
}))


#### FIG 3 -- map cumulative ACS dynamics ####

## median maps ##
# each pixel = median pixel (over 200 runs)
mu_csg <- calc(csg_map,median)
mu_csl <- calc(csl_map,median)
mu_crr <- calc(crr_map,median)
mu_crg <- calc(crg_map,median)
mu_crl <- calc(crl_map,median)

cflux_maps <- brick(mu_csg, mu_csl, mu_crr, mu_crg, mu_crl)
names(cflux_maps) <- c("cSg","cSl","cRr","cRg","cRl")

## choose color palette ##
maxcfl <- cellStats(cflux_maps,stat="max", na.rm=T)
mincfl <- cellStats(cflux_maps,stat="min", na.rm=T)
paletteSg <- colorRampPalette(c("white","gold","chartreuse3","chartreuse4","darkgreen", "#000000"))(ceiling(maxcfl[1]))
paletteRr <- paletteSg[floor(mincfl[3]):maxcfl[3]]
paletteRg <- paletteSg[1:ceiling(maxcfl[4])]
paletteSl <- colorRampPalette(c("navy","dodgerblue2","cornflowerblue","white"))(-floor(mincfl[2]))
paletteRl <- paletteSl[(length(paletteSl)+floor(mincfl[5])):length(paletteSl)]

## color bar ##
par(mfrow=c(1,1), mar=c(0,0,0,0), oma=c(0,1,0,3))
plot.new()
r <- cflux_maps[[1]]
r[1] <- -maxcfl[2]
plot(r, col=c(paletteSl,paletteSg), legend.only=T,smallplot=c(0.45,0.55, 0.05,0.95),legend.args=list(text='Cumulative ACS change (MgC/ha)',side=2, font=2, line=1))

## map 3.a -- median cSg over 10 years ##
par(mfrow=c(1,1), oma=c(0,0,0,0), mar=c(0,0,0,0))
plot(cflux_maps[[1]], axes=FALSE, box=FALSE, interpolate=T, col=paletteSg,legend=F)
points(x=site_info$long, y=site_info$lat, pch=16)

## map 3.b -- median cSl over 10 years ##
par(mfrow=c(1,1), oma=c(0,0,0,0), mar=c(0,0,0,0))
cflux_maps[[2]][1] <- 0
plot(cflux_maps[[2]], axes=FALSE, box=FALSE, col=paletteSl,legend=F)
points(x=site_info$long, y=site_info$lat, pch=16)

## map 3.c --median cRr over 10 years ##
par(mfrow=c(1,1), oma=c(0,0,0,0), mar=c(0,0,0,0))
plot(cflux_maps[[3]], axes=FALSE, box=FALSE, col=paletteRr,legend=F)
points(x=site_info$long, y=site_info$lat, pch=16)

## map 3.d --median cRg over 10 years ##
par(mfrow=c(1,1), oma=c(0,0,0,0), mar=c(0,0,0,0))
plot(cflux_maps[[4]], axes=FALSE, box=FALSE, col=paletteRg,legend=F)
points(x=site_info$long, y=site_info$lat, pch=16)

## map 3.e --median cRl over 10 years ##
par(mfrow=c(1,1), oma=c(0,0,0,0), mar=c(0,0,0,0))
plot(cflux_maps[[5]], axes=FALSE, box=FALSE, col=paletteRl,legend=F)
points(x=site_info$long, y=site_info$lat, pch=16)



#### FIG 4 -- map total change ####

# net ACS change 
change_cum <- csg_map + csl_map + crr_map + crg_map + crl_map
# mean and coefficient of variation = mean/sd
mean_change <- calc(change_cum, mean)
sd_change <- calc(change_cum, sd)
cv_change <- sd_change/mean_change*100

### plot mean change and 4 rectangles (sample dynamics)
### areas to sample ###
GS <- extent(-54,-51,1,4)
WA <- extent(-70,-67,-2,1)
CA <- extent(-61,-58,-6,-3)
SA <- extent(-66,-63,-12,-9)
sample_areas <- list(GS, WA, CA, SA)

## fig 4.a : net ACS change (mean)
cols <- colorRampPalette(c("orange","yellow", "forestgreen", "darkgreen"))(100)
plot(mean_change, axes=FALSE, box=FALSE, col=cols)
plot(GS, add=T, col=2, lwd=2)
text("GS", x=-52.5, y=2.5, cex=0.65)
plot(WA, add=T, col=2,lwd=2)
text("NWA", x=-68.5, y=-0.5,cex=0.65)
plot(CA, add=T, col=2,lwd=2)
text("CA", x=-59.5, y=-4.5,cex=0.65)
plot(SA, add=T, col=2, lwd=2)
text("SA", x=-64.5,y=-10.5,cex=0.65)

## fig 4.b : coefficient of variation
par(mfrow=c(1,1), oma=c(0,0,0,0), mar=c(0,0,0,4))
r1 <- raster(ncol=100, nrow=100)
r1[1:101] <- 0:100
cols <- colorRampPalette(c("navy", "darkgreen", "chartreuse3","yellow","orange", "red", 'firebrick'))(101)
brks <- 0:100
plot(cv_change, axes=FALSE, interpolate=T, col=cols, box=FALSE, breaks=brks, legend=F)
plot(r1, legend.only=T, col=cols)



####    MAPS STATS    ####

# pearson's r correlation
layerStats(brick(mean_change, cflux_maps), stat='pearson', na.rm = T)

### mean per area ###
cflux_area_mean <- sapply(sample_areas, function(exte){
  cellStats(crop(brick(mean_change, cflux_maps),exte), stat='mean',na.rm=TRUE)
})
colnames(cflux_area_mean) <- c("GS", "WA", "CA", "SA")

### sd per area 
cflux_area_sd <- sapply(sample_areas, function(exte){
  cellStats(crop(brick(mean_change, cflux_maps),exte), stat='sd',na.rm=TRUE)
})
colnames(cflux_area_sd) <- c("GS", "WA", "CA", "SA")



#### FIG 5 -- Stack plots ####

# time period : from 1 to 200 yrs, evenly distributed on a log scale
period <- c(round(exp(seq(0,log(200),0.3)), 1),200)

## median maps of annual ACS changes for every t in the time period

### Sg ###
# map_sg : function to calculate raster of annual ACS changes for survivors growth (Sg)
map_sg <- function(param, t=10) { 
  brick(apply(param, 1, function(coef){
    effect <- function(x) {
      alpha <- coef[7]
      beta <- coef[1]*x[[1]]+coef[2]*x[[2]]+coef[3]*x[[3]]+coef[4]*x[[4]] + coef[5]*acsloss + coef[6]
      flux <- alpha*beta*exp(-1*beta*t)
      return(flux)}
    calc(var_sd, fun=effect)
  }))}
# brick of Sg for each time step in period
brick_sg_t <- sapply(period, function(tx) calc(map_sg(param=par_sg ,t=tx), median))

### Sl ###
map_sl <- function(param, t=10) { 
  brick(apply(param, 1, function(coef){
    effect <- function(x) {
      acs0 <- (x[[1]]*sd(Adata$acs_i) + (mean(Adata$acs_i)))
      alpha <- -(coef[7]+0.6*acs0)
      beta <- coef[1]*x[[1]]+coef[2]*x[[2]]+coef[3]*x[[3]]+coef[4]*x[[4]] + coef[5]*acsloss + coef[6]
      flux <- alpha*beta*exp(-1*beta*t)
      return(flux)}
    calc(var_sd, fun=effect)
  }))}
brick_sl_t <- sapply(period, function(tx) calc(map_sl(param=par_sl ,t=tx), median))

### Rr, Rg and Rl ###
map_r <- function(param, t=10) { 
  brick(apply(param, 1, function(coef){
    effect <- function(x) {
      alpha <- coef[7]
      eta <- -1
      if (length(coef)>7) { eta <- coef[8] }
      beta <- coef[1]*x[[1]]+coef[2]*x[[2]]+coef[3]*x[[3]]+coef[4]*x[[4]] + coef[5]*acsloss + coef[6]
      flux <- alpha*(1+eta*exp(-beta*t))
      return(flux) }
    calc(var_sd, fun=effect) }))}
brick_rr_t <- sapply(period, function(tx) calc(map_r(param=par_rr ,t=tx), median))
brick_rg_t <- sapply(period, function(tx) calc(map_r(param=par_rg ,t=tx), median))
brick_rl_t <- sapply(period, function(tx) calc(map_r(param=par_rl ,t=tx), median))

# list of 5 bricks
acs_changes_t <- list(brick_sg_t, brick_sl_t, brick_rr_t, brick_rg_t,brick_rl_t)
acs_changes_t <- lapply(acs_changes_t,brick)

# sample_areas : 4 areas (Guiana Shield, Northwerstern Amazonia, ...) previously defined
# sample_dyn = list of 4 matrices (one per sample area)
# with columns = annual ACS change / rows = t in period

sample_dyn <- lapply(sample_areas, function(A){
  XA <- sapply(acs_changes_t, function(Ch){
    Ch <- crop(Ch, A)
    sapply(1:length(period),function(tx){
      median(values(Ch[[tx]]), na.rm=T)
    })
  })
  colnames(XA) <- c("Sg", "Sl", "Rr", "Rg", "Rl")
  rownames(XA) <- paste(as.character(period), " yr")
  return(XA)
})
names(sample_dyn) <- c("Guiana shield", "Northwestern Amazonia", "Central Amazonia", "Southern Amazonia")

net_change <- sapply(sample_dyn, rowSums)


###    Graph    ###

par(mfrow=c(2,2), mar=c(0,0,0,0), oma=c(4.5,4.5,1,1))

# calibration period (outside : transparent fill)
calib <- which(period<30)

# make 4 plots, one for each area
sapply(c(1:4), function(i){
  X <- sample_dyn[[i]]
  plot(x=period, y=X[,1], log="x", ylim=c(-2.5,5.5),xaxt="n", yaxt="n", col="white")
  if (i>2){axis(1)}
  if (i%in%c(1,3)){axis(2)}
  legend('top', c("Sg","Sl","Rr","Rg","Rl"), ncol=5,
         text.width= 0.09,title=names(sample_dyn)[i], adj = 0.6,
         fill=c("#66CD00", "#006400", "#A52A2A", "#FF8C00","#B8860B"), bty="n")
  ## gain: + ##
  #sg
  polygon(y=c(rep(0, dim(X)[1]),rev(X[,1])), x=c(period, rev(period)), col="#66CD0070", border = NA)
  polygon(y=c(rep(0, length(calib)),rev(X[calib,1])), x=c(period[calib], rev(period[calib])), col="#66CD00", border = NA)
  # rg
  polygon(y=c(X[,1], rev(X[,1]+X[,4])), x=c(period, rev(period)), col="#FF8C0070", border = NA)
  polygon(y=c(X[calib,1], rev(X[calib,1]+X[calib,4])), x=c(period[calib], rev(period[calib])), col="#FF8C00", border = NA)
  # rr
  polygon(y=c(X[,1]+X[,4], rev(X[,1]+X[,4]+X[,3])), x=c(period, rev(period)), col="#A52A2A70", border = NA)
  polygon(y=c(X[calib,1]+X[calib,4], rev(X[calib,1]+X[calib,4]+X[calib,3])), x=c(period[calib], rev(period[calib])), col="#A52A2A", border = NA)
  ## loss : - ##
  # sl
  polygon(y=c(rep(0, dim(X)[1]), rev(X[,2])), x=c(period, rev(period)), col="#00640070", border=NA)
  polygon(y=c(rep(0, length(calib)), rev(X[calib,2])), x=c(period[calib], rev(period[calib])), col="#006400", border=NA)
  # rl
  polygon(y=c(X[,2], rev(X[,2]+X[,5])), x=c(period, rev(period)), col="#B8860B70", border=NA)
  polygon(y=c(X[calib,2], rev(X[calib,2]+X[calib,5])), x=c(period[calib], rev(period[calib])), col="#B8860B", border=NA)
  # net change
  lines(y=net_change[,i], x = period, col="white", lty=2)
  lines(y=net_change[calib,i], x = period[calib], col="white", lty=1, lwd=2)
  abline(0,0)
})
mtext(outer=TRUE, side=1, "Time (yr)" , line=2.5)
mtext(outer=TRUE, side=2, "ACS changes (MgC/ha/yr)", line=2.5)


