% LONGTERM.M Runs simulation for long-term growth via strategic switching

% See mixed.m for detailed documentation of parameters
switchConst = 1000; % Speed at which organisms switch modes
nGrowthConst = -1; % Nomadic growth is negative (slow decay)
cGrowthConst = 10; % Colonial growth is high if alleeCap < n2 << K
alleeCap = 1.001; % When n2 < alleeCap, colonial growth is negative
capFunc = @(y) 1 - y(1); % K drops when n2 is high, grows when n2 is low

% Uncomment the following two lines to bound the carrying capacity K
% capMax = 7.5; % Maxmium value that K can rise to
% capFunc = @(y)(1 - y(1))*(1 - y(2)/capMax); % Equation for bounded K

n0 = [0; 2; 5]; % Initial population sizes / carrying capacity
tStart = 0; % Starting time
tEnd = 100; % Ending time

% No switching at first
nSwitchLevel = 0; % Switch to nomadic mode if K < nSwitchLevel
cSwitchLevel = Inf; % Switch back to colonial if K > cSwitchLevel

% Store solutions and switch levels
tInt = linspace(tStart, tEnd, 10000);
nInt = [];
solx = [];
soly = [];
nSwitchHistory = [];
cSwitchHistory = [];

% Temporary initial values for each piece of solution
y0 = n0;

while tStart < tEnd
    % Parameterize function for use in ODE solver
    odeFunc = @(t, yCur) ... 
        mixed(t, yCur, switchConst, nSwitchLevel, cSwitchLevel, ...
        nGrowthConst, cGrowthConst, alleeCap, capFunc);

    % Stop solution right before switching
    eventsFunc = @(t, yCur) longtermEvents(t, yCur, 1.01);
    options = odeset('NonNegative', 1, 'Events', eventsFunc, ...
                     'RelTol',1e-9,'AbsTol',1e-9);

    sol = ode23(odeFunc, [tStart tEnd], y0, options);
    
    % Append solution to previous ones
    if isempty(sol.xe)
        tTerm = tEnd;
        solx = [solx sol.x];
        soly = [soly sol.y];
    else
        tTerm = sol.xe(end);
        solx = [solx sol.x(1:end-1)];
        soly = [soly sol.y(:,1:end-1)];
    end
    tNew = tInt(tInt >= tStart);
    if tTerm == tEnd
        tNew = tNew(tNew <= tEnd);
    else
        tNew = tNew(tNew < tTerm);
    end
    nNew = deval(sol, tNew);
    nInt = [nInt nNew];

    % Prepare to evaluate next piece
    if tTerm == tEnd
        break;
    end
    tStart = tTerm;
    y0 = sol.ye(:,end);

    % Adjust switch levels
    if sol.ie(end) == 1
        nSwitchHistory = [nSwitchHistory nSwitchLevel];
        nSwitchLevel = sol.ye(3,end) + eps;
        cSwitchHistory = [cSwitchHistory cSwitchLevel];
        cSwitchLevel = Inf;
    elseif sol.ie(end) == 2
        cSwitchLevel = sol.ye(3,end) - eps;
    end
    
end

plot(solx, soly);
xlabel('Time');
ylabel('Population size');
legend('n_1', ...
       'n_2', ...
       'K');
titleStr = 'Long-term growth';
if switchConst == 0
    subStr = sprintf('No switching, A = %0.3f', alleeCap);
else
    subStr = sprintf('Strategic switching, up to t=%0.0f', tEnd);
end
title({['\fontsize{12}', titleStr], ...
      ['\fontsize{10}\rm', subStr]});
   