clear all;
close all;

%%%By Alex Sigal, 22/09/2016

%%%The purpose of the simulation is to follow infection chains which either
%%%allow or not allow bacterial growth in dead cells before next pickup. 

%%%Simulation starts with a dead cell with an input number of intracellular Mtb.

%%%Assumes no growth in live cells, and a 1 day window between death and
%%%pickup

%%%% fixed parameters

%%%Death probabilities 1 day after pickup, measured from cells with single

%%%pickups

freqDeathUnderTen =0.0571; %%%prob(death) for pickups <10 Mtb after 1 day

freqDeathUnderOneHundred =0.5000; %%%prob(death) for pickups 10-100 Mtb after 1 day

freqDeathOverOneHundred = 0.9286; %%%prob(death) for pickups >100 Mtb after 1 day

%%% Time of simulation and doubling time

maxTime =60 ;%%%Time of simulation in days 

doublingTime = 1; %%%Mtb doubling time in days
slope = log(2)/doublingTime; %%%

%%%%variable parameters
    
pick = 1;  %%%Number of Mtb in a dead cell initially 

numberRuns =10000; %%% Number simulations run

numberRunsPlotting =50; %%%%number simulations plotted

thresholdStats = 1000; %%%find number of runs with this number or greater Mtb

%%%%Initialization

noPickupNextCycle =0; %%% If there is a cell death event and a delay, noPickupNextCycle =1, since dead cell cannot internalize bacteria

for iteration = 1:numberRuns;
   
amountBact = pick; %%%Initializes with the input number of Mtb. Will grow if number of Mtb per cell grows
 
for time = 1:maxTime;

%%%Probability of death (pDeath) determined by number of Mtb in dead cell in current cycle
if amountBact <10; 
    
    pDeath = freqDeathUnderTen;
    
elseif amountBact <100;

    pDeath = freqDeathUnderOneHundred;
    
else
        
     pDeath = freqDeathOverOneHundred;
    
end

%%%%Pickup event. Choose random number 0 to 1. If smaller than pDeath, cell dies.
chooseRand = rand;

    if chooseRand < pDeath && noPickupNextCycle ==0; %%%Pickup allowed and death occurs
 
        cellsDeadVect(time) = 1; %%%Adds a dead cell in a serial infection chain
        
        amountBact = amountBact*exp(slope); %%%New number of Mtb 
        
        amountBactVect(time) = amountBact; %%%vector of Mtb numbers through time
        
        noPickupNextCycle =1;  %%%Dead cell - no pickups of other cells
        
    else
        
        noPickupNextCycle =0;%%%No death occured, or cell dead for 1 day before it is picked up by next cell. 
        %%%Therefore noPickupNextCycle resets
        
        cellsDeadVect(time) = 0; %%%No dead cell added
        
        amountBactVect(time) = amountBact;  %%%Does not change
               
    end              
 
end

cellsDeadMatrix(iteration, :) = cellsDeadVect;

amountBactMatrix(iteration, :) = amountBactVect;

end

%%%%unrestricted growth with same doubling time
amountBactUnrestricted = pick; %%%Initializes with the input number of Mtb
for iteration2 = 1:maxTime;
   
    amountBactUnrestricted = amountBactUnrestricted*exp(slope);
    
    vectUnrestricted(iteration2) = amountBactUnrestricted;
    
end

%%%%Plot cumulative dead cells in each infection chain
figure1 = figure('InvertHardcopy','off','Color',[1 1 1]);%%%invert hardcopy keeps background black

yscaleInt = maxTime/6;
axes1 = axes('Parent',figure1,'Color',[0 0 0], 'LineWidth',2, 'FontSize',24, 'XTick',[0 10 20 30 40 50],...
    'YTick',[yscaleInt*2 yscaleInt*4 yscaleInt*6]);
view(axes1,[33 26]);
colormap(axes1,colorcube(100));
cmap = colormap;
set(gca,'YTickLabel',{'20', '40','60'});
xlim([1 50])
ylim([1 maxTime])
zlim([0 25])
hold(axes1,'all');  

for iteration = 1:numberRunsPlotting;
    
    numberCellsDeadVector = cumsum(cellsDeadMatrix(iteration, :));%%%Cumulative sum of dead cells in each infection chain
   
  plot3(iteration*ones(1,maxTime), 1:maxTime, numberCellsDeadVector, 'Color', cmap(iteration, :), 'LineWidth',2);
    
    hold on          

end


print -r600 -dtiff simulationLinesDeadCellsOneDay;

hold off;

%%%%Plot total number of Mtb in each infection chain
figure2 = figure('InvertHardcopy','off','Color',[1 1 1]);%%%invert hardcopy keeps background black
yscaleInt = maxTime/6;
axes2 = axes('Parent',figure2,'Color',[0 0 0], 'LineWidth',2, 'FontSize',24, 'XTick',[0 10 20 30 40 50],...
    'YTick',[yscaleInt*2 yscaleInt*4 yscaleInt*6]);
view(axes2,[33 26]);
colormap(axes2,colorcube(100));
cmap = colormap;
set(gca,'YTickLabel',{'20', '40','60'});
xlim([1 50])
ylim([1 maxTime])
zlim([0 1000])
hold(axes2,'all');  

for iteration = 1:numberRunsPlotting+1;
    
    
    if iteration == 1;
        
         plot3(iteration*ones(1,maxTime), 1:maxTime, vectUnrestricted, 'Color', [1 1 1], 'LineWidth',3);
        
    else
   numberBactVector = amountBactMatrix(iteration-1, :);
  plot3(iteration*ones(1,maxTime), 1:maxTime, numberBactVector, 'Color', cmap(iteration-1, :), 'LineWidth',2);
    
    hold on 
    
    end

end


%%%%stats

numAbove = 0;
for iteration = 1:numberRuns;
   numberBactVector = amountBactMatrix(iteration, :);
   aboveThresh = find (numberBactVector >thresholdStats);
   if numel(aboveThresh)>0;
       numAbove = numAbove+1;
   end
    
end

numAboveThresh = numAbove/numberRuns

print -r600 -dtiff simulationLinesBactOneDay;