# This code provides illustrative examples of the use of our model of RBCs and primaquine-induced haemolysis
# The main model is driven by the c code 

rm(list=ls())
source('setupmodel.R')  # load the functions you need to do the simulations

# Parameters of the compartmental model: these determine how the drug acts and how the body responds to the drug
# d: parameter from lysing equation (Equation 4)
# mid_hemat: haematocrit for which bone marrow RBC production is half of maximum
# hill: hill coefficient in sigmoid function (Equation 2)
# mid_transit: the haematocrit corresponding to the midpoint of Equation 2
# steady_hemat: steady state haematocrit (Hb* in Equation 3)
# max_production: max fold increase in RBC production in severe aneamia


# An example parametrisation
theta = list(d = 0.05,          
             mid_hemat = 30, 
             hill = 10, 
             mid_transit = 30, 
             steady_hemat=45, 
             max_production = 5)

#########################################################################################################################
## The following example simulates primaquine induced haemolysis that is similar to to experiments done by Alving (1954),
# e.g. 30mg of primaquine daily in Africain A- individuals (healthy volunteers).


# Parameters for the simulations
ndays = 30
nhours = ndays*24 # number of hours to run the simulation forward
drug_regime = rep(1, nhours) # this determines when exactly there is circulating primaquine 
# 1: primaquine present; 0: primaquine not present

## Some parameter initialistions:
# We need to specify the age distributions of RBCs 
# the easiest way to do this is to make everything uniform distributions (lambda is number of cells produced at steady state)
pps = initialise_params()
steady_dist_norm =   rep(pps$lambda, pps$T_m )
steady_dist_retics = rep(pps$lambda, pps$T_retic)
steady_dist_rbcs =   rep(pps$lambda, pps$T_RBC)
drug_memory = 0 # this is important for tracking the lag_effect

## Now run the simulation and save the results in a list
sim_results = sim_patient(theta = theta, nhours=nhours, initial_dist_normobl = steady_dist_norm, 
                          initial_dist_retics = steady_dist_retics, initial_dist_rbcs = steady_dist_rbcs, 
                          drug_regime = drug_regime, drug_memory = drug_memory, hemat_0 = theta$steady_hemat,
                          time_lag = 24)

# Lets plot what has happened to the reticulocyte count (% retics) and to the hematocrit over time
# as a result of the drug
par(mfrow=c(1,2), bty='n', las=1, cex.lab=1.5) # I want two plots side by side
plot(sim_results[[6]][seq(1,nhours, by=24)], type='l', col='black', xlab='days', ylab='hematocrit')
plot(sim_results[[7]][seq(1,nhours, by=24)], type='l', col='black', xlab='days', ylab='reticulocytes %')

# Now let's have a look at what the age-distributions of the three types of blood cells look
# like. These were uniform distributions at the start
## I've plotted the log base 10 of the number of cells, makes it easier to visualise
par(mfrow=c(2,2), bty='n')
plot(log(sim_results[[2]][seq(1,pps$T_m, by=24)],base=10), type='l', 
     ylab='log-number of cells', xlab='age in days', main='normoblasts')
plot(log(sim_results[[3]][seq(1,pps$T_retic, by=24)],base=10), type='l', 
     ylab='log-number of cells', xlab='age in days', main='reticulocytes')
plot(log(sim_results[[4]][seq(1,pps$T_RBC, by=24)],base=10), type='l', 
     ylab='log-number of cells', xlab='age in days', main='Mature RBCs')


## Here is an example of giving the drug once a week for 8 weeks (the variable drug_regime changes)
drug_regime = rep(c(rep(1, 24), rep(0, 24*6)), 8) 
nhours = 8*7*24
sim_results = sim_patient(theta = theta, nhours=nhours, initial_dist_normobl = steady_dist_norm, 
                          initial_dist_retics = steady_dist_retics, initial_dist_rbcs = steady_dist_rbcs, 
                          drug_regime = drug_regime, drug_memory = 0, hemat_0 = theta$steady_hemat,
                          time_lag = 24)

par(mfrow=c(1,2), bty='n', las=1)
plot(sim_results[[6]][seq(1,nhours, by=24)], type='l', col='black', xlab='days', ylab='hematocrit')
plot(sim_results[[7]][seq(1,nhours, by=24)], type='l', col='black', xlab='days', ylab='reticulocyte %')


