%"Physical limits of flow sensing in the left-right organizer"
%Rita R. Ferreira, Andrej Vilfan, Frank Jlicher, Willy Supatto, and Julien Vermot
%eLife 2017
%Supplementary data file

%Script using KVdata.mat file to display a sample figure of cilia
%distribution within a vesicle

% Willy Supatto 
% Ecole Polytechnique, Palaiseau, FRANCE
% willy.supatto@polytechnique.edu
% Creation: May 19, 2017


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%NOTES
%(eA,eL,eD): body plan reference frame (A:Anterior, L:Left, D:Dorsal)
%(ef,en,em): local reference frame in unit sphere (f:flow, n:normal from surface to center, m:meridional)
%See Figure 1 of the manuscript
%cilim vector is a unit vector representing the axis of rotation from base to tip

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%VesCase structure
%KVdata.mat contains a VesCase with the following structure
%   .name: list of case names
%         CASE #1: wt9-14s corresponding to 9- to 14-SS embryos (late stages)
%         CASE #2: wt8s corresponding to 8SS embryos (mid stage)
%         CASE #3: wt3s corresponding to 3SS embryos (early stage)
%   .vesiclenum: Vesicle number
%   .Ncilia: total number of cilia per vesicle
%   .EllipsoidAxisa: long axis of fitted ellipsoid
%   .EllipsoidAxisb: short axis of fitted ellipsoid
%   .R: radius of sphere after ellipsod to sphere transformation
%   .Original
%         .X: eA coordinate of cilia base position in original dataset
%         .Y: eL coordinate of cilia base position in original dataset
%         .Z: eD coordinate of cilia base position in original dataset
%   .Ellipsoid
%         .X: eA coordinate of cilia base projected position on fitted ellipsoid
%         .Y: eL coordinate of cilia base projected position on fitted ellipsoid
%         .Z: eD coordinate of cilia base projected position on fitted ellipsoid
%         .ef: coordinates of ef on fitted ellipsoid (eA,eL,eD)
%         .em: coordinates of em on fitted ellipsoid (eA,eL,eD)
%         .en: coordinates of en on fitted ellipsoid (eA,eL,eD)
%   .UnitSphere
%       .X: eA coordinate of cilia base position on unit sphere
%       .Y: eL coordinate of cilia base position on unit sphere
%       .Z: eD coordinate of cilia base position on unit sphere
%       .alpha: spherical coordinate (AP/LR plane) of cilia base on unit sphere
%       .beta: spherical coordinate (from eD axis) of cilia base on unit sphere
%       .U: eA coordinate of cilia vector in (eA,eL,eD)
%       .V: eL coordinate of cilia vector in (eA,eL,eD)
%       .W: eD coordinate of cilia vector in (eA,eL,eD)
%       .u: eA coordinate of cilia vector in (ef,en,em)
%       .v: eL coordinate of cilia vector in (ef,en,em)
%       .w: eD coordinate of cilia vector in (ef,en,em)
%       .theta: theta angle of cilia vector in (ef,en,em)
%       .phi: phi angle of cilia vector in (ef,en,em)
%       .ef: coordinates of ef in (eA,eL,eD)
%       .em: coordinates of em in (eA,eL,eD)
%       .en: coordinates of en in (eA,eL,eD)
%   .ProcessingQuality
%       .CiliaCategory:
%                   1 = motile + clear orientation
%                   2 = immotile + clear orientation
%                   3 = immotile + unclear orientation
%                   4 = unclear (motility & orientation)
%       .EllispoidFitRMS: RMS error of Ellpsoid fitting

clear all
close all
scrsz = get(0,'ScreenSize');

load('KVdata.mat');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Message box : display Number of vesicles, Number of cilia and Number of cilia in each Category (1-4) for each case
NCiliaTot=0;
NCat1Tot=0;
NCat2Tot=0;
NCat3Tot=0;
NCat4Tot=0;
NVesTot=0;

for k=1:3   
        for i=1:length(VesCase(k).Ncilia)
            a1(i)=sum(VesCase(k).ProcessingQuality(i).CiliaCategory==1);%Category 1 = clear orientation and motile cilia
            a2(i)=sum(VesCase(k).ProcessingQuality(i).CiliaCategory==2);%Category 2 = clear orientation and non-motile cilia
            a3(i)=sum(VesCase(k).ProcessingQuality(i).CiliaCategory==3);%Category 3 = unclear orientation and non-motile cilia
            a4(i)=sum(VesCase(k).ProcessingQuality(i).CiliaCategory==4);%Category 4 = unclear orientation and unclear motility
        end
        NCiliaTot=NCiliaTot+sum(VesCase(k).Ncilia);
        NCat1Tot=NCat1Tot+sum(a1);
        NCat2Tot=NCat2Tot+sum(a2);
        NCat3Tot=NCat3Tot+sum(a3);
        NCat4Tot=NCat4Tot+sum(a4);        
        NVesTot=NVesTot+length(VesCase(k).Ncilia);
        msgboxText{k} = [VesCase(k).name,': Nves=',num2str(length(VesCase(k).Ncilia)),' Ncilia=',num2str(sum(VesCase(k).Ncilia)),' NCat1=',num2str(sum(a1)),' NCat2=',num2str(sum(a2)),' NCat3=',num2str(sum(a3)),' NCat4=',num2str(sum(a4))];
        clear a
end
msgboxText{k+1} = ['TOTAL: Nves=',num2str(NVesTot),' Ncilia=',num2str(NCiliaTot),' NCat1=',num2str(NCat1Tot),' NCat2=',num2str(NCat2Tot),' NCat3=',num2str(NCat3Tot),' NCat4=',num2str(NCat4Tot)];
msgbox(msgboxText);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Sample Figure: Drawing cilia vectors in unit sphere (category 1 in black, 2 in green, 3 in reen and 4 in red)

figure(1)
k=1;%case number (here wt9-14s case)
i=5;% vesicle number (here, vesicle #1)
X=VesCase(k).UnitSphere(i).X;%cilia base coordinates in (eA,eL,eD)
Y=VesCase(k).UnitSphere(i).Y;
Z=VesCase(k).UnitSphere(i).Z;
u=VesCase(k).UnitSphere(i).U;%cilia vector coordinates in (eA,eL,eD)
v=VesCase(k).UnitSphere(i).V;
w=VesCase(k).UnitSphere(i).W;
CiliaCategory=VesCase(k).ProcessingQuality(i).CiliaCategory;

%Drawing cilia vectors
plot3(X,Y,Z,'ok');hold on;
scale=0.2;
quiver3(X(CiliaCategory==1),Y(CiliaCategory==1),Z(CiliaCategory==1),scale*u(CiliaCategory==1),scale*v(CiliaCategory==1),scale*w(CiliaCategory==1),0,'k','LineWidth',2);hold on;
quiver3(X(CiliaCategory==2),Y(CiliaCategory==2),Z(CiliaCategory==2),scale*u(CiliaCategory==2),scale*v(CiliaCategory==2),scale*w(CiliaCategory==2),0,'b','LineWidth',2);hold on;
quiver3(X(CiliaCategory==3),Y(CiliaCategory==3),Z(CiliaCategory==3),scale*u(CiliaCategory==3),scale*v(CiliaCategory==3),scale*w(CiliaCategory==3),0,'g','LineWidth',2);hold on;
quiver3(X(CiliaCategory==4),Y(CiliaCategory==4),Z(CiliaCategory==4),scale*u(CiliaCategory==4),scale*v(CiliaCategory==4),scale*w(CiliaCategory==4),0,'r','LineWidth',2);hold on;
view( 56, 24 );

%Drawing (eA, eL, eD) Reference frame
pos=[-1 -1 -1];R=1;
hold on; eA=quiver3(R*pos(1),R*pos(2),R*pos(3),R*0.25,0,0,0,'color',[0.7 0 0],'LineWidth',3);
hold on; eL=quiver3(R*pos(1),R*pos(2),R*pos(3),0,R*0.25,0,0,'color',[0 0.7 0],'LineWidth',3);
hold on; eD=quiver3(R*pos(1),R*pos(2),R*pos(3),0,0,R*0.25,0,'color',[0 0 0.7],'LineWidth',3);
text(R*pos(1)+R*1*0.4,R*pos(2),R*pos(3),'a','FontSize',12,'FontAngle','italic','FontWeight','bold')
text(R*pos(1),R*pos(2)+R*1*0.4,R*pos(3),'l','FontSize',12,'FontAngle','italic','FontWeight','bold')
text(R*pos(1),R*pos(2),R*pos(3)+R*1*0.4,'d','FontSize',12,'FontAngle','italic','FontWeight','bold')

title(['Cilia vector in unit sphere from ',VesCase(k).name, ' vesicle #',num2str(VesCase(k).vesiclenum(i)),' (',num2str(VesCase(k).Ncilia(i)),' cilia | 1=black 2=blue 3=green 4=red)'])
set(gca,'LineWidth',1,'FontSize',12,'FontWeight','bold');
set(gca,'XTick',[-1 0 1],'YTick',[-1 0 1],'ZTick',[-1 0 1])
axis equal; axis vis3d;
axis([-1.3 1.3 -1.3 1.3 -1.3 1.3]);box on;