%% LFP and MUA analysis of experiments of cortical inhibitory interneurons optogenetic stimulation
% Created by Valentina Pasquale in 2017, based upon a previous version
% by Fellin's lab (IIT).
% Contact: valentina.pasquale@iit.it.
%% Clean workspace and screen
clear all;
close all;
%% Load parameter file
% The user must select an .m file containing all the relevant parameters
% for the analysis.
[paramFileName,paramPathName] = uigetfile('*.m','Select the .m file containing analysis parameters:');
if strcmp(num2str(paramFileName),'0')          % halting case
    return
end
paramFullFileName = fullfile(paramPathName,paramFileName);
run(paramFullFileName);
%% Create output folder and files
[expPath,expName,expFileExt] = fileparts(dataFilename);
% A folder named after the parameter file is created in the experiment folder
% (the one containing the data file) and it will contain all the results
% NB: if you change the parameter filename, another folder will be created,
% if you don't change the parameter filename, all results will be
% overwritten.
outputName = paramFileName(1:end-2);
try
    mkdir(expPath,outputName);
catch ME
    error(ME.identifier,ME.message);
end
outputSubFolder = fullfile(expPath,outputName);
% In the same folder containing this code, you should have these templates ready.
% These are templates for the XLS file that will contain your results.
outputXls = fullfile(outputSubFolder,[expName,'_analysis.xlsx']);
if strcmp(type,'ChR2')
    [path,name] = fileparts(xlsChR2Template);
    try
        copyfile(xlsChR2Template,outputXls);
    catch ME
        error(ME.identifier,ME.message);
    end
else if strcmp(type,'Arch')
        [path,name] = fileparts(xlsArchTemplate);
        try
            copyfile(xlsArchTemplate,outputXls);
        catch ME
            error(ME.identifier,ME.message);
        end
    end
end
%% Load data
% In the same folder containing this code, you should have this function
% too (abfload).
X = abfload(dataFilename);
% X is a 3D matrix, size #samples x #channels x #sweeps
% channel 1: stim channel
% channel 2: recording channel LFP
% channel 3: recording channel MUA
%% Pre-processing
% ---> Prepare useful variables
nf = sampling_frequency/downsampling_factor;    % downsampled rate
if 1/nf > L     % the sampling period is larger that the time lapse chosen for spectrogram computation
    errordlg('The sampling period is larger that the time lapse chosen for spectrogram computation. Processing is aborted.','Error');
    return
end
fk = ((0:round((N-1)/2))*nf/N)';                % FFT frequency vector
frequencies = round([f_min_band1 f_min_band2 f_min_band3 f_min_band4
                     f_max_band1 f_max_band2 f_max_band3 f_max_band4]'.*N/nf+1);
f_ini = round(f_ini_s.*N/nf+1);
f_fin = round(f_fin_s.*N/nf+1);
SF = (1-shrinking_factor)/2;              % SF = 0 if shrinking_factor = 1
si = 1/sampling_frequency;                % sampling period
channel_number = size(X,2);               % number of channels
sample_number = size(X,1);                % number of samples per sweep
sweep_number = size(X,3);                 % number of sweeps
% time = si.*(0:1:sample_number-1);         % time vector (full sampling)
time = si.*(1:1:sample_number);         % time vector (full sampling)
% PSTH parameters
bin_duration_samples = round(bin_duration_s/si);
numBins = floor(sample_number/bin_duration_samples);
PSTHdata = [(1:numBins)' zeros(numBins,1)];
PSTHdata = [PSTHdata zeros(numBins,2*length(selected_sweep))];
% Spectrogram parameters
R_t = R*nf;               % R_t: block length, in number of samples
window = hamming(R_t);    % returns a Hamming window of length R_t                                                                   % window function of length R                                                                % N: frequency discretization
L_t = L*nf;               % L_t: time lapse between segments, in number of samples
noverlap = R_t-L_t;       % number of overlapping samples
% --> Downsampling (LFP channel and stim channel)
sample_number_dec = ceil(sample_number/downsampling_factor);
Z = zeros(sample_number_dec,length(selected_sweep));
% Z_stim = zeros(sample_number_dec,sweep_number);
for ss = 1:length(selected_sweep)   % only for the selected sweeps
    Z(:,ss) = decimate(X(:,recch_LFP,selected_sweep(ss)),downsampling_factor);
%     Z_stim(:,ss) = decimate(X(:,stimch,selected_sweep(ss)),downsampling_factor);
end
% t = (0:1:size(Z)-1).*(1/nf);    % time vector decimated
t = (1:1:size(Z)).*(1/nf);    % time vector decimated
% sample_number = size(Z,1); % now sample_number = sample_number_dec
%
% If the protocol is Arch, the light phase must be analyzed; if it is ChR2,
% the light phase is discarded.
if strcmp(type,'ChR2')
    lightPhaseAnalysis = 0;   
else if strcmp(type,'Arch')
        lightPhaseAnalysis = 1; % 1, analysis of the light phase is performed
    end
end  
if lightPhaseAnalysis
    %     PSDdata = zeros(length(fk),6*length(selected_sweep));
    PSDdata = fk;
    PSDdata = [PSDdata zeros(length(fk),3*length(selected_sweep))];
else
    %     PSDdata = zeros(length(fk),4*length(selected_sweep));
    PSDdata = fk;
    PSDdata = [PSDdata zeros(length(fk),2*length(selected_sweep))];
end
%% Processing
for ss = 1:length(selected_sweep)   % the same analysis is repeated on all selected sweeps
    curSweep = selected_sweep(ss);  % this is the number of the current selected sweep
    % --> Artefact detection
    stimChannel = X(:,1,curSweep);
    stimTrain = (abs([0;diff(stimChannel)])>stimChannelTh).*sign([0;diff(stimChannel)]);
    startSampleStim = find(stimTrain>0);
    endSampleStim = find(stimTrain<0);  
    startSampleStimDec = round(startSampleStim./downsampling_factor);
    endSampleStimDec = round(endSampleStim./downsampling_factor);
    stimTrainDec = zeros(sample_number_dec,1);
    stimTrainDec(startSampleStimDec)=1;
    stimTrainDec(endSampleStimDec)=-1;
    % --> Spectrogram computation
    [S,f,T,P] = spectrogram(Z(:,ss),window,noverlap,N,nf);
    % --> Windows definition (pre, stim and post)
    % This matrix is sized number of windows x 2; the windows to consider
    % are pre-light, light and post-light, as defined by the user.
    % The 1st column contains the start sample index, the 2nd column
    % contains the end sample index.
    % It is defined at each sweep, although it should be the same for all
    % sweeps (the light stimulation is given at the same time instant for
    % all sweeps).
    wins = [];
    if lightPhaseAnalysis~=0
        wins(1,:) = [startSampleStimDec-pre_stimulus_window_lenght*nf startSampleStimDec-1];
        wins(2,:) = [startSampleStimDec+artifact_window*nf endSampleStimDec-1];
        wins(3,:) = [endSampleStimDec+artifact_window*nf endSampleStimDec+(post_stimulus_window_lenght+artifact_window)*nf-1];
    else
        wins(1,:) = [startSampleStimDec-pre_stimulus_window_lenght*nf startSampleStimDec-1];
        wins(2,:) = [startSampleStimDec endSampleStimDec-1];
        wins(3,:) = [endSampleStimDec+artifact_window*nf endSampleStimDec+(post_stimulus_window_lenght+artifact_window)*nf-1];
    end
    % NB: if artifact_window > 0 (blanking is applied), the post stimulus
    % window is evaluated after the blanking
    if wins(1,1)<0  % the start sample of the pre-stimulus window is < 0
        errordlg('Pre-stimulus window is too long. Processing is aborted.','Error');
        return
    end
    if wins(3,2)>sample_number_dec  % the end sample of the post-stimulus window is > sample_number_dec
        errordlg('Post-stimulus window is too long. Processing is aborted.','Error');
        return        
    end
    %% 1st FIGURE: LFP channel with spectrogram (power spectral density)
    h1 = figure('Name','LFP CHANNEL WITH SPECTROGRAM');
    subplot(2,1,1)
    plot(t,Z(:,ss),'b',...
        t,stimTrainDec,'r');
    hold on
    if lightPhaseAnalysis
        line([wins(2,1)/nf wins(2,1)/nf],[min(Z(:,ss)) max(Z(:,ss))],'Color','k');
        line([wins(3,1)/nf wins(3,1)/nf],[min(Z(:,ss)) max(Z(:,ss))],'Color','k');
    else
        line([wins(3,1)/nf wins(3,1)/nf],[min(Z(:,ss)) max(Z(:,ss))],'Color','k');
    end
    axis([0 sample_number_dec/nf min(Z(:,ss)) max(Z(:,ss))])
    xlabel('Time [s]');
    ylabel('Voltage [mV]');
    title('LFP');
    subplot(2,1,2)
    imagesc(T,f(f_ini:f_fin),log10(P(f_ini:f_fin,:)));
    axis([0 sample_number_dec/nf -inf inf])
    ylabel('Frequency [Hz]');set(gca,'YDir','normal');
    xlabel('Time [s]');
    title ('Spectrogram (PSD)')
    colormap(jet); colorbar;
    %% 2nd FIGURE: LFP channel with spectrogram (zoom pre-, light and post-light phases)
    h2 = figure('Name','LFP CHANNEL WITH SPECTROGRAM: LIGHT EVOKED ACTIVITY');
    if lightPhaseAnalysis > 0   % the light phase is plotted
        subplot(2,3,1)
        plot(t(wins(1,1):wins(1,2)),Z(wins(1,1):wins(1,2),ss),'b');
        axis([wins(1,1)/nf,wins(1,2)/nf,min(Z(:,ss)),max(Z(:,ss))])       
        xlabel('Time [s]')
        ylabel('[mV]')
        title('PRE LIGHT PHASE')
        %
        subplot(2,3,2)
        plot(t(wins(2,1):wins(2,2)),Z(wins(2,1):wins(2,2),ss),'r');
        axis([wins(2,1)/nf,wins(2,2)/nf,min(Z(:,ss)),max(Z(:,ss))])        
        xlabel('Time [s]')
        ylabel('[mV]')
        title('LIGHT PHASE')
        %
        subplot(2,3,3)       
        plot(t(wins(3,1):wins(3,2)),Z(wins(3,1):wins(3,2),ss),'g');
        axis([wins(3,1)/nf,wins(3,2)/nf,min(Z(:,ss)),max(Z(:,ss))])
        xlabel('Time [s]')
        ylabel('[mV]')
        title('POST LIGHT PHASE')
        %
        subplot(2,3,4)
%         imagesc(T(unique(round((wins(1,1)-R_t/2+1:wins(1,2)-R_t/2+1)./(nf*L)))),...
%             f(f_ini:f_fin),log10(P(f_ini:f_fin,unique(round((wins(1,1)-R_t/2+1:wins(1,2)-R_t/2+1)./(nf*L))))));% /max(max(log10(abs(S(f_ini:f_fin,:))))));
        imagesc(T,f(f_ini:f_fin),log10(P(f_ini:f_fin,:)));% /max(max(log10(abs(S(f_ini:f_fin,:))))));
        axis([T(round(wins(1,1)-R_t/2+1./(nf*L))) T(round(wins(1,2)-R_t/2+1./(nf*L))) f(f_ini) f(f_fin)]);
        axis xy; colormap(jet);
        xlabel('Time [s]');
        ylabel('Frequency [Hz]');
        title('PSD')
        %
        subplot(2,3,5)
%         imagesc(T(unique(round((wins(2,1)-R_t/2+1:wins(2,2)-R_t/2+1)./(nf*L)))),...
%             f(f_ini:f_fin),log10(P(f_ini:f_fin,unique(round((wins(2,1)-R_t/2+1:wins(2,2)-R_t/2+1)./(nf*L))))));% /max(max(log10(abs(S(f_ini:f_fin,:))))));
        imagesc(T,f(f_ini:f_fin),log10(P(f_ini:f_fin,:)));% /max(max(log10(abs(S(f_ini:f_fin,:))))));
        axis([T(round(wins(2,1)-R_t/2+1./(nf*L))) T(round(wins(2,2)-R_t/2+1./(nf*L))) f(f_ini) f(f_fin)]);
        axis xy; colormap(jet);
        xlabel('Time [s]');
        ylabel('Frequency [Hz]');
        title('PSD')
        %
        subplot(2,3,6)
%         imagesc(T(unique(round((wins(3,1)-R_t/2+1:wins(3,2)-R_t/2+1)./(nf*L)))),...
%             f(f_ini:f_fin),log10(P(f_ini:f_fin,unique(round((wins(3,1)-R_t/2+1:wins(3,2)-R_t/2+1)./(nf*L))))));
        imagesc(T,f(f_ini:f_fin),log10(P(f_ini:f_fin,:)));% /max(max(log10(abs(S(f_ini:f_fin,:))))));
        axis([T(round(wins(3,1)-R_t/2+1./(nf*L))) T(round(wins(3,2)-R_t/2+1./(nf*L))) f(f_ini) f(f_fin)]);
        axis xy; colormap(jet);
        xlabel('Time [s]');
        ylabel('Frequency [Hz]');
        title('PSD')
    else                    % light phase is discarded
        subplot(2,2,1)
        plot(t(wins(1,1):wins(1,2)),Z(wins(1,1):wins(1,2),ss),'b');
        axis([wins(1,1)/nf,wins(1,2)/nf,min(Z(:,ss)),max(Z(:,ss))])       
        xlabel('Time [s]')
        ylabel('[mV]')
        title('PRE LIGHT PHASE')
        %
        subplot(2,2,2)     
        plot(t(wins(3,1):wins(3,2)),Z(wins(3,1):wins(3,2),ss),'g');
        axis([wins(3,1)/nf,wins(3,2)/nf,min(Z(:,ss)),max(Z(:,ss))])
        xlabel('Time [s]')
        ylabel('[mV]')
        title('POST LIGHT PHASE')
        %
        subplot(2,2,3)
        imagesc(T,f(f_ini:f_fin),log10(P(f_ini:f_fin,:)));
        axis([T(round(wins(1,1)-R_t/2+1./(nf*L))) T(round(wins(1,2)-R_t/2+1./(nf*L))) f(f_ini) f(f_fin)]);
%         imagesc(T(unique(round((wins(1,1)-R_t/2+1:wins(1,2)-R_t/2+1)./(nf*L)))),...
%             f(f_ini:f_fin),log10(P(f_ini:f_fin,unique(round((wins(1,1)-R_t/2+1:wins(1,2)-R_t/2+1)./(nf*L))))));% /max(max(log10(abs(S(f_ini:f_fin,:))))));
        axis xy; colormap(jet);
        xlabel('Time [s]');
        ylabel('Frequency [Hz]');
        title('PSD')
        %
        subplot(2,2,4)
        imagesc(T,f(f_ini:f_fin),log10(P(f_ini:f_fin,:)));
        axis([T(round(wins(3,1)-R_t/2+1./(nf*L))) T(round(wins(3,2)-R_t/2+1./(nf*L))) f(f_ini) f(f_fin)]);
%         imagesc(T(unique(round((wins(3,1)-R_t/2+1:wins(3,2)-R_t/2+1)./(nf*L)))),...
%             f(f_ini:f_fin),log10(P(f_ini:f_fin,unique(round((wins(3,1)-R_t/2+1:wins(3,2)-R_t/2+1)./(nf*L))))));
        axis xy; colormap(jet);
        xlabel('Time [s]');
        ylabel('Frequency [Hz]');
        title('PSD')
    end
    %% LFP SPECTRAL ANALYSIS
    if SF > 0   % if shrinking has been selected
        winsRestrict = [round(wins(:,1)+(wins(:,2)-wins(:,1))*SF) round(wins(:,2)-(wins(:,2)-wins(:,1))*SF)];
    else
        winsRestrict = wins;
    end
    % if shrinking has been selected, all windows are shorter than expected
    winDuration_sample = winsRestrict(:,2)-winsRestrict(:,1)+1;
    % PRE analysis
    win_welch_pre = round((winDuration_sample(1)/windowWelchFactor));
    % Initialize spectrum objects by using a Hamming window of defined
    % length
    HwelchSpectrumPre = spectrum.welch('Hamming',win_welch_pre,overlapPrc);
    % Compute PSD by using the spectrum objects
    Hpsd_pre = psd(HwelchSpectrumPre,Z(winsRestrict(1,1):winsRestrict(1,2),ss),'Fs',nf,'NFFT',N);
    P_pre = Hpsd_pre.Data;
    ff_pre = Hpsd_pre.Frequencies;
    % LIGHT analysis
    if lightPhaseAnalysis
        win_welch_li = round((winDuration_sample(2)/windowWelchFactor));
        HwelchSpectrumLi = spectrum.welch('Hamming',win_welch_li,overlapPrc);
        Hpsd_li = psd(HwelchSpectrumLi,Z(winsRestrict(2,1):winsRestrict(2,2),ss),'Fs',nf,'NFFT',N);
        P_li = Hpsd_li.Data;
        ff_li = Hpsd_li.Frequencies;
    end
    % POST analysis
    win_welch_post = round((winDuration_sample(3)/windowWelchFactor));
    HwelchSpectrumPost = spectrum.welch('Hamming',win_welch_post,overlapPrc);
    Hpsd_post = psd(HwelchSpectrumPost,Z(winsRestrict(3,1):winsRestrict(3,2),ss),'Fs',nf,'NFFT',N);
    P_post = Hpsd_post.Data;
    ff_post = Hpsd_post.Frequencies;
    % Previous code
    if relativePowerCompOnSpectrogram~=0
        winsRestrictOnSpec = round((winsRestrict-R_t/2+1)./(nf*L));
        % 21-06-2013: originally the normalization factor on the
        % pre-stimulus window was computed as a double sum over the rows
        % (freq) and columns (time) of S (now P), whereas the power content
        % in the different frequency bands was computed as a sum over the
        % rows (freq) and a mean over the columns (time windows). 
        % This mismatch has been corrected in the present version, where
        % also the normalization factor is computed as a sum over rows and
        % a mean over columns.
%         norm_pre_spec = sum(sum(abs(S(:,winsRestrictOnSpec(1,1):winsRestrictOnSpec(1,2)))));
%         norm_pre_spec = sum(sum(P(:,winsRestrictOnSpec(1,1):winsRestrictOnSpec(1,2)))); 
        norm_pre_spec = mean(sum(P(:,winsRestrictOnSpec(1,1):winsRestrictOnSpec(1,2))));
        for kk = 1:size(frequencies,1)
            %                 bPreSpec(kk) = (1/norm_pre_spec)*mean(sum(abs(S(frequencies(kk,1):frequencies(kk,2),winsRestrict(1,1):winsRestrict(1,2)))));
            bPreSpec(kk) = (1/norm_pre_spec)*mean(sum(P(frequencies(kk,1):frequencies(kk,2),winsRestrictOnSpec(1,1):winsRestrictOnSpec(1,2))));
            if lightPhaseAnalysis ~= 0
                %                     bLiSpec(kk) = (1/norm_pre_spec)*mean(sum(abs(S(frequencies(kk,1):frequencies(kk,2),winsRestrict(2,1):winsRestrict(2,2)))));
                bLiSpec(kk) = (1/norm_pre_spec)*mean(sum(P(frequencies(kk,1):frequencies(kk,2),winsRestrictOnSpec(2,1):winsRestrictOnSpec(2,2))));
            end
            %                 bPostSpec(kk) = (1/norm_pre_spec)*mean(sum(abs(S(frequencies(kk,1):frequencies(kk,2),winsRestrict(3,1):winsRestrict(3,2)))));
            bPostSpec(kk) = (1/norm_pre_spec)*mean(sum(P(frequencies(kk,1):frequencies(kk,2),winsRestrictOnSpec(3,1):winsRestrictOnSpec(3,2))));
        end
        %             %
        %             b1pre=(1/norm_pre_spec)*mean(sum(abs(S(frequencies(1,1):frequencies(1,2),winsRestrict(1,1):winsRestrict(1,2)))));
        %             b2pre=(1/norm_pre_spec)*mean(sum(abs(S(frequencies(2,1):frequencies(2,2),winsRestrict(1,1):winsRestrict(1,2)))));
        %             b3pre=(1/norm_pre_spec)*mean(sum(abs(S(frequencies(3,1):frequencies(3,2),winsRestrict(1,1):winsRestrict(1,2)))));
        %             b4pre=(1/norm_pre_spec)*mean(sum(abs(S(frequencies(4,1):frequencies(4,2),winsRestrict(1,1):winsRestrict(1,2)))));
        %
        %             b1li=(1/norm_pre_spec)*mean(sum(abs(S(frequencies(1,1):frequencies(1,2),winsRestrict(2,1):winsRestrict(2,2)))));
        %             b2li=(1/norm_pre_spec)*mean(sum(abs(S(frequencies(2,1):frequencies(2,2),winsRestrict(2,1):winsRestrict(2,2)))));
        %             b3li=(1/norm_pre_spec)*mean(sum(abs(S(frequencies(3,1):frequencies(3,2),winsRestrict(2,1):winsRestrict(2,2)))));
        %             b4li=(1/norm_pre_spec)*mean(sum(abs(S(frequencies(4,1):frequencies(4,2),winsRestrict(2,1):winsRestrict(2,2)))));
        %
        %             b1post=(1/norm_pre_spec)*mean(sum(abs(S(frequencies(1,1):frequencies(1,2),winsRestrict(3,1):winsRestrict(3,2)))));
        %             b2post=(1/norm_pre_spec)*mean(sum(abs(S(frequencies(2,1):frequencies(2,2),winsRestrict(3,1):winsRestrict(3,2)))));
        %             b3post=(1/norm_pre_spec)*mean(sum(abs(S(frequencies(3,1):frequencies(3,2),winsRestrict(3,1):winsRestrict(3,2)))));
        %             b4post=(1/norm_pre_spec)*mean(sum(abs(S(frequencies(4,1):frequencies(4,2),winsRestrict(3,1):winsRestrict(3,2)))));
        %             %
        if lightPhaseAnalysis~=0
            band1Spec = [bPreSpec(1),bLiSpec(1),bPostSpec(1)];
            band2Spec = [bPreSpec(2),bLiSpec(2),bPostSpec(2)];
            band3Spec = [bPreSpec(3),bLiSpec(3),bPostSpec(3)];
            band4Spec = [bPreSpec(4),bLiSpec(4),bPostSpec(4)];
        else
            band1Spec = [bPreSpec(1),bPostSpec(1)];
            band2Spec = [bPreSpec(2),bPostSpec(2)];
            band3Spec = [bPreSpec(3),bPostSpec(3)];
            band4Spec = [bPreSpec(4),bPostSpec(4)];
        end
    end
    % %%   
    % The normalization factor has been set to the total power in the
    % pre-stimulus window
    norm_welch = sum(P_pre);
    for ii = 1:size(frequencies,1)
        bPre(ii) = 1/norm_welch.*sum(P_pre(frequencies(ii,1):frequencies(ii,2)));
        if lightPhaseAnalysis
            bLi(ii) = 1/norm_welch.*sum(P_li(frequencies(ii,1):frequencies(ii,2)));
        end
        bPost(ii) = 1/norm_welch.*sum(P_post(frequencies(ii,1):frequencies(ii,2)));
    end
    if lightPhaseAnalysis
        band1 = [bPre(1),bLi(1),bPost(1)];
        band2 = [bPre(2),bLi(2),bPost(2)];
        band3 = [bPre(3),bLi(3),bPost(3)];
        band4 = [bPre(4),bLi(4),bPost(4)];  
        h3 = figure('Name','LFP SPECTRAL ANALYSIS');
        subplot(2,2,1),bar(band1,'b');
        title('Delta');
        ylabel('Relative PSD content');
        set(gca,'XTick',[1 2 3],'XTickLabel',{'Pre','Light','Post'});
        subplot(2,2,2),bar(band2,'r');
        title('Alpha');
        set(gca,'XTick',[1 2 3],'XTickLabel',{'Pre','Light','Post'});
        subplot(2,2,3),bar(band3,'g');
        title('Gamma low');
        set(gca,'XTick',[1 2 3],'XTickLabel',{'Pre','Light','Post'});
        subplot(2,2,4),bar(band4,'y');
        title('Gamma high');
        set(gca,'XTick',[1 2 3],'XTickLabel',{'Pre','Light','Post'});
        %
        h4 = figure('Name','WELCH PSD');
        semilogx(ff_pre, db(P_pre), '.-b');
        hold on
        semilogx(ff_li, db(P_li), '.-r');
        semilogx(ff_post, db(P_post), '.-g');
        axis([f_ini_s f_fin_s -inf inf]);
        xlabel('Frequency')
        ylabel('dB')
        title('PSD computed with Welch method')
        legend('pre','light','post')
    else
        band1 = [bPre(1),bPost(1)];
        band2 = [bPre(2),bPost(2)];
        band3 = [bPre(3),bPost(3)];
        band4 = [bPre(4),bPost(4)];       
        h3 = figure('Name','LFP SPECTRAL ANALYSIS');
        subplot(2,2,1),bar(band1,'b');
        title('Delta');
        ylabel('Relative PSD content');
        set(gca,'XTick',[1 2],'XTickLabel',{'Pre','Post'});
        subplot(2,2,2),bar(band2,'r');
        title('Alpha');
        set(gca,'XTick',[1 2],'XTickLabel',{'Pre','Post'});
        subplot(2,2,3),bar(band3,'g');
        title('Gamma low');
        set(gca,'XTick',[1 2],'XTickLabel',{'Pre','Post'});
        subplot(2,2,4),bar(band4,'y');
        title('Gamma high');
        set(gca,'XTick',[1 2],'XTickLabel',{'Pre','Post'});
        %
        h4 = figure('Name','WELCH PSD');
        semilogx(ff_pre, db(P_pre), '.-b');
        hold on
        semilogx(ff_post, db(P_post), '.-r');
        axis([f_ini_s f_fin_s -inf inf]);
        xlabel('Frequency')
        ylabel('dB')
        title('PSD computed with Welch method')
        legend('pre','post')
    end
    if relativePowerCompOnSpectrogram~=0
        if lightPhaseAnalysis
            h7 = figure('Name','LFP SPECTRAL ANALYSIS ON SPECTROGRAM');
            subplot(2,2,1),bar(band1Spec,'b');
            title('Delta');
            ylabel('Relative PSD content');
            set(gca,'XTick',[1 2 3],'XTickLabel',{'Pre','Light','Post'});
            subplot(2,2,2),bar(band2Spec,'r');
            title('Alpha');
            set(gca,'XTick',[1 2 3],'XTickLabel',{'Pre','Light','Post'});
            subplot(2,2,3),bar(band3Spec,'g');
            title('Gamma low');
            set(gca,'XTick',[1 2 3],'XTickLabel',{'Pre','Light','Post'});
            subplot(2,2,4),bar(band4Spec,'y');
            title('Gamma high');
            set(gca,'XTick',[1 2 3],'XTickLabel',{'Pre','Light','Post'});
        else
            h7 = figure('Name','LFP SPECTRAL ANALYSIS ON SPECTROGRAM');
            subplot(2,2,1),bar(band1Spec,'b');
            title('Delta');
            ylabel('Relative PSD content');
            set(gca,'XTick',[1 2],'XTickLabel',{'Pre','Post'});
            subplot(2,2,2),bar(band2Spec,'r');
            title('Alpha');
            set(gca,'XTick',[1 2],'XTickLabel',{'Pre','Post'});
            subplot(2,2,3),bar(band3Spec,'g');
            title('Gamma low');
            set(gca,'XTick',[1 2],'XTickLabel',{'Pre','Post'});
            subplot(2,2,4),bar(band4Spec,'y');
            title('Gamma high');
            set(gca,'XTick',[1 2],'XTickLabel',{'Pre','Post'});
        end
    end
    %% MUA SPIKE ANALYSIS
    MUdata = X(:,recch_MU,curSweep)+offset_value_mV;
    % If the threshold is positive, positive deflections are detected; if
    % it is negative, negative deflections are detected
    if threshold_MU_mV > 0
        threshMUdata = MUdata > threshold_MU_mV;
    else if threshold_MU_mV < 0
            threshMUdata = MUdata < threshold_MU_mV;
        end
    end
    threshMUdataDiff = [0;diff(threshMUdata(:,1))];
    peak_train = threshMUdataDiff>0;
    % binning the data to build PSTH
    blankWindow_samples = round(artifact_window/si);
    if artifact_removal~=0
        if lightPhaseAnalysis~=0
            peak_train(startSampleStim+(0:blankWindow_samples)) = 0;
            peak_train(endSampleStim+(0:blankWindow_samples)) = 0;
        else
            peak_train(endSampleStim+(0:blankWindow_samples)) = 0;
        end        
    end
    peak_train = peak_train(1:(numBins*bin_duration_samples));
    spikeCount = sum(reshape(peak_train,bin_duration_samples,numBins),1);
    psth = spikeCount';
    psthNorm = psth./bin_duration_s;
    timeBinned = (1:1:size(psth,1)).*bin_duration_s;
%     timeBinned = (0:1:size(psth,1)-1).*bin_duration_s;
    h5 = figure('Name','MUA CHANNEL AND PSTH');
    subplot(2,1,1),plot(time,MUdata);
    hold on
    plot(time(peak_train),MUdata(peak_train),'r*')
    plot(t,stimTrainDec,'r')
    line([0 size(MUdata,1)*si],[threshold_MU_mV threshold_MU_mV],'Color','k')
    if lightPhaseAnalysis
        line([wins(2,1)/nf wins(2,1)/nf],[-0.1,0.1],'Color','k');
        line([wins(3,1)/nf wins(3,1)/nf],[-0.1,0.1],'Color','k');
    else
        line([wins(3,1)/nf wins(3,1)/nf],[-0.1,0.1],'Color','k');
    end
    axis([0,time(end),-0.1,0.1])
    xlabel('Time [s]')
    ylabel('Voltage [mV]')
    title('MUA channel')
    subplot(2,1,2),bar(timeBinned,psth);
    axis([-inf,inf,-inf,inf])
    xlabel('Time [s]')
    ylabel('#spikes')
    title('PSTH')
    %% Computing statistics on MUA analysis
    winsRestrictUpsampled = round(winsRestrict/nf*sampling_frequency); % *downsampling_factor;
    nspikesPre = sum(peak_train(winsRestrictUpsampled(1,1):winsRestrictUpsampled(1,2)));
    if lightPhaseAnalysis
        nspikesLi = sum(peak_train(winsRestrictUpsampled(2,1):winsRestrictUpsampled(2,2)));
    end
    nspikesPost = sum(peak_train(winsRestrictUpsampled(3,1):winsRestrictUpsampled(3,2)));
    if lightPhaseAnalysis
        nspikes = [nspikesPre,nspikesLi,nspikesPost]';
        duration = winDuration_sample./nf;    
    else
        nspikes = [nspikesPre,nspikesPost]';
        duration = winDuration_sample([1;3])./nf;
    end
    FR = nspikes./duration;
    h6 = figure('name','MUA SPIKES ANALYSIS');
    if lightPhaseAnalysis
        subplot(2,1,1),bar(nspikes,'b') ;
        title('Spike number')
        ylabel('#spikes');
        set(gca,'XTick',[1 2 3],'XTickLabel',{'Pre','Light','Post'});
        subplot(2,1,2),bar(FR,'g') ;
        title('FR')
        ylabel('Firing rate [#spikes/s]');
        set(gca,'XTick',[1 2 3],'XTickLabel',{'Pre','Light','Post'});
    else
        subplot(2,1,1),bar(nspikes,'b') ;
        title('Spike number')
        ylabel('#spikes');
        set(gca,'XTick',[1 2],'XTickLabel',{'Pre','Post'});
        subplot(2,1,2),bar(FR,'g') ;
        title('FR')
        ylabel('Firing rate [#spikes/s]');
        set(gca,'XTick',[1 2],'XTickLabel',{'Pre','Post'});
    end
    %% SAVING DATA
    % MATLAB format: as many files as the number of selected sweeps
    outputMUAnalysis = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_MUAnalysis.mat']);   
    save(outputMUAnalysis,'MUdata','FR','duration','nspikes','numBins','peak_train','psth','psthNorm','stimChannel','stimTrain','time','timeBinned');
    outputLFPAnalysis = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_LFPSpectralAnalysis.mat']);
    save(outputLFPAnalysis,'S','T','P','f','ff_post','P_post','ff_pre','P_pre',...
        'Hpsd_post','Hpsd_pre','HwelchSpectrumPost','HwelchSpectrumPre',...
        'bPost','bPre','band1','band2','band3','band4','norm_welch','stimTrainDec',...
        'bPostSpec','bPreSpec','band1Spec','band2Spec','band3Spec','band4Spec','norm_pre_spec',...
        't','win_welch_pre','win_welch_post','wins','winsRestrict','winsRestrictUpsampled');
    if lightPhaseAnalysis
        save(outputLFPAnalysis,'bLi','bLiSpec','ff_li','P_li','Hpsd_li','HwelchSpectrumLi','win_welch_li','-append');
    end
    % Save figures
    if figSave
        fig1filenameFig = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_1_spectrogram.fig']);
        fig1filenameJpg = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_1_spectrogram.jpg']);
        saveas(h1,fig1filenameFig,'fig');
        saveas(h1,fig1filenameJpg,'jpg');
        fig2filenameFig = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_2_pre-light-post_spectrogram.fig']);
        fig2filenameJpg = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_2_pre-light-post_spectrogram.jpg']);
        saveas(h2,fig2filenameFig,'fig');
        saveas(h2,fig2filenameJpg,'jpg');
        fig3filenameFig = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_3_relativePowerHist.fig']);
        fig3filenameJpg = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_3_relativePowerHist.jpg']);
        saveas(h3,fig3filenameFig,'fig');
        saveas(h3,fig3filenameJpg,'jpg');
        fig4filenameFig = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_4_PSDWelch.fig']);
        fig4filenameJpg = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_4_PSDWelch.jpg']);
        saveas(h4,fig4filenameFig,'fig');
        saveas(h4,fig4filenameJpg,'jpg');
        fig5filenameFig = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_5_MUAdataPSTH.fig']);
        fig5filenameJpg = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_5_MUAdataPSTH.jpg']);
        saveas(h5,fig5filenameFig,'fig');
        saveas(h5,fig5filenameJpg,'jpg');
        fig6filenameFig = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_6_nSpikesFRHist.fig']);
        fig6filenameJpg = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_6_nSpikesFRHist.jpg']);
        saveas(h6,fig6filenameFig,'fig');
        saveas(h6,fig6filenameJpg,'jpg');
        if relativePowerCompOnSpectrogram~=0
            fig7filenameFig = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_7_relativePowerHistOnSpec.fig']);
            fig7filenameJpg = fullfile(outputSubFolder,[expName,'_',num2str(curSweep),'_7_relativePowerHistOnSpec.jpg']);
            saveas(h7,fig7filenameFig,'fig');
            saveas(h7,fig7filenameJpg,'jpg');
        end
    end
    if pausing
        fprintf('Paused: press any key to continue...\n');
        pause;
    end
    close all
    % XLS format: only one xls file, all sweeps together
    if lightPhaseAnalysis~=0
        xlswrite(outputXls,curSweep,'MUA Analysis Statistics',['A',num2str(ss+2),':A',num2str(ss+2)]);
        xlswrite(outputXls,duration','MUA Analysis Statistics',['B',num2str(ss+2),':D',num2str(ss+2)]);
        xlswrite(outputXls,nspikes','MUA Analysis Statistics',['E',num2str(ss+2),':G',num2str(ss+2)]);
        xlswrite(outputXls,FR','MUA Analysis Statistics',['H',num2str(ss+2),':J',num2str(ss+2)]);
        xlswrite(outputXls,curSweep,'LFP Spectral Analysis',['A',num2str(ss+2),':A',num2str(ss+2)]);
        xlswrite(outputXls,band1,'LFP Spectral Analysis',['B',num2str(ss+2),':D',num2str(ss+2)]);
        xlswrite(outputXls,band2,'LFP Spectral Analysis',['E',num2str(ss+2),':G',num2str(ss+2)]);
        xlswrite(outputXls,band3,'LFP Spectral Analysis',['H',num2str(ss+2),':J',num2str(ss+2)]);
        xlswrite(outputXls,band4,'LFP Spectral Analysis',['K',num2str(ss+2),':M',num2str(ss+2)]);
        if relativePowerCompOnSpectrogram~=0
            xlswrite(outputXls,curSweep,'LFP Spectral Analysis Spec',['A',num2str(ss+2),':A',num2str(ss+2)]);
            xlswrite(outputXls,band1Spec,'LFP Spectral Analysis Spec',['B',num2str(ss+2),':D',num2str(ss+2)]);
            xlswrite(outputXls,band2Spec,'LFP Spectral Analysis Spec',['E',num2str(ss+2),':G',num2str(ss+2)]);
            xlswrite(outputXls,band3Spec,'LFP Spectral Analysis Spec',['H',num2str(ss+2),':J',num2str(ss+2)]);
            xlswrite(outputXls,band4Spec,'LFP Spectral Analysis Spec',['K',num2str(ss+2),':M',num2str(ss+2)]);
        end
    else
        xlswrite(outputXls,curSweep,'MUA Analysis Statistics',['A',num2str(ss+2),':A',num2str(ss+2)]);
        xlswrite(outputXls,duration','MUA Analysis Statistics',['B',num2str(ss+2),':C',num2str(ss+2)]);
        xlswrite(outputXls,nspikes','MUA Analysis Statistics',['D',num2str(ss+2),':E',num2str(ss+2)]);
        xlswrite(outputXls,FR','MUA Analysis Statistics',['F',num2str(ss+2),':G',num2str(ss+2)]);
        xlswrite(outputXls,curSweep,'LFP Spectral Analysis',['A',num2str(ss+2),':A',num2str(ss+2)]);
        xlswrite(outputXls,band1,'LFP Spectral Analysis',['B',num2str(ss+2),':C',num2str(ss+2)]);
        xlswrite(outputXls,band2,'LFP Spectral Analysis',['D',num2str(ss+2),':E',num2str(ss+2)]);
        xlswrite(outputXls,band3,'LFP Spectral Analysis',['F',num2str(ss+2),':G',num2str(ss+2)]);
        xlswrite(outputXls,band4,'LFP Spectral Analysis',['H',num2str(ss+2),':I',num2str(ss+2)]);
        if relativePowerCompOnSpectrogram~=0
            xlswrite(outputXls,curSweep,'LFP Spectral Analysis Spec',['A',num2str(ss+2),':A',num2str(ss+2)]);
            xlswrite(outputXls,band1Spec,'LFP Spectral Analysis Spec',['B',num2str(ss+2),':C',num2str(ss+2)]);
            xlswrite(outputXls,band2Spec,'LFP Spectral Analysis Spec',['D',num2str(ss+2),':E',num2str(ss+2)]);
            xlswrite(outputXls,band3Spec,'LFP Spectral Analysis Spec',['F',num2str(ss+2),':G',num2str(ss+2)]);
            xlswrite(outputXls,band4Spec,'LFP Spectral Analysis Spec',['H',num2str(ss+2),':I',num2str(ss+2)]);
        end
    end
    % TXT format: all sweeps together
    PSTHdata(:,2*(ss-1)+(1:2)+2) = [psth psthNorm];
    if lightPhaseAnalysis~=0
        PSDdata(:,3*(ss-1)+(1:3)+1) = [P_pre P_li P_post];
    else
        PSDdata(:,2*(ss-1)+(1:2)+1) = [P_pre P_post];
    end
end
% Save data common to all sweeps
% outputRawData = fullfile(outputSubFolder,[expName,'_rawDATA.mat']);
% save(outputRawData,'X','Z');
% save(outputRawData,'X','Z','Z_stim');
% Adding saving of average PSTH over all sweeps (last column)
% NB: In the PSTH data text file, you will find:
% -->   1st col: bin numbers
% -->   2nd col: trigger signal (0 no light, 1 light)
% -->   odd columns starting from 3rd (3-5-7-9-etc.): spike count data for each analyzed sweep
% -->   even columns starting from 4th (4-6-8-10-etc.): firing rate data for each
%       analyzed sweep
% -->   last 2 columns: average spike count and average firing rate over
%       all analyzed sweeps
% NB: startSampleStim and endSampleStim are the same for all sweeps
startStimBin = round(startSampleStim./bin_duration_samples);
endStimBin = round(endSampleStim./bin_duration_samples);
if startStimBin < endStimBin
    PSTHdata(startStimBin+1:endStimBin,2)=ones(endStimBin-startStimBin,1);
else if startStimBin == endStimBin
        PSTHdata(startStimBin,2)=1;
    end    
end
avgSpikeCount = mean(PSTHdata(:,3:2:size(PSTHdata,2)),2);
avgPSTH = mean(PSTHdata(:,4:2:size(PSTHdata,2)),2);
PSTHdata = [PSTHdata, avgSpikeCount, avgPSTH];
outputPSTHdata = fullfile(outputSubFolder,[expName,'_PSTHdata.txt']);
save(outputPSTHdata,'PSTHdata','-ASCII');
outputPSDdata = fullfile(outputSubFolder,[expName,'_PSDdata.txt']);
save(outputPSDdata,'PSDdata','-ASCII');