function [outputFolder] = UP_DOWN_DET_analyzeJuxta(paramFullFileName)
% Function containing code to perform UP/DOWN state detection on local field potential recordings
% and analyze firing properties of simultaneous juxtasomal recordings of cortical interneurons.
% Written by Valentina Pasquale (2017)
% Contact: valentina.pasquale@iit.it

%% Clean workspace and screen
pinkColor = [244 194 194]./255; % RGB coordinates of pink color
nBins = 100;
%% Load parameter file
run(paramFullFileName);
[expPath,paramFileName] = fileparts(paramFullFileName);
% %% Input data folder
% [~,expName] = fileparts(dataFilename);
%% Create output folder and files
[~,expName,~] = fileparts(dataFilename);
outputFolder = [expPath,filesep,paramFileName,'_',datestr(now,'yyyymmdd_HHMM')];
try
    if ~isdir(outputFolder)
        mkdir(outputFolder);
    else
        rmdir(outputFolder,'s');
        mkdir(outputFolder);
    end
catch ME
    outputFolder = [];
    error(ME.identifier,ME.message);
    return
end
%% Load data and join 1st and 2nd trace of the same sweep in a unique variable, 
... then join sweeps of the same series in a unique variable
dataFilename = fullfile(expPath,[expName,'.mat']);
if selectWinFlag == true
    fullTrace = UP_DOWN_DET_organizeData(dataFilename,seriesNumbers,seriesSweeps);
else
    [fullTrace,seriesNumbers,~] = UP_DOWN_DET_organizeData(dataFilename);
end
if all(cellfun('isempty',fullTrace))    % if no selected series/sweeps are present, no analysis is performed
    return
end
fullTrace = fullTrace(~cellfun('isempty',fullTrace));
%% Determine the best parameters for the detection of UP & DOWN states from LFP signal for each series
... also saves S_comb for each series, for the detection of UP & DOWN states
nSeries = length(fullTrace);
thUP = zeros(nSeries,1);
thDOWN = zeros(nSeries,1);
S_comb = cell(1,nSeries);
for ii = 1:nSeries
    curSeries = fullTrace{ii};
    %% Pre-processing of LFP trace
    % Low-pass filtering of LFP
    LFPseries_mV = curSeries(:,3).*1e+3;    % 3rd column
    LFPdataFilt = filtfilt(bLow,aLow,LFPseries_mV); % LP filter, cut-off 500 Hz
    % Downsampling to 1 KHz
    nSamples = length(LFPdataFilt);
    nSamplesDec = nSamples/(sf/sfLFP);
    timeDec = (1/sfLFP).*(1:1:nSamplesDec);
    LFPdata_dec = decimate(LFPdataFilt,sf/sfLFP);
    if inverted
        LFPdata_dec = -LFPdata_dec;
    end
    [b_deltaFilt,a_deltaFilt] = ellip(2,0.1,40,[0.1 4]./sfLFP);
    deltaFiltLFP = filtfilt(b_deltaFilt,a_deltaFilt,LFPdata_dec);
    %% Filter LFP in different frequency bands
    LFPdataDecFilt = zeros(length(LFPdata_dec),size(fBands,1));
    for bb = 1:size(fBands,1)
        if fBands(bb,1) == 0 % lowpass
            [b_LFPBandFilt,a_LFPBandFilt] = ellip(2,0.1,40,fBands(bb,2)./sfLFP,'low');
        else    % bandpass
            [b_LFPBandFilt,a_LFPBandFilt] = ellip(2,0.1,40,fBands(bb,:)./sfLFP);
        end
        LFPdataDecFilt(:,bb) = filtfilt(b_LFPBandFilt,a_LFPBandFilt,LFPdata_dec);
    end
    %% For each frequency band, approximate L_t
    k = zeros(size(LFPdataDecFilt,1),size(fBands,1));
    phi = zeros(size(LFPdataDecFilt,1),size(fBands,1));
    L_t = zeros(size(LFPdataDecFilt,1),size(fBands,1));
    for bb = 1:size(fBands,1)
        [k(:,bb),phi(:,bb)] = UP_DOWN_DET_compInstPhaseAmpl(LFPdataDecFilt(:,bb),phaseMethod);
        phi(phi(:,bb)<=0,bb) = phi(phi(:,bb)<=0,bb)+2*pi; % rescale from 0 to 2*pi
        phi(:,bb) = round(phi(:,bb)./(2*pi/360)).*(2*pi/360); % round to 1 degree resolution
        L_t(:,bb) = cos(phi(:,bb)-theta(bb));
    end
    %% Compute k_high
    [b_high,a_high] = ellip(2,0.1,40,fHigh./sfLFP);
    LFPdataHigh = filtfilt(b_high,a_high,LFPdata_dec);
    [k_high,~] = UP_DOWN_DET_compInstPhaseAmpl(LFPdataHigh,'hilbert');
    %% Derive S_LFP
    K = k./repmat(k_high+sum(k,2),1,size(fBands,1));
    S_LFP = (sum(K.*L_t,2)+1)./2;
    %% Compute k_alphaBeta
    if combFlag==1
        [b_alphaBeta,a_alphaBeta] = ellip(2,0.1,40,fAlphaBeta./sfLFP);
        LFPdataAlphaBeta = filtfilt(b_alphaBeta,a_alphaBeta,LFPdata_dec);
        rmsLFPdata = rmsFilt(LFPdataAlphaBeta,win_samples);
        smoothLFPdata = smooth(rmsLFPdata,winSmooth_samples);
        S_alphaBeta = smoothLFPdata./max(smoothLFPdata);
        S_comb{ii} = (S_LFP+S_alphaBeta)./2;
    else
        S_comb{ii} = S_LFP;
    end
	% Compute thresholds by fitting S_comb distribution with GM
    options = statset('Display','final','MaxIter',500);
    try
        GMmodel = gmdistribution.fit(S_comb{ii},3,'Options',options);
    catch err
        keyboard;
        continue;
    end
    mu = GMmodel.mu;
    sigma = squeeze(GMmodel.Sigma);
    figure(1)
    [hAmpl,bins] = hist(S_comb{ii},nBins);
    bar(bins,hAmpl./length(S_comb{ii}).*100);
    ylabel('# occurrences')
    xlabel('S_{LFP}')
    hold all
    yFit = pdf(GMmodel,bins');
    plot(bins,yFit,'r');
    %% Compute thresholds for UP & DOWN state detection
    [~,idxUP] = max(mu);
    [~,idxDOWN] = min(mu);
    thUP(ii) = mu(idxUP)-nSTD_UP*sqrt(sigma(idxUP));
    thDOWN(ii) = mu(idxDOWN)+nSTD_DOWN*sqrt(sigma(idxDOWN));
    figure(1)
    line([thUP(ii) thUP(ii)],[0 max(yFit)],'LineStyle','--','LineWidth',1,'Color','c')
    line([thDOWN(ii) thDOWN(ii)],[0 max(yFit)],'LineStyle','--','LineWidth',1,'Color','g')
    legend('S_{LFP} PDF','GM fit','UP state th','DOWN state th');
    saveas(1,fullfile(outputFolder,[expName,'_',num2str(seriesNumbers(ii)),'_S_comb_distribution.fig']),'fig');
    close(1)
    %% Compute transition-triggered LFP average
    [UP_states_DET_full, DOWN_states_DET_full] = UP_DOWN_DET_detectStates(S_comb{ii}, thUP(ii), thDOWN(ii), stateIntervTh, stateDurTh);
    LFPtriggAvg_UP = zeros(halfTransTriggWin*2,size(UP_states_DET_full,1));
    UP_states_DET_full(UP_states_DET_full(:,1)<=halfTransTriggWin|UP_states_DET_full(:,1)>(length(LFPdata_dec)-halfTransTriggWin),:)=[];
    for uu = 1:size(UP_states_DET_full,1)
        LFPtriggAvg_UP(:,uu) = LFPdata_dec(UP_states_DET_full(uu,1)-halfTransTriggWin:UP_states_DET_full(uu,1)+halfTransTriggWin-1);
    end
    LFPtriggAvg_UP_mean = mean(LFPtriggAvg_UP,2);
    LFPtriggAvg_UP_std = std(LFPtriggAvg_UP,[],2);
    LFPtriggAvg_DOWN = zeros(halfTransTriggWin*2,size(DOWN_states_DET_full,1));
    DOWN_states_DET_full(DOWN_states_DET_full(:,1)<=halfTransTriggWin|DOWN_states_DET_full(:,1)>(length(LFPdata_dec)-halfTransTriggWin),:)=[];
    for uu = 1:size(DOWN_states_DET_full,1)
        LFPtriggAvg_DOWN(:,uu) = LFPdata_dec(DOWN_states_DET_full(uu,1)-halfTransTriggWin:DOWN_states_DET_full(uu,1)+halfTransTriggWin-1);
    end
    LFPtriggAvg_DOWN_mean = mean(LFPtriggAvg_DOWN,2);
    LFPtriggAvg_DOWN_std = std(LFPtriggAvg_DOWN,[],2);
    timeWin = (-halfTransTriggWin:(halfTransTriggWin-1))./sfLFP;
    scrsz = get(0,'ScreenSize');
    h3 = figure('Position',[scrsz(3)/3 1 scrsz(3)/3 scrsz(4)]);
    subplot(3,1,1)
    shadedErrorBar(timeWin,LFPtriggAvg_UP_mean,LFPtriggAvg_UP_std,{'LineWidth',2,'Color','r'},1);
    line([0 0],ylim,'LineWidth',1,'Color','k')
    xlabel('Time [s]')
    ylabel('Voltage [mV]')
    h4 = figure('Position',[scrsz(3)/3 1 scrsz(3)/3 scrsz(4)]);
    subplot(3,1,1)
    shadedErrorBar(timeWin,LFPtriggAvg_DOWN_mean,LFPtriggAvg_DOWN_std,{'LineWidth',2,'Color','b'},1);
    line([0 0],ylim,'LineWidth',1,'Color','k')
    xlabel('Time [s]')
    ylabel('Voltage [mV]')
    %% Analyze separately each sweep
    spkCntData = [];
    LFPtriggAvgFR_UP = [];
    LFPtriggAvgFR_DOWN = [];
    raster_data_UP = [];
    raster_data_DOWN = [];
    for jj = 1:length(seriesSweeps{ii})
        curJuxtaRec = curSeries((1:sweepDur*sf)+(jj-1).*(sweepDur*sf),2);
        curLFPRec_delta = deltaFiltLFP((1:sweepDur*sfLFP)+(jj-1).*(sweepDur*sfLFP));
        curLFPRec = LFPdata_dec((1:sweepDur*sfLFP)+(jj-1).*(sweepDur*sfLFP));
        curS_comb = S_comb{ii}((1:sweepDur*sfLFP)+(jj-1).*(sweepDur*sfLFP));
        %% Detect UP and DOWN states from LFP trace
        [UP_states_DET, DOWN_states_DET] = UP_DOWN_DET_detectStates_wBound(curS_comb, thUP(ii), thDOWN(ii), stateIntervTh, stateDurTh);
        %% Detect spikes from juxtasomal trace
        [spikeTimes] = UP_DOWN_DET_detectSpikesJuxtasomal(curJuxtaRec,HPcutOff,sf,multCoeff);
        %% Plot result figure (for visualization)
        timeDec = (1:1:sweepDur*sfLFP)./sfLFP;
        time = (1:1:sweepDur*sf)./sf;
        figure(jj+10)
        % S_comb trace with UP & DOWN state detection threshold
        subplot(3,1,1)
        plot(timeDec,curS_comb) % S_comb trace
        line([0 sweepDur],[thUP(ii) thUP(ii)],'Color',pinkColor,'LineWidth',2)
        line([0 sweepDur],[thDOWN(ii) thDOWN(ii)],'Color','b','LineWidth',2)
        xlim([0 sweepDur])
        xlabel('Time [s]')
        ylabel('AU')
        title('Likelihood variable')
        subplot(3,1,2)
        % Filtered LFP (0.1-4 Hz) with indication of UP & DOWN states
        plot(timeDec,curLFPRec,'k','LineWidth',2)
        hold all
        if ~isempty(UP_states_DET)
            for uu = 1:size(UP_states_DET,1)
                plot(timeDec(UP_states_DET(uu,1):UP_states_DET(uu,2)),curLFPRec(UP_states_DET(uu,1):UP_states_DET(uu,2)),'Color',pinkColor,'LineWidth',2)
            end
        end
        if ~isempty(DOWN_states_DET)
            for dd = 1:size(DOWN_states_DET,1)
                plot(timeDec(DOWN_states_DET(dd,1):DOWN_states_DET(dd,2)),curLFPRec(DOWN_states_DET(dd,1):DOWN_states_DET(dd,2)),'Color','b','LineWidth',2)
            end
        end
        xlim([0 sweepDur])
        xlabel('Time [s]')
        ylabel('Voltage [V]')
        title('LFP')
        subplot(3,1,3)
        % Juxtasomal trace with indication of detected spikes and UP & DOWN
        % states        
        Wn_norm = HPcutOff./(0.5*sf);
        [bHP,aHP] = ellip(4,0.1,70,Wn_norm,'high');
        curJuxtaRecFilt = filtfilt(bHP,aHP,curJuxtaRec);      
        plot(time,curJuxtaRecFilt,'k')
        hold all
        if ~isempty(spikeTimes)
            plot(spikeTimes./sf,curJuxtaRecFilt(spikeTimes),'*r')
        end      
        hold all
        UP_states_DET_upSampl = [];
        if ~isempty(UP_states_DET)
            UP_states_DET_upSampl = UP_states_DET.*(sf/sfLFP);
            for uu = 1:size(UP_states_DET_upSampl,1)
                plot(time(UP_states_DET_upSampl(uu,1):UP_states_DET_upSampl(uu,2)),curJuxtaRecFilt(UP_states_DET_upSampl(uu,1):UP_states_DET_upSampl(uu,2)),'Color',pinkColor,'LineWidth',1)
            end
        end
        DOWN_states_DET_upSampl = [];
        if ~isempty(DOWN_states_DET)
            DOWN_states_DET_upSampl = DOWN_states_DET.*(sf/sfLFP);
            for dd = 1:size(DOWN_states_DET_upSampl,1)
                plot(time(DOWN_states_DET_upSampl(dd,1):DOWN_states_DET_upSampl(dd,2)),curJuxtaRecFilt(DOWN_states_DET_upSampl(dd,1):DOWN_states_DET_upSampl(dd,2)),'Color','b','LineWidth',1)
            end
        end
        xlim([0 sweepDur])
        xlabel('Time [s]')
        ylabel('Voltage [V]')
        if sweepSaveFlag == true
            saveas(jj+10,fullfile(outputFolder,[expName,'_',num2str(seriesNumbers(ii)),'_',num2str(seriesSweeps{ii}(jj)),'_results.tif']),'tiff');
			saveas(jj+10,fullfile(outputFolder,[expName,'_',num2str(seriesNumbers(ii)),'_',num2str(seriesSweeps{ii}(jj)),'_results.fig']),'fig');
            close(jj+10) 
        else
            pause;
            close(jj+10);
        end
        if ~isempty(spikeTimes) && ~isempty(UP_states_DET_upSampl) && ~isempty(DOWN_states_DET_upSampl)
            %% Computation of spike counts during UP/DOWN/IND states
            spikeTrain = zeros(length(time),1);
            spikeTrain(spikeTimes) = 1;
            num_UP_states = size(UP_states_DET_upSampl,1);
            num_DOWN_states = size(DOWN_states_DET_upSampl,1);
            UP_states_signal = convert2stateSignal(UP_states_DET_upSampl,length(time));
            DOWN_states_signal = convert2stateSignal(DOWN_states_DET_upSampl,length(time));
            numSpikes_TOT = length(spikeTimes);
            numSpikes_UP = nnz(spikeTrain & UP_states_signal);
            numSpikes_DOWN = nnz(spikeTrain & DOWN_states_signal);
            numSpikes_IND = numSpikes_TOT-(numSpikes_UP+numSpikes_DOWN);
            if num_UP_states~=0
                numSpikes_per_UP_state = numSpikes_UP./num_UP_states;
            else
                numSpikes_per_UP_state = 0;
            end
            if num_DOWN_states~=0
                numSpikes_per_DOWN_state = numSpikes_DOWN./num_DOWN_states;
            else
                numSpikes_per_DOWN_state = 0;
            end
            fr_UP_states = zeros(num_UP_states,1);
            num_act_UP_states = 0;
            for uu = 1:num_UP_states
               UP_state_dur = UP_states_DET_upSampl(uu,2)-UP_states_DET_upSampl(uu,1);
               fr_UP_states(uu) = nnz(spikeTimes>=UP_states_DET_upSampl(uu,1)&spikeTimes<=UP_states_DET_upSampl(uu,2))/(UP_state_dur./sf);
               if fr_UP_states(uu) ~= 0 
                   num_act_UP_states = num_act_UP_states+1;
               end
            end
            if num_act_UP_states~=0
                numSpikes_per_active_UP_state = numSpikes_UP./num_act_UP_states;
            else
                numSpikes_per_active_UP_state = 0;
            end
            if num_UP_states~=0
                fract_act_UP_states = num_act_UP_states./num_UP_states;
            else
                fract_act_UP_states = 0;
            end
            if isempty(nonzeros(fr_UP_states))
                avr_fr_UP_states = 0;
            else
                avr_fr_UP_states = mean(nonzeros(fr_UP_states));
            end
            fr_DOWN_states = zeros(num_DOWN_states,1);
            num_act_DOWN_states = 0;
            for uu = 1:num_DOWN_states
               DOWN_state_dur = DOWN_states_DET_upSampl(uu,2)-DOWN_states_DET_upSampl(uu,1);
               fr_DOWN_states(uu) = nnz(spikeTimes>=DOWN_states_DET_upSampl(uu,1)&spikeTimes<=DOWN_states_DET_upSampl(uu,2))/(DOWN_state_dur./sf);
               if fr_DOWN_states(uu) ~= 0 
                   num_act_DOWN_states = num_act_DOWN_states+1;
               end
            end
            if num_act_DOWN_states~=0
                numSpikes_per_active_DOWN_state = numSpikes_DOWN./num_act_DOWN_states;
            else
                numSpikes_per_active_DOWN_state = 0;
            end
            if num_DOWN_states~=0
                fract_act_DOWN_states = num_act_DOWN_states./num_DOWN_states;
            else
                fract_act_DOWN_states = 0;
            end
            if isempty(nonzeros(fr_DOWN_states))
                avr_fr_DOWN_states = 0;
            else
                avr_fr_DOWN_states = mean(nonzeros(fr_DOWN_states));
            end
            % 1st col: sweep number
            % 2nd-5th col: total spikes, UP spikes, DOWN spikes, IND spikes
            % 6th-8th col: percent of UP spikes, percent of DOWN spikes,
            % percent of IND spikes
            % 9th-10th col: average number of spikes per UP/DOWN state
            % 11th-12th col: average firing rate (spk/s) during UP/DOWN
            % 13th-14th col: fraction of active UP/DOWN states
            % 15th-16th col: average number of spikes per active UP/DOWN state
            % states
            % 17th-18th col: number of UP states, number of DOWN states
            spkCntData = [spkCntData; seriesSweeps{ii}(jj), numSpikes_TOT,...
                numSpikes_UP, numSpikes_DOWN, numSpikes_IND, ...
                numSpikes_UP/numSpikes_TOT, numSpikes_DOWN/numSpikes_TOT, numSpikes_IND/numSpikes_TOT, ...
                numSpikes_per_UP_state, numSpikes_per_DOWN_state, avr_fr_UP_states, avr_fr_DOWN_states, ...
                fract_act_UP_states, fract_act_DOWN_states, numSpikes_per_active_UP_state,...
                numSpikes_per_active_DOWN_state, num_UP_states, num_DOWN_states];
            %% Computation of transition-triggered average firing rate function
            halfTransTriggWin_long = halfTransTriggWin*(sf/sfLFP);
            UP_states_DET_upSampl(UP_states_DET_upSampl(:,1)<=halfTransTriggWin_long|UP_states_DET_upSampl(:,1)>(length(curLFPRec_delta)*sf/sfLFP-halfTransTriggWin_long),:)=[];
            for uu = 1:size(UP_states_DET_upSampl,1)
                spikeTrainTransTrigg = spikeTrain(UP_states_DET_upSampl(uu,1)-halfTransTriggWin_long:UP_states_DET_upSampl(uu,1)+halfTransTriggWin_long-1);
                raster_data_UP = [raster_data_UP; spikeTrainTransTrigg'];
                numSamples = length(spikeTrainTransTrigg);
                spkCnt = sum(reshape(spikeTrainTransTrigg(1:winSize*floor(numSamples/winSize)),winSize,floor(numSamples/winSize)),1);
                LFPtriggAvgFR_UP = [LFPtriggAvgFR_UP; spkCnt];
            end
            %
            DOWN_states_DET_upSampl(DOWN_states_DET_upSampl(:,1)<=halfTransTriggWin_long|DOWN_states_DET_upSampl(:,1)>(length(curLFPRec_delta)*sf/sfLFP-halfTransTriggWin_long),:)=[];
            for uu = 1:size(DOWN_states_DET_upSampl,1)
                spikeTrainTransTrigg = spikeTrain(DOWN_states_DET_upSampl(uu,1)-halfTransTriggWin_long:DOWN_states_DET_upSampl(uu,1)+halfTransTriggWin_long-1);
                raster_data_DOWN = [raster_data_DOWN; spikeTrainTransTrigg'];
                numSamples = length(spikeTrainTransTrigg);
                spkCnt = sum(reshape(spikeTrainTransTrigg(1:winSize*floor(numSamples/winSize)),winSize,floor(numSamples/winSize)),1);
                LFPtriggAvgFR_DOWN = [LFPtriggAvgFR_DOWN; spkCnt];
            end
        end
    end
    % last row is:
    % 1st col: number of considered sweeps
    % all other columns: average of all previous rows (average over all
    % sweeps)
    spkCntData = [spkCntData; size(spkCntData,1) mean(spkCntData(:,2:end),1)];
    save(fullfile(outputFolder,[expName,'_',num2str(seriesNumbers(ii)),'_spkCntResults.txt']),'spkCntData','-ASCII');																																 
    %% Plot transition triggered average FR function
    LFPtriggAvgFR_UP_mean = mean(LFPtriggAvgFR_UP,1);
    LFPtriggAvgFR_UP_stde = std(LFPtriggAvgFR_UP,[],1)./size(LFPtriggAvgFR_UP,1);
    LFPtriggAvgFR_UP_std = std(LFPtriggAvgFR_UP,[],1);
    LFPtriggAvgFR_DOWN_mean = mean(LFPtriggAvgFR_DOWN,1);
    LFPtriggAvgFR_DOWN_stde = std(LFPtriggAvgFR_DOWN,[],1)./size(LFPtriggAvgFR_DOWN,1);
    LFPtriggAvgFR_DOWN_std = std(LFPtriggAvgFR_DOWN,[],1);
    figure(h3)
    subplot(3,1,2)
    [y,x] = find(raster_data_UP);
    plot((x-halfTransTriggWin_long)./sf,y,'r.')
    xlim([min(timeWin) max(timeWin)])
    ylim([1 size(raster_data_UP,1)])
    line([0 0],ylim,'LineWidth',1,'Color','k')
    xlabel('Time [s]')
    ylabel('UP state no.')
    subplot(3,1,3)
    timeWin = (-halfTransTriggWin_long:winSize:(halfTransTriggWin_long-1))./sf;
    shadedErrorBar(timeWin,LFPtriggAvgFR_UP_mean,LFPtriggAvgFR_UP_std,{'LineWidth',2,'Color','r'},1);
    line([0 0],ylim,'LineWidth',1,'Color','k')
    xlim([min(timeWin) max(timeWin)])
    xlabel('Time [s]')
    ylabel('Spike count')
    title('Transition triggered spike count')
    saveas(h3,fullfile(outputFolder,[expName,'_',num2str(seriesNumbers(ii)),'_transTriggeredAFR_UP.fig']),'fig');
    close(h3)
    figure(h4)
    subplot(3,1,2)
    [y,x] = find(raster_data_DOWN);
    plot((x-halfTransTriggWin_long)./sf,y,'b.')
    xlim([min(timeWin) max(timeWin)])
    ylim([1 size(raster_data_DOWN,1)])
    line([0 0],ylim,'LineWidth',1,'Color','k')
    xlabel('Time [s]')
    ylabel('DOWN state no.')
    subplot(3,1,3)
    timeWin = (-halfTransTriggWin_long:winSize:(halfTransTriggWin_long-1))./sf;
    shadedErrorBar(timeWin,LFPtriggAvgFR_DOWN_mean,LFPtriggAvgFR_DOWN_std,{'LineWidth',2,'Color','b'},1);
    line([0 0],ylim,'LineWidth',1,'Color','k')
    xlim([min(timeWin) max(timeWin)])
    xlabel('Time [s]')
    ylabel('Spike count')
    title('Transition triggered spike count')
    saveas(h4,fullfile(outputFolder,[expName,'_',num2str(seriesNumbers(ii)),'_transTriggeredAFR_DOWN.fig']),'fig');
    close(h4)   
end