classdef DisplayWindow < siman.SimanObject
    % DisplayWindow
    properties
        DataSources
        Panels
    end
    properties (SetObservable, GetObservable)
        Color = [.8, .8, .8];
        Size = [800, 600];
        Location = [100, 100];
        
        ShowImageProp = false;
        ShowInventory = false;
        ShowProcessingDef = false;
        ShowCellDef = false;
        ShowROIGraph = false;
        ShowTransientDetection = false;
        ShowTransientList = false;
        ShowTransientGraph = false;
        ShowSparkDetection = false;
        ShowSparkList = false;
        ShowSparkGraph = false;
        ShowSparkletDetection = false;
        ShowSparkletList = false;
        ShowSparkletGraph = false;
        ShowSparkletSiteDetection = false;
        ShowSparkletSiteList = false;
        ShowSparkletSiteGraph = false;
        ShowContractionDetection = false;
        ShowContractionGraph = false;
    end
    properties (Transient)
        FigureID
        PlotPanel
        InventoryPanelID
        
        PlotPanelListener
        
        CurrentPlot
        ButtonDownPanel
        
        NeedsLayout = true;
    end
    properties (Dependent)
        CurrentROISource
    end
    
    events
        FigureResized
        MouseMotion
        ButtonClicked
        CurrentPlotChanged
    end
    
    methods % Properties
        function value = get.CurrentPlot(obj)
            value = [];
            if ~isempty(obj.PlotPanel)
                value = obj.PlotPanel.CurrentPlot;
            end
        end
        function value = get.CurrentROISource(obj)
            value = obj.CurrentPlot;
        end
    end
    
    methods
        % Constructor
        function obj = DisplayWindow()
            obj@siman.SimanObject();
            
            mgr = siman.ImageManager.GetManager(); % Initializes the object if it doesn't exists
            mgr.Init(); % Initialize if needed
            
            obj.Panels = siman.List;
            obj.Panels.DeleteElementsOnDelete = true;
            
            obj.InitCommands();
            obj.InitFigure();
            
            obj.SelfListener = addlistener(obj, 'PropertyChanged', @obj.OnPropertyChanged);
            mgr.RegisterDisplay(obj);
            
            %%obj.ApplyChanges();
        end
        
        function InitFigure(obj)
            loc = obj.Location;
            sz = obj.Size;
            pos = [loc sz];
            screenSize = get(0, 'ScreenSize');
            pos(1) = min(pos(1), screenSize(3)-pos(3)-1);
            pos(2) = min(pos(2), screenSize(4)-pos(4)-1);
            closeCallback = {@closed, obj};
            resizeCallback = {@resized, obj};
            motionCallback = {@mouseMoved, obj};
            buttonUpCallback = {@buttonUp, obj};
            buttonDownCallback = {@buttonDown, obj};
            scrollWheelCallback = {@scrollWheel, obj};
            keyCallback = {@keyPress, obj};
            obj.FigureID = figure('Name', 'Siman', 'NumberTitle', 'off', 'Color', obj.Color, ...
                'Units', 'pixels', 'OuterPosition', pos, ...
                'MenuBar', 'none', 'DockControls', 'off', ...
                'ResizeFcn', resizeCallback, ...
                'Renderer', 'zbuffer', ...
                'WindowButtonMotionFcn', motionCallback, ...
                'WindowButtonUpFcn', buttonUpCallback, ...
                'WindowButtonDownFcn', buttonDownCallback, ...
                'KeyPressFcn', keyCallback, ...
                'WindowScrollWheelFcn', scrollWheelCallback, ...
                'CloseRequestFcn', closeCallback);
            iptPointerManager(obj.FigureID);
            
            function resized(src, event, varargin)
                try
                    windowObj = varargin{1};
                    windowObj.OnFigureResized();
                    mgr = siman.ImageManager.GetManager();
                    mgr.CurrentDisplay = obj;
                    mgr.ApplyChanges();
                catch ex
                    siman.Debug.ReportError(ex, 'Error while resizing figure');
                end
            end
            function buttonUp(src, event, varargin)
                try
                    windowObj = varargin{1};
                    windowObj.OnButtonUp();
                    mgr = siman.ImageManager.GetManager();
                    mgr.CurrentDisplay = obj;
                    mgr.ApplyChanges();
                catch ex
                    siman.Debug.ReportError(ex, 'Error during button up');
                end
            end
            function buttonDown(src, event, varargin)
                try
                    windowObj = varargin{1};
                    windowObj.OnButtonDown();
                catch ex
                    siman.Debug.ReportError(ex, 'Error during button down');
                end
            end
            function mouseMoved(src, event, varargin)
                try
                    windowObj = varargin{1};
                    windowObj.OnMouseMotion();
                catch ex
                    siman.Debug.ReportError(ex, 'Error during mouse motion');
                end
            end
            function scrollWheel(src, event, varargin)
                try
                    windowObj = varargin{1};
                    windowObj.OnScrollWheelMoved();
                catch ex
                    siman.Debug.ReportError(ex, 'Error during scroll wheel moved');
                end
            end
            function keyPress(src, event, varargin)
                try
                    windowObj = varargin{1};
                    windowObj.OnKeyPress(event);
                catch ex
                    siman.Debug.ReportError(ex, ['Error on keypress: ' event.Key]);
                end
            end
            function closed(src, event, varargin)
                id = gcf;
                if ~isempty(id) && ishandle(id) && isvalid(obj)
                    set(id, 'Units', 'pixels');
                    pos = get(id, 'Position');
                    obj.Location = pos([1 2]);
                end
                if isvalid(obj)
                    delete(obj);
                end
                delete(gcf);
            end
            
            obj.PlotPanel = siman.PlotsPanel(obj);
            obj.PlotPanelListener = addlistener(obj.PlotPanel, 'ChangedCurrentPlot', @obj.OnCurrentPlotChanged);
            obj.Panels.Add(obj.PlotPanel);
            obj.InitMenu();
            
            list = {'ImageProp', 'Inventory', 'ProcessingDef', 'CellDef', 'ROIGraph', ...
                'TransientDetection', 'TransientList', 'TransientGraph', ...
                'SparkDetection', 'SparkList', 'SparkGraph', ...
                'SparkletDetection', 'SparkletList', 'SparkletGraph', ...
                'SparkletSiteDetection', 'SparkletSiteList', 'SparkletSiteGraph', ...
                'ContractionDetection', 'ContractionGraph'};
            for i = 1:numel(list)
                if obj.(['Show' list{i}])
                    obj.ShowPanel([ list{i} 'Panel']);
                end
            end
        end
                
        function enabling = ResetCommandEnabling(obj, cmd, context)
            if ~isempty(strfind(cmd.Name, 'Window.Show'))                
                prop = strrep(cmd.Name, 'Window.', '');
                if (~isprop(context, prop))
                    siman.Debug.ReportProblem(['Display - command to refresh has invalid panel name: ' cmd.Name]);
                    return;
                end
                enabling = true;
                return;
            end
            switch cmd.Name
                case {'File.SaveFullImageAs', 'File.SaveImageStackAs', 'File.SaveVisibleImageAs', 'File.SaveDataAs'}
                    list = obj.GetSelectedPlots();
                    enabling = list.Count > 0;
                case {'View.ZoomIn', 'View.ZoomOut', 'View.ShowColorbar'}
                    list = obj.GetSelectedPlots();
                    enabling = list.Count > 0;
                case {'View.LockPlot'}
                    list = obj.GetSelectedPlots();
                    enabling = list.Count > 0;
                case {'Edit.CopyFullImage', 'Edit.CopyVisibleImage', 'Edit.CopyData'}
                    list = obj.GetSelectedPlots();
                    enabling = list.Count > 0;
                case {'Tools.MakeMovie'}
                    list = obj.GetSelectedPlots();
                    enabling = list.Count > 0;
                    if (enabling)
                        plot = list.ElementAt(1);
                        enabling = plot.HasImageSeries;
                    end
                case {'Tools.ROI.AddROIType', 'Tools.ROI.AddROI'}
                    list = obj.GetSelectedPlots();
                    enabling = list.Count > 0;
                case {'Window.CloseAllToolPanels'}
                    enabling = obj.Panels.Count > 1;
                otherwise
                    siman.Debug.ReportProblem(['Display - unknown command to enable: ' cmd.Name]);
            end
        end
        
        function checked = ResetCommandChecked(obj, cmd, context)
            if ~isempty(strfind(cmd.Name, 'Window.Show'))                
                prop = strrep(cmd.Name, 'Window.', '');
                if (~isprop(context, prop))
                    siman.Debug.ReportProblem(['Display - command to refresh has invalid panel name: ' cmd.Name]);
                    return;
                end
                checked = obj.(prop);
                return;
            end
            switch cmd.Name
                case {'View.LockPlot'}
                    list = obj.GetSelectedPlots();
                    plot = list.ElementAt(1);
                    if isempty(plot)
                        checked = false;
                    else
                        checked = plot.IsLocked;
                    end
                case {'View.ShowColorbar'}
                    list = obj.GetSelectedPlots();
                    plot = list.ElementAt(1);
                    if isempty(plot)
                        checked = false;
                    else
                        checked = plot.ShowColorbar;
                    end
                otherwise
                    siman.Debug.ReportProblem(['Display - unknown command to set check mark: ' cmd.Name]);
            end
        end
        
        function OnCommand(obj, cmd, id, context)
            if ~isempty(strfind(cmd.Name, 'Window.Show'))                
                prop = strrep(cmd.Name, 'Window.', '');
                if (~isprop(context, prop))
                    siman.Debug.ReportProblem(['Display command to show has invalid panel name: ' cmd.Name]);
                    return;
                end
                obj.(prop) = ~obj.(prop);
                return;
            end
            
            switch cmd.Name
                case {'File.SaveFullImageAs', 'File.SaveImageStackAs', 'File.SaveVisibleImageAs', 'File.SaveDataAs'}
                    command = strrep(cmd.Name, 'File.', '');
                    list = obj.GetSelectedPlots();
                    plot = list.ElementAt(1);
                    plot.OnCommand(command);
                case {'View.ZoomIn', 'View.ZoomOut', 'View.ShowColorbar'}
                    command = strrep(cmd.Name, 'View.', '');
                    obj.PlotPanel.OnCommand(command);
                case {'View.LockPlot'}
                    command = strrep(cmd.Name, 'View.', '');
                    obj.PlotPanel.OnCommand(command);
                case {'Edit.CopyVisibleImage', 'Edit.CopyFullImage', 'Edit.CopyData'}
                    command = strrep(cmd.Name, 'Edit.', '');
                    list = obj.GetSelectedPlots();
                    plot = list.ElementAt(1);
                    plot.OnCommand(command);
                case {'Tools.ROI.AddROIType'}
                    list = obj.GetSelectedPlots();
                    plot = list.ElementAt(1);
                    type = get(id, 'Label');
                    plot.AddROIType(type);
                case {'Tools.MakeMovie'}
                    command = strrep(cmd.Name, 'Tools.', '');
                    list = obj.GetSelectedPlots();
                    plot = list.ElementAt(1);
                    plot.OnCommand(command);
                case {'Window.CloseAllToolPanels'}
                    obj.CloseAllToolPanels();
                otherwise
                    siman.Debug.ReportProblem(['Display - unknown command to execute: ' cmd.Name]);
            end
        end
        
        function list = GetROIList(obj)
            plot = obj.CurrentPlot;
            if ~isempty(plot)
                list = plot.ROIList;
            end
        end
        
        function ResetTitle(obj)
            
        end
        
        function OnFigureResized(obj)
            if isempty(obj.PlotPanel)
                return
            end
            figID = obj.FigureID;
            set(figID, 'Units', 'pixels');
            pos = get(figID, 'OuterPosition');
            obj.Location = pos([1 2]);
            obj.Size = pos([3 4]);
            obj.QueueLayout();
            %%obj.ApplyChanges();
        end
        
        function OnButtonUp(obj)
            point = get(obj.FigureID, 'CurrentPoint');
            panel = obj.ButtonDownPanel;
            if ~isempty(panel)
                pos = get(panel.PanelID, 'Position');
                point(1) = point(1) - pos(1) + 1;
                point(2) = point(2) - pos(2) + 1;
                obj.ButtonDownPanel = [];
                panel.OnButtonUp(point);
            end
        end
        
        function OnButtonDown(obj)
            point = get(obj.FigureID, 'CurrentPoint');
            panel = obj.FindHitPanel(point);
            if ~isempty(panel)
                pos = get(panel.PanelID, 'Position');
                point(1) = point(1) - pos(1) + 1;
                point(2) = point(2) - pos(2) + 1;
                panel.OnButtonDown(point);
                obj.ButtonDownPanel = panel;
            end
        end
        
        function OnMouseMotion(obj)
            panel = obj.ButtonDownPanel;
            if ~isempty(panel)
                point = get(obj.FigureID, 'CurrentPoint');
                panel.OnMouseMotion(point);
            end
            notify(obj, 'MouseMotion');
        end
        
        function OnScrollWheelMoved(obj)
            %disp('scroll wheel');
        end
        
        function panel = FindHitPanel(obj, point)
            for i = 1:obj.Panels.Count
                panel = obj.Panels.ElementAt(i);
                if panel.HitTest(point)
                    return
                end
            end
            panel = [];
        end
        
        function RemovePanelType(obj, type)
            for i = 1:obj.Panels.Count
                panel = obj.Panels.ElementAt(i);
                if strcmp(class(panel), ['siman.' type]);
                    obj.RemovePanel(panel);
                    break;
                end
            end
        end
        
        function RemovePanel(obj, panel)
            obj.Panels.Remove(panel);
            delete(panel);
            obj.QueueLayout();
        end
        
        function CloseAllToolPanels(obj)
            props = properties(obj);
            for i = 1:numel(props)
                name = props{i};
                if length(name) > 4 && strcmp(name(1:4), 'Show')
                    if obj.(name)
                        obj.(name) = false;
                    end
                end
            end
        end
        
        function ShowPanel(obj, type)
            switch type
                case 'ImagePropPanel'
                    panel = siman.ImagePropPanel(obj);
                case 'InventoryPanel'
                    panel = siman.InventoryPanel(obj);
                case 'ProcessingDefPanel'
                    panel = siman.ProcessingDefPanel(obj);
                case 'CellDefPanel'
                    panel = siman.CellDefPanel(obj);
                case 'ROIGraphPanel'
                    panel = siman.ROIGraphPanel(obj);
                case 'TransientDetectionPanel'
                    panel = siman.TransientDetectionPanel(obj);
                case 'TransientListPanel'
                    panel = siman.TransientListPanel(obj);
                case 'TransientGraphPanel'
                    panel = siman.TransientGraphPanel(obj);
                case 'SparkDetectionPanel'
                    panel = siman.SparkDetectionPanel(obj);
                case 'SparkListPanel'
                    panel = siman.SparkListPanel(obj);
                case 'SparkGraphPanel'
                    panel = siman.SparkGraphPanel(obj);
                case 'SparkletDetectionPanel'
                    panel = siman.SparkletDetectionPanel(obj);
                case 'SparkletListPanel'
                    panel = siman.SparkletListPanel(obj);
                case 'SparkletGraphPanel'
                    panel = siman.SparkletGraphPanel(obj);
                case 'SparkletSiteDetectionPanel'
                    panel = siman.SparkletSiteDetectionPanel(obj);
                case 'SparkletSiteListPanel'
                    panel = siman.SparkletSiteListPanel(obj);
                case 'SparkletSiteGraphPanel'
                    panel = siman.SparkletSiteGraphPanel(obj);
                case 'ContractionDetectionPanel'
                    panel = siman.ContractionDetectionPanel(obj);
                case 'ContractionGraphPanel'
                    panel = siman.ContractionGraphPanel(obj);
            end
            obj.Panels.Add(panel);
            obj.QueueLayout();
        end
        
        function panel = GetPanel(obj, type)
            panel = [];
            if ~strcmp(type(1:3), 'sim')
                type = ['siman.' type];
            end
            for i = 1:obj.Panels.Count
                element = obj.Panels.ElementAt(i);
                if isa(element, type)
                    panel = element;
                end
            end
        end
        
        function ShowNoticeMessage(obj, str)
            if ~isempty(obj.CurrentPlot)
                obj.CurrentPlot.SetNoticeMessage(str);
            end
        end
        
        function OnManagerSourcesOpened(obj, data_sources)
            obj.DataSources = data_sources(1);
            obj.PlotPanel.SetDataSources(data_sources(1));
        end
        
        function list = GetSelectedPlots(obj)
            list = obj.PlotPanel.SelectedPlots;
        end
        
        function OnCurrentPlotChanged(obj, src, evnt)
            %TODO: put in a check for bad objects in listeners (I think something adds a listener and doesn't remove it on deletion)
            notify(obj, 'CurrentPlotChanged');
        end
        
        function OnPropertyChanged(obj, src, evnt)
            prop = evnt.PropertyName;
            if length(prop) > 4 && strcmp(prop(1:4), 'Show')
                panelName = [prop(5:end) 'Panel'];
                if obj.(prop)
                    obj.ShowPanel(panelName);
                else
                    obj.RemovePanelType(panelName);
                end
            end
            switch evnt.PropertyName
            end
        end
        
        function ApplyChanges(obj)
            if obj.NeedsLayout
                obj.ResetLayout();
                obj.NeedsLayout = false;
            end
            for i = obj.Panels.Count:-1:1  % Go backwards so that plot panel refreshes last (i.e. other panels aren't blocked by processing)
                panel = obj.Panels.ElementAt(i);
                panel.ApplyChanges();
            end
            % This should have been fixed when the HasPendingChanges methods were implemented
            %for i = 2:obj.Panels.Count  % Apply again in case panel plot produced changes (e.g. color range on image panel)
            %    panel = obj.Panels.ElementAt(i);
            %    panel.ApplyChanges();
            %end
        end
        
        function tf = HasPendingChanges(obj)
            tf = obj.NeedsLayout;
            if tf
                return;
            end
            for i = 1:obj.Panels.Count  % Go backwards so that plot panel refreshes last (panels show true data while drawing is happening)
                panel = obj.Panels.ElementAt(i);
                tf = panel.HasPendingChanges();
                if tf
                    return;
                end
            end            
        end
        
        function QueueLayout(obj)
            obj.NeedsLayout = true;
        end
        
        function delete(obj)
            delete(obj.PlotPanelListener);
            obj.PlotPanelListener = [];
            delete(obj.Panels);
            obj.Panels = [];
            siman.ImageManager.UnregisterDisplay(obj);
            siman.DefaultsRegistry.SaveRegistry(true);
        end
    end
end



