classdef (ConstructOnLoad) SimanObject < siman.Notifier
    % SimanObject - Base class for object of the siman package who need:
    % 1. Automatic storage of properties (i.e. last value always remembered)
    % 2. PropertyChanged event for listening (from Notifier base class)
    properties
        RegistryCategory
    end
    properties (Transient)
        RegisteredPropertyList
        RegisteredPropertyListener
    end
    
    methods % properties
        function value = get.RegistryCategory(obj)
            value = class(obj);
        end       
    end
    
    methods
        function obj = SimanObject() % Constructor
            obj@siman.Notifier();
            obj.RegisteredPropertyList = obj.GetRegisteredPropertyNames();
            
            ioHelper = siman.FileIOHelper.GetHelper();
            if ~ioHelper.IsLoading
                obj.NotificationActive = false;
                obj.InitRegisteredProperties();
                obj.NotificationActive = true;
            end
            obj.InitSimanObjectListeners();
        end
        
        function list = GetRegisteredPropertyNames(obj)
            % Determine list of properties that need entering in the
            % registry (i.e. automatic storage of last value set)
            % This should be a static property of the class, but Matlab
            % properties cannot be static.
            info = metaclass(obj);
            props = info.PropertyList;
            names = {props.Name};
%             indexes = false(numel(props), 1);
%             for i = 1:length(props)
%                 % The GetObservable atrribute is not often useful, so I am
%                 % using it to signify a property as needing automatic registration.  If the
%                 % attribute is really being used and registry entry is not
%                 % desired, override this function to remove/add props.
%                 if props{i}.GetObservable
%                     names{i} = props{i}.Name;
%                     indexes(i) = true;
%                 end
%             end
            list = names([props.SetObservable]);
        end
        
        function InitRegisteredProperties(obj)
            list = obj.RegisteredPropertyList;        
            category = obj.RegistryCategory;
            for i = 1:length(list)
                prop = list{i};
                % Initialize last value (assuming this is creation)
                [value, found] = siman.DefaultsRegistry.GetRegistryEntry(category, prop);
                if found
                    %HACK to prevent saved deleted objects
                    if isobject(value) && ~isvalid(value)
                        continue;
                    end
                    obj.(prop) = value;
                end
            end
        end
        
        function InitSimanObjectListeners(obj)
            obj.RegisteredPropertyListener = addlistener(obj, 'PropertyChanged', @obj.OnRegisteredPropertyChanged);
        end
        
        function OnRegisteredPropertyChanged(obj, src, evnt)
            if ~isvalid(obj)
                return;
            end
            list = obj.RegisteredPropertyList;
            prop = evnt.PropertyName;
            if ismember(prop, list) %% This could potentially be skiped if it is agreed that NotiferProperties and RegisteredProperties are the same
                category = obj.RegistryCategory;
                siman.DefaultsRegistry.SetRegistryEntry(category, prop, obj.(prop));
            end
        end
        
        %function HandleRegisteredPropertyChanged(obj, src, evnt)
        %    category = obj.RegistryCategory;
        %    prop = src.Name;
        %    siman.DefaultsRegistry.SetRegistryEntry(category, prop, obj.(prop));
        %end
        
        function delete(obj)
            delete(obj.RegisteredPropertyListener);
            obj.RegisteredPropertyListener = [];
        end
    end
    
    methods(Access = protected)
        % Override copyElement method:
        function cpObj = copyElement(obj)
            % Make a shallow copy of all properties
            cpObj = copyElement@siman.Notifier(obj);
            % Reset the listeners (since they don't copy)
            cpObj.InitSimanObjectListeners();
        end
    end
end
