classdef DefaultsRegistry < handle
    % DefaultsRegistry.
    properties (Constant)
        Instance = siman.DefaultsRegistry;
        RegistryFilename = 'SimanRegistry.mat';
        Version = 1.0;
        MinSaveInterval = 60; % seconds
    end
    properties (Access = private)
        RegistryFile
        RegistryData
    end
    properties (Transient)
        RegistryPath
        RegistryCategories
        NeedsSaving
        LastSaveTime
    end
    methods
        function obj = DefaultsRegistry() % Singleton constructor
            obj.RegistryPath = siman.FileIOHelper.GetSimanPath();
            obj.Initialize();
        end
        function Initialize(obj)
            file = fullfile(obj.RegistryPath, obj.RegistryFilename);
            obj.RegistryFile = file;
            ioHelper = siman.FileIOHelper.GetHelper();
            if exist(file, 'file')
                try
                    ioHelper.IsLoading = true;
                    load(file, 'registry', 'version');
                catch exception
                    siman.Debug.ReportError(exception, 'Error while loading registry');
                end
                ioHelper.IsLoading = false;
                if exist('registry', 'var')
                    obj.RegistryData = registry;
                    if ~isempty(registry)
                        obj.RegistryCategories = {registry.Category};
                    end
                end
            else
                obj.RegistryData = [];
            end
            obj.LastSaveTime = now;
        end
       
        function delete(obj)
            siman.DefaultsRegistry.SaveRegistry(true);
        end
    end
    methods (Static = true)
        function reg = GetRegistry()
            reg = siman.DefaultsRegistry.Instance;
        end
        function SaveRegistry(force)
            import siman.DefaultsRegistry;
            reg = siman.DefaultsRegistry.GetRegistry();
            if ~isempty(reg) && ~isempty(reg.RegistryFile)
                registry = reg.RegistryData;
                if isempty(registry)
                    return
                end
                if nargin < 1
                    force = false;
                end
                if ~force
                    saveInterval = (now - reg.LastSaveTime) * 24 * 3600;
                    if saveInterval < reg.MinSaveInterval
                        return
                    end
                end
                try
                    version = reg.Version;
                    save(reg.RegistryFile, 'registry', 'version');
                    reg.LastSaveTime = now;
                catch
                end
            end
        end     
        function [value, found] = GetRegistryEntry(category, name)
            reg = siman.DefaultsRegistry.GetRegistry();
            found = false;
            value = [];
            if isempty(reg) || isempty(reg.RegistryData)
                return
            end
            cat_index = find(strcmp(reg.RegistryCategories, category), 1);
            if isempty(cat_index)
                return
            end
            entryNames = reg.RegistryData(cat_index).EntryNames;
            if isempty(entryNames)
                return
            end
            entry_index = find(strcmp(name, entryNames), 1);
            if isempty(entry_index)
                return
            end
            found = true;
            value = reg.RegistryData(cat_index).EntryValues{entry_index};
            if ~isempty(value) && isobject(value)
                value = copy(value);
            end
        end     
        function SetRegistryEntry(category, name, value)
            import siman.DefaultsRegistry;
            reg = DefaultsRegistry.Instance;
            if isempty(reg)
                return
            end
            if ~isempty(value) && isobject(value)
                value = copy(value);
            end
            DefaultsRegistry.QueueSave();
            if isempty(reg.RegistryData)
                registry.Category = category;
                registry.EntryNames = {name};
                registry.EntryValues = {value};
                reg.RegistryCategories = {category};
                reg.RegistryData = registry;
                return;
            end
            cat_index = find(strcmp(category, reg.RegistryCategories), 1);
            if isempty(cat_index)
                registry = reg.RegistryData;
                registry(end+1).Category = category;
                registry(end).EntryNames = {name};
                registry(end).EntryValues = {value};
                reg.RegistryCategories{end + 1} = category;
                reg.RegistryData = registry;
                return;
            end
            
            entry_index = find(strcmp(name, reg.RegistryData(cat_index).EntryNames), 1);
            if isempty(entry_index)
                reg.RegistryData(cat_index).EntryNames{end + 1} = name;
                reg.RegistryData(cat_index).EntryValues{end + 1} = value;
                return;
            end
            oldValue = reg.RegistryData(cat_index).EntryValues{entry_index};
            if ~isempty(oldValue) && isobject(oldValue)
                delete(oldValue);
            end
            reg.RegistryData(cat_index).EntryValues{entry_index} = value;
        end
        
        function QueueSave
            import siman.DefaultsRegistry;
            reg = DefaultsRegistry.Instance;
            if isempty(reg)
                return
            end
            reg.NeedsSaving = true;
        end
        
        function ApplyChanges()
            import siman.DefaultsRegistry;
            reg = DefaultsRegistry.Instance;
            if isempty(reg)
                return
            end
            if (reg.NeedsSaving)
                DefaultsRegistry.SaveRegistry();
                reg.NeedsSaving = false;
            end
        end
    end
end





