classdef FeatureContainer < siman.SimanObject
    % Must be a subclass of handle
    properties
        FeatureList
    end
    properties (Dependent = true)
        HasFeatures
        FeatureCount
        IsTransientSource
    end
    properties (GetObservable, SetObservable) % Registered properties
        ExportFeatureDataMethod = 'multiple files';
        ExportFeatureDataIncludeLabels = false;
        ExportFeaturePropsIncludeLabels = true;
        CopyFeatureDataIncludeLabels = true;
        CopyFeaturePropsIncludeLabels = true;
        
        LastFeatureReportPath
        LastFeatureReportDir
        LastFeatureExportPath
    end
    properties (Transient)
        FeatureListListeners
        LastRequestedFeatureType
        LastRequestedFeatureList
    end
    properties
        LastTransientDetectionMethod
        LastContractionDetectionMethod
        LastSparkDetectionMethod
        LastSparkletDetectionMethod
        LastSparkletSiteDetectionMethod
    end
    properties (Constant)
        ExportMethodOptions = {'single file', 'multiple files'};
    end
    
    events
        FeaturesChange %TODO rename to FeatureListChange
        FeatureChange
    end
    
    methods % Properties
        function set.FeatureList(obj, value)
            if isempty(value) % internal code for nulling the property during a copy
                obj.FeatureList = [];
                obj.FeatureListListeners = [];
                return
            end
            if ~isempty(obj.FeatureList)
                delete(obj.FeatureList);
                if ~isempty(obj.FeatureListListeners)
                    delete(obj.FeatureListListeners)
                end
            end
            obj.FeatureList = value;
            if ~isempty(value)
                obj.FeatureList.DeleteElementsOnClear = true;
                obj.FeatureList.DeleteElementsOnDelete = true;
                obj.FeatureList.NotifyOnItemChange = true;
                obj.FeatureListListeners = addlistener(obj.FeatureList, 'CollectionChange', @obj.OnFeatureListChanged);
                obj.FeatureListListeners(2) = addlistener(obj.FeatureList, 'ItemChange', @obj.OnFeatureChanged);
            end
        end
        function count = get.FeatureCount(obj)
            count = obj.FeatureList.Count;
        end
        function result = get.HasFeatures(obj)
            result = obj.FeatureList.Count > 0;
        end
    end
    
    methods
        % Constructor
        function obj = FeatureContainer()
            obj.FeatureList = siman.ObservableCollection;
            obj.FeatureList.IsMulticlass = true;
        end
        
        function AddFeature(obj, feature)
            obj.FeatureList.Add(feature);
        end
        function AddFeatures(obj, list)
            obj.FeatureList.AppendList(list);
        end
        function RemoveFeature(obj, feature)
            delete(feature);
            obj.FeatureList.Remove(feature);
        end
        function RemoveSelectedFeatures(obj, type)
            indexes = [];
            for i = 1:obj.FeatureList.Count
                feature = obj.FeatureList.ElementAt(i);
                if feature.IsSelected && (isa(feature, type) || strcmp(type, feature.Type))
                	indexes(end + 1) = i;
                end
            end
            obj.FeatureList.RemoveIndexes(indexes);
        end
        function ClearFeatures(obj)
            for i = 1:obj.FeatureList.Count
                feature = obj.FeatureList.ElementAt(i);
                delete(feature);
            end
            obj.FeatureList.Clear();
        end
        function ClearFeatureType(obj, type)
            indexes = [];
            for i = 1:obj.FeatureList.Count
                feature = obj.FeatureList.ElementAt(i);
                if isa(feature, type) || strcmp(type, feature.Type)
                	indexes(end + 1) = i;
                end
            end
            obj.FeatureList.RemoveIndexes(indexes);
        end
        function feature = GetFeature(obj, index)
            feature = [];
            if index <= obj.FeatureList.Count
                feature = obj.FeatureList.ElementAt(index);
            end
        end
        function list = GetFeatureType(obj, type)
            if (~isempty(obj.LastRequestedFeatureType) && strcmp(type, obj.LastRequestedFeatureType))
                list = obj.LastRequestedFeatureList;
                return;
            end
            obj.LastRequestedFeatureType = type;
            list = siman.List;
            count = obj.FeatureList.Count;
            for i = 1:count
                feature = obj.FeatureList.ElementAt(i);
                if isa(feature, type) || strcmp(type, feature.Type)
                    list.Add(feature);
                end
            end
            obj.LastRequestedFeatureList = list;
        end
        function list = GetSelectedFeatureType(obj, type)
            list = siman.List;
            baseList = obj.GetFeatureType(type);
            count = baseList.Count;
            array = baseList.ToArray();
            for i = 1:count
                feature = array(i);
                if feature.IsSelected
                    list.Add(feature);
                end
            end
        end
        function SelectFeatures(obj, features_to_select, type)
            fullList = obj.FeatureList;
            count = fullList.Count;
            if fullList.Count == 0
                return
            end
            isSelected = zeros(count, 2);
            for i = 1:count
                feature = fullList.ElementAt(i);
                isSelected(i, [1 2]) = feature.IsSelected;
                if isa(features_to_select, 'siman.ObservableCollection') || isa(features_to_select, 'siman.List')
                    isMatch = features_to_select.IsMember(feature);
                else
                    isMatch = isequal(feature, features_to_select);
                end
                switch (type)
                    case 'select'
                        isSelected(i,2) = isMatch;
                    case 'add'
                        isSelected(i,2) = isMatch || isSelected(i,2);
                    case 'swap'
                        if isMatch
                            isSelected(i,2) = ~isSelected(i,2);
                        end
                end
            end
            for i = 1:count
                if isSelected(i,1) ~= isSelected(i,2)
                    feature = fullList.ElementAt(i);
                    feature.IsSelected = isSelected(i,2);
                end
            end
        end
        function OnFeatureListChanged(obj, src, evnt)
            obj.FireFeatureListChange(evnt.Type, evnt.Param);
            obj.LastRequestedFeatureType = [];
            if ~isempty(obj.LastRequestedFeatureList)
                if isvalid(obj.LastRequestedFeatureList)
                    delete(obj.LastRequestedFeatureList);
                    obj.LastRequestedFeatureList = [];
                end
            end
        end
        function OnFeatureChanged(obj, src, evnt)
            obj.FireFeatureChange([]);
        end
        
        function FireFeatureListChange(obj, type, param)
            notify(obj, 'FeaturesChange');
        end
        function FireFeatureChange(obj, feature)
            notify(obj, 'FeatureChange');
        end
        
        
        function method = GetLastDetectionMethod(obj, type)
            switch type
                case 'contraction'
                    method = obj.LastContractionDetectionMethod;
                case 'transient'
                    method = obj.LastTransientDetectionMethod;
                case 'spark'
                    method = obj.LastSparkDetectionMethod;
                case 'sparklet'
                    method = obj.LastSparkletDetectionMethod;
                case 'sparklet site'
                    method = obj.LastSparkletSiteDetectionMethod;
            end
        end
        function mgr = GetFeatureManager(obj, type)
            switch type
                case 'contraction'
                    mgr = siman.ContractionManager.GetManager();
                case 'transient'
                    mgr = siman.TransientManager.GetManager();
                case 'spark'
                    mgr = siman.SparkManager.GetManager();
                case 'sparklet'
                    mgr = siman.SparkletManager.GetManager();
                case 'sparklet site'
                    mgr = siman.SparkletSiteManager.GetManager();
            end
        end
        function tf = WasDetected(obj, type)
            tf = ~isempty(obj.GetLastDetectionMethod(type));
        end
        
        % Export and copy routines
        function CopyFeatureData(obj, list)
            result = obj.GetDataExportText(list, obj.CopyFeatureDataIncludeLabels);
            clipboard('copy', result);
        end
        function CopyFeatureProps(obj, list)
            result = obj.GetPropExportText(list, obj.CopyFeaturePropsIncludeLabels);
            clipboard('copy', result);
        end
        function ExportFeatureData(obj, list)
            switch obj.ExportFeatureDataMethod
                case 'single file'
                    obj.ExportFeatureDataAsSingleFile(list);
                case 'multiple files'
                    obj.ExportFeatureDataAsMultipleFiles(list);
            end
        end
        
        function ExportFeatureDataAsSingleFile(obj, list)
            filename = siman.FileIOHelper.ChooseOutputFile({'*.xml', 'Xml files'}, obj, 'LastFeatureExport');
            if isempty(filename)
                return
            end
            warndlg('Not fully implemented');
            %result = obj.GetDataExportText(list, obj.ExportFeatureDataIncludeLabels);            
        end
        
        function ExportFeatureDataAsMultipleFiles(obj, list)
            dir = siman.FileIOHelper.ChooseOutputPath(obj, 'LastFeatureExport');
            if isempty(dir)
                return
            end
            result = obj.GetExportDataMatrix(list);
            for i = 1:list.Count
                feature = list.ElementAt(i);
                obj.WriteFeatureDataFile(feature, dir, result(:,i));
            end
        end
        function WriteFeatureDataFile(obj, feature, dir, data)
            filename = fullfile(dir, [obj.Name '_' feature.Name '.txt']);
            if exist(filename, 'file')
                delete(filename);
            end
            str = '';
            if obj.ExportFeatureDataIncludeLabels
                id = fopen(filename);
                fprintf(id, '%s%s', feature.Name, siman.UIHelper.GetLineDelimiter());
                fclose(id);
            end
            if ispc
                newline = 'pc';
            else
                newline = 'unix';
            end
            dlmwrite(filename, data, '-append', 'delimiter', '\t', 'newline', newline);
        end
        
        function result = GetExportDataMatrix(obj, list)
            count = list.Count;
            result = [];
            if count == 0
                return
            end
            
            feature = list.ElementAt(1);
            vector = feature.GetExportData();
            vector = vector(:); % ensure row vector
            
            result = zeros(numel(vector), count);
            result(:,1) = vector;
            for i = 2:count
                feature = list.ElementAt(i);
                vector = feature.GetExportData();
                if ~isempty(vector)
                    result(:,i) = vector;
                end
            end
        end
        
        function result = GetExportPropCellArray(obj, list, include_labels)
            count = list.Count;
            result = {};
            if count == 0
                return
            end
            if nargin < 3
                include_labels = true;
            end
            
            result = cell(count+include_labels, 1);
            for i = 1:count
                feature = list.ElementAt(i);
                result{i+include_labels} = siman.UIHelper.WriteLine(feature.GetPropExportString());
            end
            if include_labels
                labels = feature.GetExportPropList();
                str = 'Name';
                for i = 1:length(labels)
                    str = [str ',' labels{i}];
                end
                result{1} = siman.UIHelper.WriteLine(str);
            end
        end
        function result = GetExportDataStrings(obj, list, include_labels)
            count = list.Count;
            result = {};
            if count == 0
                return
            end
            if nargin < 3
                include_labels = true;
            end
            
            data = obj.GetExportDataMatrix(list);
            dim = size(data);
            vectorLength = dim(1);
            result = cell(vectorLength+include_labels, 1);
            for row = 1:vectorLength
                str = num2str(data(row,1));
                for i = 2:count
                    str = [str ',' num2str(data(row,i))];
                end
                result{row+include_labels} = siman.UIHelper.WriteLine(str);
            end
            if include_labels
                feature = list.ElementAt(1);
                str = feature.Name;
                for i = 2:count
                    feature = list.ElementAt(i);
                    str = [str ',' feature.Name];
                end
                result{1} = siman.UIHelper.WriteLine(str);
            end
        end
        
        function str = GetPropExportText(obj, list, include_labels)
            if nargin < 3
                include_labels = true;
            end
            result = obj.GetExportPropCellArray(list, include_labels);
            str = '';
            for i = 1:length(result)
                str = [str result{i}];
            end
        end
        function str = GetDataExportText(obj, list, include_labels)
            if nargin < 3
                include_labels = true;
            end
            result = obj.GetExportDataStrings(list, include_labels);
            str = '';
            for i = 1:length(result)
                str = [str result{i}];
            end
        end
        
        
        % Destructor
        function delete(obj)
            if ~isempty(obj.FeatureList)
                delete(obj.FeatureList);
                delete(obj.FeatureListListeners);
                obj.FeatureList = [];
            end
           if ~isempty(obj.LastTransientDetectionMethod)
               delete(obj.LastTransientDetectionMethod);
               obj.LastTransientDetectionMethod = [];
           end
           if ~isempty(obj.LastContractionDetectionMethod)
               delete(obj.LastContractionDetectionMethod);
               obj.LastContractionDetectionMethod = [];
           end
           if ~isempty(obj.LastSparkDetectionMethod)
               delete(obj.LastSparkDetectionMethod);
               obj.LastSparkDetectionMethod = [];
           end
           if ~isempty(obj.LastSparkletDetectionMethod)
               delete(obj.LastSparkletDetectionMethod);
               obj.LastSparkletDetectionMethod = [];
           end
           if ~isempty(obj.LastSparkletSiteDetectionMethod)
               delete(obj.LastSparkletSiteDetectionMethod);
               obj.LastSparkletSiteDetectionMethod = [];
           end
        end
    end
end