classdef FeatureManager < siman.SimanObject
    % FeatureManager
    properties (SetObservable, GetObservable)
        FeatureDetectionMethod
    end
    properties (GetObservable, SetObservable) % Registered properties
        LastFeatureMethodPath
        LastFeatureReportPath
        LastFeatureReportStyleSheetPath
    end
    properties
        SourceArchivingProperty
    end
    properties (Transient)
        DetectionMethodListener
        
        ReportContext
    end
    
    events
        MethodChanged
    end
    
    methods % Properties
        function set.FeatureDetectionMethod(obj, value)
            if isempty(value) % internal code for nulling the property during a copy
                obj.FeatureDetectionMethod = [];
                obj.DetectionMethodListener = [];
                return
            end
            if ~isempty(obj.FeatureDetectionMethod) && isvalid(obj.FeatureDetectionMethod)
                delete(obj.FeatureDetectionMethod);
                if ~isempty(obj.DetectionMethodListener)
                    delete(obj.DetectionMethodListener)
                end
            end
            obj.FeatureDetectionMethod = value;
            if ~isempty(value)
                obj.DetectionMethodListener = addlistener(obj.FeatureDetectionMethod, 'MethodChange', @obj.OnMethodChanged);
            end
        end
    end
    
    methods
        % Constructor
        function obj = FeatureManager()
            obj@siman.SimanObject();
            
            obj.InitCommands();
        end
        
    end
    
    methods
        function InitCommands(obj)
            mgr = siman.ImageManager.GetManager();
            if isempty(mgr)
                return
            end

            % This command needs a context with a TargetDataSources property
            cmd = mgr.RegisterCommand([obj.CommandPrefix 'Detect'], obj);
            cmd.Label = 'Detect';
            cmd.Tooltip = ['Detect ' obj.FeatureType 's in the currently selected image'];
            
            cmd = mgr.RegisterCommand([obj.CommandPrefix 'GenerateReport'], obj);
            cmd.Label = 'r';
            cmd.Tooltip = ['Generate a ' obj.FeatureType ' report for the currently selected image'];
            
            % These commands need a context with a GetFeatureList and GetSelectedFeaturesList method
            cmd = mgr.RegisterCommand([obj.CommandPrefix 'CopyAllFeatures'], obj);
            cmd.Label = 'c';
            cmd.Tooltip = ['Copy all ' obj.FeatureType ' properties of the currently selected image'];
            
            cmd = mgr.RegisterCommand([obj.CommandPrefix 'ExportAllFeatures'], obj);
            cmd.Label = 'e';
            cmd.Tooltip = ['Export all ' obj.FeatureType ' properties of the currently selected image to a file'];
            
            cmd = mgr.RegisterCommand([obj.CommandPrefix 'DeleteAllFeatures'], obj);
            cmd.Label = '-';
            cmd.Tooltip = ['Delete all ' obj.FeatureType 's of the image'];
            
            cmd = mgr.RegisterCommand([obj.CommandPrefix 'DeleteSelectedFeatures'], obj);
            cmd.Label = '-';
            cmd.Tooltip = ['Delete selected ' obj.FeatureType 's of the image'];
            
            cmd = mgr.RegisterCommand([obj.CommandPrefix 'CopyAllFeatureData'], obj);
            cmd.Label = 'c';
            cmd.Tooltip = ['Copy the data of all ' obj.FeatureType 's of the currently selected image'];
            
        end
        
        function InitDefinitions(obj)
            mgr = siman.ImageManager.GetManager();
            
            % Get definitions from manager if they exist, otherwise register them
            % (If they exist, it's because they were recalled from a saved version)
            % (A saved version should always exists except at first startup - once implemented at least)
            if obj.FeatureDetectionMethod.HasDetectionDefinition
                def = mgr.FindDefinition(obj.FeatureDetectionMethod.DetectionDefName);
                if isempty(def) % This should be the case unless we start saving "system" defs with user defs
                    if isempty(obj.FeatureDetectionMethod.DetectionDefinition)
                        def = obj.FeatureDetectionMethod.GetDefaultDetectionDefinition();
                        obj.FeatureDetectionMethod.DetectionDefinition = def;
                    else
                        % This should be the normal case if we are relying on SimanObject to save prop FeatureDetectionMethod
                        def = obj.FeatureDetectionMethod.DetectionDefinition;
                    end
                    mgr.RegisterDefinition(def);
                else
                    obj.FeatureDetectionMethod.DetectionDefinition = def;
                end
            end
            
            if obj.FeatureDetectionMethod.HasUnitsDefinition
                def = mgr.FindDefinition(obj.FeatureDetectionMethod.UnitsDefName);
                if isempty(def)
                    if isempty(obj.FeatureDetectionMethod.UnitsDefinition)
                        def = obj.FeatureDetectionMethod.GetDefaultUnitsDefinition();
                        obj.FeatureDetectionMethod.UnitsDefinition = def;
                    else
                        % This should be the normal case if we are relying on SimanObject to save prop FeatureDetectionMethod
                        def = obj.FeatureDetectionMethod.UnitsDefinition;
                    end
                    mgr.RegisterDefinition(def);
                else
                    obj.FeatureDetectionMethod.UnitsDefinition = def;
                end
            end
        end
        
        function OnMethodChanged(obj, ~, ~)
            siman.DefaultsRegistry.SetRegistryEntry(obj.RegistryCategory, ...
                'FeatureDetectionMethod', obj.FeatureDetectionMethod);
        end
        
        function OnCommand(obj, cmd, ~, context) % id is missing argument
            prefix = obj.CommandPrefix;
            switch cmd.Name
                case [prefix 'CopyAllFeatures']
                    list = context.GetFeatureList();
                    context.CurrentFeatureSource.CopyFeatureProps(list);
                case [prefix 'CopySelectedFeatures']
                    list = context.GetSelectedFeatureList();
                    context.CurrentFeatureSource.CopyFeatureProps(list);
                case [prefix 'ExportAllFeatures']
                    list = context.GetFeatureList();
                    context.CurrentFeatureSource.ExportFeatureData(list);
                case [prefix 'CopyAllFeatureData']
                    list = context.GetFeatureList();
                    context.CurrentFeatureSource.CopyFeatureData(list);
                case [prefix 'CopySelectedFeatureData']
                    list = context.GetSelectedFeatureList();
                    context.CurrentFeatureSource.CopyFeatureData(list);
                case [prefix 'DeleteSelectedFeatures']
                    choice = questdlg('Are you sure you want to delete the selected feature(s)?', ...
                        'Confirm delete', 'Yes', 'No', 'Cancel', 'Cancel');
                    if strcmp(choice, 'Yes')
                        context.CurrentFeatureSource.RemoveSelectedFeatures(obj.FeatureType);
                    end
                case [prefix 'SelectFeatures']
                    list = context.GetSelectedFeatureList();
                    context.CurrentFeatureSource.CopyFeatureData(list);
                case [prefix 'Detect']
                    sources = context.TargetDataSources;
                    count = sources.Count;
                    for i = 1:count
                        target = sources.ElementAt(i);
                        obj.DetectFeatures(target);
                    end
                case [prefix 'GenerateReport']
                    sources = context.TargetDataSources;
                    count = sources.Count;
                    for i = 1:count
                        target = sources.ElementAt(i);
                        obj.ProduceReport(target);
                    end
                otherwise
                    siman.Debug.ReportProblem(['Unknown feature command to execute: ' cmd.Name]);
            end
        end
        
        function OnPreDetection(obj, method, target)
            target.ClearFeatureType(obj.FeatureType);
            if ~isempty(obj.SourceArchivingProperty)
                prop = obj.SourceArchivingProperty;
                if ~isempty(target.(prop))
                    delete(target.(prop))
                end
            end
        end
        
        function OnPostDetection(obj, method, target)
            if ~isempty(obj.SourceArchivingProperty)
                % This must run post since the cell definition is stored during processing
                target.(obj.SourceArchivingProperty) = copy(method); % Store the parameters used
            end
            method.FireMethodRan();
            method.DetectionDate = datestr(now);
        end
        
        function DetectFeatures(obj, target)
            method = obj.FeatureDetectionMethod;
            mgr = siman.ImageManager.GetManager();
            mgr.ShowNoticeMessage(['Detecting ' obj.FeatureType 's....']);
            
            obj.OnPreDetection(method, target);
            [list, message] =  method.Detect(target);
            mgr.ShowNoticeMessage('');
            if ~isempty(message)
                warndlg(message);
            elseif list.Count == 0
                warndlg(['No ' obj.FeatureType 's detected']);
            end
            
            obj.OnPostDetection(method, target);
        end
        
        function enabling = ResetCommandEnabling(obj, cmd, context)
            prefix = obj.CommandPrefix; % This just makes cross-feature copy/paste of code easier
            switch cmd.Name
                case [prefix 'CopyAllFeatures']
                    list = context.FeatureList;
                    enabling = list.Count > 0;
                case [prefix 'CopySelectedFeatures']
                    list = context.SelectedFeatureList;
                    enabling = list.Count > 0;
                case [prefix 'ExportAllFeatures']
                    list = context.FeatureList;
                    enabling = list.Count > 0;
                case [prefix 'CopyAllFeatureData']
                    list = context.FeatureList;
                    enabling = list.Count > 0;
                case [prefix 'CopySelectedFeatureData']
                    list = context.SelectedFeatureList;
                    enabling = list.Count > 0;
                case [prefix 'DeleteSelectedFeatures']
                    list = context.SelectedFeatureList;
                    enabling = list.Count > 0;
                case [prefix 'DeleteAllFeatures']
                    list = context.FeatureList;
                    enabling = list.Count > 0;
                case [prefix 'Detect']
                    enabling = ~isempty(context.TargetDataSources) && context.TargetDataSources.Count > 0;
                    %TODO Need to check for valid detection method?
                case [prefix 'GenerateReport']
                    enabling = ~isempty(context.TargetDataSources) && context.TargetDataSources.Count > 0;
                    if enabling
                        source = context.TargetDataSources.ElementAt(1);
                        enabling = source.WasDetected(obj.FeatureType);
                    end
                    %TODO Need to check for valid detection method?
                otherwise
                    siman.Debug.ReportProblem(['Unknown feature command to enable: ' cmd.Name]);
                    enabling = false;
            end
        end
        
        function delete(obj)
            delete(obj.FeatureDetectionMethod);
            obj.FeatureDetectionMethod = [];
        end
        
        function ApplyChanges(obj)
            method = obj.FeatureDetectionMethod;
            if ~isempty(method)
                method.ApplyChanges();
            end
        end
        
        function tf = HasPendingChanges(obj)
            method = obj.FeatureDetectionMethod;
            if ~isempty(method)
                tf = method.HasPendingChanges();
            end
        end
        
    end
end





