classdef ImageFile < siman.ImageSource & siman.FeatureContainer
    % Must be a subclass of handle
    properties
        FullFilename = '';
        Path = '';
        Extension = '';
        Format
        
        CellDefinition
    end
    properties (Transient = true)
        ImageDataCacheID
        Thumbnail
        CellDefListener
    end
    properties (Dependent)
        IsLoaded
    end
    
    events
        CellDefinitionChange
        CellMaskUpdated
    end
    
    methods % Properties
        function loaded = get.IsLoaded(obj)
            loaded = siman.DataCache.DataIsCached(obj.ImageDataCacheID);
        end
    end
    
    methods
        % Constructors
        function obj = ImageFile(arg)
            if nargin > 0
                if ischar(arg)
                    obj.SetFileSource(arg);
                    obj.CellDefinition = siman.CellDefinition(obj);
                    obj.CellDefListener = addlistener(obj.CellDefinition, 'CellMaskUpdated', @obj.OnCellUpdated);
                    obj.InitCellDefinition();
                elseif isnumeric(arg)
                    obj(1:arg) = siman.ImageFile;
                end
            end
        end
        
        % Methods
        function [success, message] = SetFileSource(obj, filename)
            success = false;
            obj.Path = '';
            if ~exist(filename, 'file')
                obj.SetErrorState(true, ['File does not exist - "' filename '"']);
                siman.Debug.ReportProblem(obj.ErrorMessage);
                message = obj.ErrorMessage;
                return;
            end
            try
                info = imfinfo(filename);
            catch e
                obj.SetErrorState(true, ['Error while attempting to determine image format type: ' e.message]);
                siman.Debug.ReportProblem(obj.ErrorMessage);
                message = obj.ErrorMessage;
                return;
            end
            [path, name, ext] = fileparts(filename);
            obj.FullFilename = filename;
            obj.Name = name;
            obj.Path = path;
            obj.Extension = ext;
            
            obj.Format = info(1).Format;
            
            obj.Width = info(1).Width;
            obj.Height = info(1).Height;
            obj.BitDepth = info(1).BitDepth;
            obj.ColorType = info(1).ColorType;
            obj.ImageCount = length(info);
            if obj.ImageCount > 1
                obj.DataLayout = 'series';
            else
                obj.DataLayout = 'linescan';
            end
            success = true;
            message = '';
            
            % Test for color image that is not really color
            if strcmp(obj.ColorType, 'truecolor')
                obj.LoadData();
            end
            
            % Get a "thumbnail" (or just whole thing if 1 image)
            if obj.ImageCount > 1
                obj.Thumbnail = obj.LoadFirstImage();
            end
        end
        
        function OnCellUpdated(obj, src, evt)
            notify(obj, 'CellMaskUpdated');
        end
        
        function thumb = GetThumbnail(obj)
            if obj.ImageCount == 1
                thumb = obj.GetData();
            else
                thumb = obj.Thumbnail;
            end
        end
        
        function allData = LoadData(obj)
            allData = [];
            if ~exist(obj.FullFilename, 'file')
                return
            end
            try
                allData = zeros(obj.Height, obj.Width, obj.ImageCount);
                info = imfinfo(obj.FullFilename);
                for i = 1:obj.ImageCount
                    data = imread(obj.FullFilename, 'Index', i, 'Info', info);

                    % If file is color, make sure it's not just 3 layers of same value
                    if strcmp(obj.ColorType, 'truecolor')
                        if all(data(:,:,1) == data(:,:,2))
                            allData = data(:,:,1);
                            obj.ColorType = 'grayscale';
                        elseif all(data(:,:,2) == 0) % or all data is only in red channel
                            allData = data(:,:,1);
                            obj.ColorType = 'grayscale';
                        else
                            allData = data;
                        end
                    else
                        allData(:,:,i) = data;
                    end
                end
            catch ex
                obj.SetErrorState(true, ['File could not be loaded - ' ex.message]);
                return;
            end
            obj.ImageDataCacheID = siman.DataCache.SetCachedData(allData, obj.ImageDataCacheID, obj);
            obj.FireDataChanged('loaded');
        end
        
        function data = LoadFirstImage(obj)
            data = [];
            try
                data = imread(obj.FullFilename);
            catch
                obj.SetErrorState(true, 'File could not be loaded');
                return;
            end
        end
        
        function InitCellDefinition(obj)
            if obj.IsValid
                switch obj.DataLayout
                    case 'linescan'
                        obj.CellDefinition.Orientation = 'linescan';
                    case 'series'
                        obj.CellDefinition.Orientation = '2-d';
                end
            end
        end
        
        function data = GetData(obj)
            [data, found] = siman.DataCache.GetCachedData(obj.ImageDataCacheID);
            if found
                return;
            end
            data = obj.LoadData();
        end
        
        function data = GetImage(obj, index)
            if obj.IsValid
                
            end
        end
        
        function ApplyChanges(obj)
            if ~isempty(obj.CellDefinition)
                obj.CellDefinition.ApplyChanges();
            end
        end
        
        function tf = HasPendingChanges(obj)
            if ~isempty(obj.CellDefinition)
                tf = obj.CellDefinition.HasPendingChanges();
            end
        end
        
        % Destructor
        function delete(obj)
           siman.DataCache.RemoveCachedData(obj.ImageDataCacheID);
           if ~isempty(obj.CellDefinition)
               delete(obj.CellDefinition);
               delete(obj.CellDefListener);
               obj.CellDefinition = [];
               obj.CellDefListener = [];
           end           
        end
    end
end