classdef (ConstructOnLoad) ProcessingDefinition < siman.SimanObject
    % Must be a subclass of handle
    properties (SetObservable, GetObservable)
        Name
        ApplyMask = true
        MaskValue = 0;
    end
    properties
        OperationsList
        LastCellDefinition
        IsSystemDefinition = false;
    end
    properties (Transient = true)
        OperationsListListener
        OperationListeners
        
        LastProcessedSource = [];
        CancelRequested
        IsValid
        ValidationMessage
        ErrorMessage
        IsRGB = false;
        
        BaselineOperation
        NeedsValidating = true;
        
        Baseline
        BaselineStandardDeviation
        
        ImageMask
        
        ActiveOperationList
        IsActiveOperationListUpToDate = false;
    end
    properties (Dependent = true)
        OperationCount
        IsEditable
    end
    
    events
        DefinitionChange
        OperationsChange
    end
    
    methods % Properties
        function set.OperationsList(obj, value)
            obj.IsActiveOperationListUpToDate = false;
            if isempty(value) % internal code for nulling the property during a copy
                obj.OperationsList = [];
                obj.OperationsListListener = [];
                obj.OperationListeners = [];
                return
            end
            if ~isempty(obj.OperationsList)
                delete(obj.OperationsList);
                if ~isempty(obj.OperationsListListener)
                    delete(obj.OperationsListListener)
                end
                if ~isempty(obj.OperationListeners)
                    delete(obj.OperationListeners)
                end
            end
            obj.OperationsList = value;
            if ~isempty(value)
                obj.OperationsList.DeleteElementsOnClear = true;
                obj.OperationsList.DeleteElementsOnDelete = true;
                obj.OperationsListListener = addlistener(obj.OperationsList, 'CollectionChange', @obj.OnListChanged);
                obj.InitOperationListeners();
            end
        end
        function count = get.OperationCount(obj)
            if isempty(obj.OperationsList)
                count = 0;
            else
                count = obj.OperationsList.Count;
            end
        end
        function result = get.IsEditable(obj)
            result = ~strcmp(obj.Name, 'raw');
        end
        function result = get.IsValid(obj)
            ops = obj.GetActiveOperations();
            result = false;
            count = ops.Count;
            for i = 1:count
                op = ops.ElementAt(i);
                if ~op.IsValid
                    return
                end
            end
            result = true;
        end
    end
    
    methods
        % Constructor
        function obj = ProcessingDefinition()
            obj@siman.SimanObject();
            
            if isempty(obj.OperationsList)
                obj.OperationsList = siman.ObservableCollection;
            end
            obj.InitListeners();
            obj.InitOperationListeners();
            obj.NeedsValidating = true;
        end
        
        function InitListeners(obj)
            obj.SelfListener = addlistener(obj, 'PropertyChanged', @obj.OnPropertyChanged);
        end
        
        function InitOperationListeners(obj)
            if ~isempty(obj.OperationListeners)
                delete(obj.OperationListeners);
                obj.OperationListeners = [];
            end
            if isempty(obj.OperationsList) || obj.OperationsList.Count == 0
                return
            end
            array = obj.OperationsList.ToArray();
            listeners = addlistener(array, 'OperationChange', @obj.OnOperationChanged);
            obj.OperationListeners = listeners;
        end
        
        function OnPropertyChanged(obj, src, evnt)
            switch evnt.PropertyName
                case {'ApplyMask', 'MaskValue'}
                    obj.FireDefinitionChange('prop', evnt.PropertyName);
            end
        end
        function OnListChanged(obj, src, evnt)
            obj.IsActiveOperationListUpToDate = false;
            obj.NeedsValidating = true;
            obj.InitOperationListeners();
            obj.FireDefinitionChange(evnt.Type, evnt.Param);
            obj.FireOperationsChange(evnt.Type, evnt.Param);
        end
        function OnOperationChanged(obj, src, evt)
            obj.IsActiveOperationListUpToDate = false;
            obj.NeedsValidating = true;
            obj.ClearStoredResults(src);
            %TODO don't change if not active and property wasn't "Skip"
            obj.FireDefinitionChange('operation', []);
        end
        
        function ApplyChanges(obj)
            if obj.NeedsValidating
                obj.Validate();
                obj.NeedsValidating = false;
            end
        end
        
        function tf = HasPendingChanges(obj)
            tf = obj.NeedsValidating;
        end
        
        function op = AddOperation(obj, type)
            if nargin < 2
                type = siman.ProcessingOperation.Types{1};
            end
            op = siman.ProcessingOperation();
            op.Type = type;
            obj.OperationsList.Add(op);
            %addlistener(op, 'OperationChange', @obj.OnOperationChanged);
        end
        function op = InsertNewOperation(obj, op_to_follow, type)
            if nargin < 3
                type = siman.ProcessingOperation.Types{1};
            end
            index = obj.OperationsList.IndexOf(op_to_follow);
            if isempty(index) || index == obj.OperationCount
                op = obj.AddOperation(type);
                return;
            end
            op = siman.ProcessingOperation();
            op.Type = type;
            obj.OperationsList.Insert(index + 1, op);
            obj.ClearStoredResults(op);
            %addlistener(op, 'OperationChange', @obj.OnOperationChanged);
        end
        function RequestNewOperation(obj, op_to_follow)
            options = siman.ProcessingOperation.Types;
            currentType = 1;
            [selection,ok] = listdlg('ListString', options, 'SelectionMode', 'single', ...
                'InitialValue', currentType);
            if ok
                obj.InsertNewOperation(op_to_follow, options{selection});
            end
        end
        function RemoveOperation(obj, op)
            if isempty(op)
                return
            end
            obj.ClearStoredResults(op);
            obj.OperationsList.Remove(op);
            delete(op);
        end
        function MoveOperationUp(obj, op)
            index = obj.OperationsList.IndexOf(op);
            if isempty(index) || index < 2 || index > obj.OperationCount
                return;
            end
            obj.OperationsList.Swap(index, index-1);
            obj.ClearStoredResults(op);
        end
        function MoveOperationDown(obj, op)
            index = obj.OperationsList.IndexOf(op);
            if isempty(index) || index < 1 || index > obj.OperationCount - 1
                return;
            end
            obj.ClearStoredResults(op);
            obj.OperationsList.Swap(index, index+1);
        end
        function ChangeOperationType(obj, op)
            options = siman.ProcessingOperation.Types;
            currentType = find(strcmp(op.Type, options));
            [selection,ok] = listdlg('ListString', options, 'SelectionMode', 'single', ...
                'InitialValue', currentType);
            if ok && selection ~= currentType
                op.Type = options{selection};
            end
        end
        
        function op = GetOperation(obj, index)
            op = [];
            if index <= obj.OperationCount
                op = obj.OperationsList.ElementAt(index);
            end
        end
        function op = FindOperationByType(obj, type_to_find)
            op = [];
            count = obj.OperationCount;
            for i = 1:count
                nextOp = obj.OperationsList.ElementAt(i);
                if strcmp(nextOp.Type, type_to_find)
                    op = nextOp;
                    return
                end
            end
        end
        function list = GetActiveOperations(obj)
            if obj.IsActiveOperationListUpToDate
                list = obj.ActiveOperationList;
                return;
            end

            list = siman.List;
            if ~isempty(obj.OperationsList) && obj.OperationsList.Count > 0
                count = obj.OperationsList.Count;
                for i = 1:count
                    op = obj.OperationsList.ElementAt(i);
                    if op.Active
                        list.Add(op);
                    end
                    if op.Stop
                        break;
                    end
                end
            end
            obj.IsActiveOperationListUpToDate = true;
            obj.ActiveOperationList = list;
        end
        
        function result = GetOutputSize(obj, input)
            if isempty(input)
                result = [];
                return
            end
            
            result = input.Size;
            if isempty(result)
                return
            end
            
            list = obj.GetActiveOperations().ToArray();
            count = numel(list);
            for i = 1:count
                op = list(i);
                if strcmp(op.Type, 'restore original')
                    result = input.Size;
                else
                    result = op.GetOutputSize(result);
                end
                if isempty(result)
                    return
                end
            end
        end
        
        function output_units = GetOutputUnits(obj)
            output_units = 'raw';
            
            list = obj.GetActiveOperations();
            count = list.Count;
            for i = 1:count
                op = list.ElementAt(i);
                if strcmp(op.Type, 'restore original')
                    output_units = 'raw';
                else
                    output_units = op.GetOutputUnits(output_units);
                end
            end
        end
        function ClearStoredResults(obj, start_op)
            if nargin < 2 || isempty(start_op)
                clear = true;
            else
                clear = false;
            end
            list = obj.GetActiveOperations();
            count = list.Count;
            for i = 1:count
                op = list.ElementAt(i);
                if ~clear
                    if op == start_op
                        clear = true;
                    end
                end
                if clear
                    op.ClearStoredResults();
                end
            end
        end
        function FireDefinitionChange(obj, type, varargin)
            if nargin > 2
                notify(obj, 'DefinitionChange', siman.DefinitionChangeEventData(type, varargin{:}));
            else
                notify(obj, 'DefinitionChange', siman.DefinitionChangeEventData(type, []));
            end
        end
        function FireOperationsChange(obj, type, varargin)
            if nargin > 2
                notify(obj, 'OperationsChange', siman.DefinitionChangeEventData(type, varargin{:}));
            else
                notify(obj, 'OperationsChange', siman.DefinitionChangeEventData(type, []));
            end
        end
        
        function delete(obj)
            if ~isempty(obj.OperationsList)
                delete(obj.OperationsList);
                delete(obj.OperationsListListener);
                delete(obj.OperationListeners);
                obj.OperationsList = [];
            end
        end
    end
    
    methods(Access = protected)
        % Override copyElement method:
        function cpObj = copyElement(obj)
            % Make a shallow copy of all properties
            cpObj = copyElement@siman.SimanObject(obj);
            cpObj.InitListeners();
            % Make a deep copy of the definitions
            cpObj.OperationsList = []; % Don't let object get deleted due to following set
            cpObj.OperationsList = copy(obj.OperationsList);
            if ~isempty(cpObj.LastCellDefinition)
                cpObj.LastCellDefinition = copy(obj.LastCellDefinition);
            end
            % Remove stored data
            cpObj.Baseline = [];
            cpObj.BaselineStandardDeviation = [];
            cpObj.ImageMask = [];
        end
    end
end