classdef ROIContainer < handle
    % Must be a subclass of handle
    properties
        ROIList
    end
    properties (Dependent = true)
        HasROIs
        ROICount
        IsTransientSource
    end
    properties (Transient)
        ROIListListeners
        LastRequestedROIType
        LastRequestedROIList
    end
    
    events
        ROIListChange %TODO rename to ROIListChange
        ROIItemChange
    end
    
    methods % Properties
        function set.ROIList(obj, value)
            if isempty(value) % internal code for nulling the property during a copy
                obj.ROIList = [];
                obj.ROIListListeners = [];
                return
            end
            if ~isempty(obj.ROIList)
                delete(obj.ROIList);
                if ~isempty(obj.ROIListListeners)
                    delete(obj.ROIListListeners)
                end
            end
            obj.ROIList = value;
            if ~isempty(value)
                obj.ROIList.DeleteElementsOnClear = true;
                obj.ROIList.DeleteElementsOnDelete = true;
                obj.ROIList.NotifyOnItemChange = true;
                obj.ROIListListeners = addlistener(obj.ROIList, 'CollectionChange', @obj.OnROIListChanged);
                obj.ROIListListeners(2) = addlistener(obj.ROIList, 'ItemChange', @obj.OnROIItemChanged);
            end
        end
        function count = get.ROICount(obj)
            count = obj.ROIList.Count;
        end
        function result = get.HasROIs(obj)
            result = obj.ROIList.Count > 0;
        end
    end
    
    methods
        % Constructor
        function obj = ROIContainer()
            obj.ROIList = siman.ObservableCollection;
        end
        
        function AddROI(obj, roi)
            obj.ROIList.Add(roi);
        end
        function AddROIs(obj, list)
            obj.ROIList.AppendList(list);
        end
        function RemoveROI(obj, roi)
            delete(roi);
            obj.ROIList.Remove(roi);
        end
        function RemoveSelectedROIs(obj, type)
            indexes = [];
            for i = 1:obj.ROIList.Count
                roi = obj.ROIList.ElementAt(i);
                if roi.IsSelected && (isa(roi, type) || strcmp(type, roi.Type))
                	indexes(end + 1) = i;
                end
            end
            obj.ROIList.RemoveIndexes(indexes);
        end
        function ClearAllROIs(obj)
            obj.ROIList.Clear();
        end
        function ClearROIType(obj, type)
            indexes = [];
            for i = 1:obj.ROIList.Count
                roi = obj.ROIList.ElementAt(i);
                if isa(roi, type) || strcmp(type, roi.Type)
                	indexes(end + 1) = i;
                end
            end
            obj.ROIList.RemoveIndexes(indexes);
        end
        function roi = GetROI(obj, index)
            roi = [];
            if index <= obj.ROIList.Count
                roi = obj.ROIList.ElementAt(index);
            end
        end
        function list = GetROIType(obj, type)
            if (~isempty(obj.LastRequestedROIType) && strcmp(type, obj.LastRequestedROIType))
                list = obj.LastRequestedROIList;
                return;
            end
            obj.LastRequestedROIType = type;
            list = siman.List;
            count = obj.ROIList.Count;
            for i = 1:count
                roi = obj.ROIList.ElementAt(i);
                if isa(roi, type) || strcmp(type, roi.Type)
                    list.Add(roi);
                end
            end
            obj.LastRequestedROIList = list;
        end
        function list = GetSelectedROIType(obj, type)
            list = siman.List;
            count = obj.ROIList.Count;
            array = obj.ROIList.ToArray();
            for i = 1:count
                roi = array(i);
                if isa(roi, type) || strcmp(type, roi.Type)
                    if roi.IsSelected
                        list.Add(roi);
                    end
                end
            end
        end
        function SelectROIs(obj, rois_to_select, type)
            fullList = obj.ROIList;
            count = fullList.Count;
            if fullList.Count == 0
                return
            end
            isSelected = zeros(count, 2);
            for i = 1:count
                roi = fullList.ElementAt(i);
                isSelected(i, [1 2]) = roi.IsSelected;
                if isa(rois_to_select, 'siman.ObservableCollection') || isa(rois_to_select, 'siman.List')
                    isMatch = rois_to_select.IsMember(roi);
                else
                    isMatch = isequal(roi, rois_to_select);
                end
                switch (type)
                    case 'select'
                        isSelected(i,2) = isMatch;
                    case 'add'
                        isSelected(i,2) = isMatch || isSelected(i,2);
                    case 'swap'
                        if isMatch
                            isSelected(i,2) = ~isSelected(i,2);
                        end
                end
            end
            for i = 1:count
                if isSelected(i,1) ~= isSelected(i,2)
                    roi = fullList.ElementAt(i);
                    roi.IsSelected = isSelected(i,2);
                end
            end
        end
        function OnROIListChanged(obj, src, evnt)
            obj.FireROIListChange(evnt.Type, evnt.Param);
            obj.LastRequestedROIType = [];
            if ~isempty(obj.LastRequestedROIList)
                if isvalid(obj.LastRequestedROIList)
                    delete(obj.LastRequestedROIList);
                    obj.LastRequestedROIList = [];
                end
            end
        end
        function OnROIItemChanged(obj, src, evnt)
            obj.FireROIItemChange([]);
        end
        
        function FireROIListChange(obj, type, param)
            notify(obj, 'ROIListChange');
        end
        function FireROIItemChange(obj, roi)
            notify(obj, 'ROIItemChange');
        end
        
        function result = GetExportDataMatrix(obj, list)
            count = list.Count;
            result = [];
            if count == 0
                return
            end
            
            roi = list.ElementAt(1);
            vector = roi.GetExportData();
            vector = vector(:); % ensure row vector
            
            result = zeros(numel(vector), count);
            result(:,1) = vector;
            for i = 2:count
                roi = list.ElementAt(i);
                vector = roi.GetExportData();
                if ~isempty(vector)
                    result(:,i) = vector;
                end
            end
        end
        
        function result = GetExportPropCellArray(obj, list, include_labels)
            count = list.Count;
            result = {};
            if count == 0
                return
            end
            if nargin < 3
                include_labels = true;
            end
            
            result = cell(count+include_labels, 1);
            for i = 1:count
                roi = list.ElementAt(i);
                result{i+include_labels} = siman.UIHelper.WriteLine(roi.GetPropExportString());
            end
            if include_labels
                labels = roi.GetExportPropList();
                str = 'Name';
                for i = 1:length(labels)
                    str = [str ',' labels{i}];
                end
                result{1} = siman.UIHelper.WriteLine(str);
            end
        end
        function result = GetExportDataStrings(obj, list, include_labels)
            count = list.Count;
            result = {};
            if count == 0
                return
            end
            if nargin < 3
                include_labels = true;
            end
            
            data = obj.GetExportDataMatrix(list);
            dim = size(data);
            vectorLength = dim(1);
            result = cell(vectorLength+include_labels, 1);
            for row = 1:vectorLength
                str = num2str(data(row,1));
                for i = 2:count
                    str = [str ',' num2str(data(row,i))];
                end
                result{row+include_labels} = siman.UIHelper.WriteLine(str);
            end
            if include_labels
                roi = list.ElementAt(1);
                str = roi.Name;
                for i = 2:count
                    roi = list.ElementAt(i);
                    str = [str ',' roi.Name];
                end
                result{1} = siman.UIHelper.WriteLine(str);
            end
        end
        
        function str = GetPropExportText(obj, list, include_labels)
            if nargin < 3
                include_labels = true;
            end
            result = obj.GetExportPropCellArray(list, include_labels);
            str = '';
            for i = 1:length(result)
                str = [str result{i}];
            end
        end
        function str = GetDataExportText(obj, list, include_labels)
            if nargin < 3
                include_labels = true;
            end
            result = obj.GetExportDataStrings(list, include_labels);
            str = '';
            for i = 1:length(result)
                str = [str result{i}];
            end
        end
        
        function CopyROIData(obj, list, include_labels)
            if nargin < 3
                include_labels = true;
            end
            result = obj.GetDataExportText(list, include_labels);
            clipboard('copy', result);
        end
        function CopyROIProps(obj, list, include_labels)
            if nargin < 3
                include_labels = true;
            end
            result = obj.GetPropExportText(list, include_labels);
            clipboard('copy', result);
        end
        
        
        function delete(obj)
            if ~isempty(obj.ROIList)
                delete(obj.ROIList);
                delete(obj.ROIListListeners);
                obj.ROIList = [];
            end
        end
    end
end