function [list, message] = Detect(obj, target)

method = obj.DetectionMethod;
switch method
    case 'detection image'
        [list, message] = DetectStandard(obj, target);
    otherwise
        list = siman.List;
        message = ['Unknown detection method: ' obj.DetectionMethod];
end
end

function [list, message] = DetectStandard(obj, target)
    list = siman.List;
    if isempty(target)
        message = 'No image target';
        return;
    end
    message = '';
    
    unitsDef = obj.GetUnitsDefinition();
    if isempty(unitsDef)
        message = ['Bad units definition name: ' obj.UnitsDefinitionName];
        return;
    end
    
    detectionDef = obj.DetectionDefinition;
    units = detectionDef.GetOutputUnits();
    if ~strcmp(units, 'binary')
        message = 'Detection image is not binary';
        return
    end
    
    try
        data = detectionDef.ProcessSource(target);
    catch ex
        message = 'Error during creation of detection image';
        return;
    end
    
    connectivityType = str2double(obj.ConnectivityType);
    result = bwconncomp(data, connectivityType);
    
    rawCount = result.NumObjects;
    if result.NumObjects == 0
        return
    end
    
    % Filter out features that are too big or too small
    rawIndexList = result.PixelIdxList;
    validFeatures = cell(rawCount, 2);
    minSize = obj.MinSize;
    maxSize = obj.MaxSize;
    validCount = 0;
    for i = 1:rawCount
        len = numel(rawIndexList{i});
        if len > minSize && len < maxSize
            validCount = validCount + 1;
            validFeatures{validCount} = rawIndexList{i};
        end
    end

    if validCount == 0
        return
    end
    
    % We have features, so get the image where the peak can be taken from (different units)
    try
        unitData = unitsDef.ProcessSource(target);
    catch
        message = 'Error during creation of image for units';
        return
    end
    if ~isequal(size(unitData), size(data))
        message = 'Unequal sizes between detection and unit images';
        return
    end
    
    dim = size(data);
    width = dim(2);
    height = dim(1);
    frameCount = dim(3);
    thumbZStartOffset = obj.ThumbnailStartOffset;
    thumbZStopOffset = obj.ThumbnailStopOffset;
    thumbCrosssectionRadius = round(obj.ThumbnailWidth/2);
    traceCrosssectionRadius = round(obj.TraceWidth/2);
    traceLength = thumbZStopOffset - thumbZStartOffset + 1;
    
    maxSparkletCount = obj.MaxCount;
    if (validCount > maxSparkletCount)
        message = ['Too many sparklets detected (' num2str(validCount) ')'];
        return
    end
        
    
    % Create feature objects        
    for i = 1:validCount
        indexes = validFeatures{i};
        values = unitData(indexes);
        [peakValue, index] = max(values);
        peakIndex = indexes(index);
        [xIndexes, yIndexes, zIndexes] = xyindexes(indexes, [height width frameCount]);
      
        feature = siman.Sparklet();
        feature.Name = ['Sparklet' num2str(i)];
        feature.Index = i;
        feature.PeakIndex = peakIndex;
        feature.PeakXIndex = xIndexes(index);
        feature.PeakYIndex = yIndexes(index);
        feature.PeakZIndex = zIndexes(index);
        feature.PeakAmplitude = peakValue;
        feature.Area = numel(indexes);
        
        feature.LowerXBound = min(xIndexes);
        feature.UpperXBound = max(xIndexes);
        feature.LowerYBound = min(yIndexes);
        feature.UpperYBound = max(yIndexes);
        feature.LowerZBound = min(zIndexes);
        feature.UpperZBound = max(zIndexes);
        feature.FrameCount = feature.UpperZBound - feature.LowerZBound + 1;
        
        % Calculate some nice stored items (waveform through peak and image thumbnail)
        trace = zeros(traceLength, 1)*nan;
        startZ = max(1, feature.PeakZIndex + thumbZStartOffset);
        stopZ = min(frameCount, feature.PeakZIndex + thumbZStopOffset);
        if (feature.PeakZIndex + thumbZStartOffset < 1)
            startZ_trace = 1 - (feature.PeakZIndex + thumbZStartOffset);
        else
            startZ_trace = 1;
        end
        stopZ_trace = startZ_trace + (stopZ-startZ+1) - 1;
        startY = max(1, feature.PeakYIndex - traceCrosssectionRadius);
        stopY = min(height, feature.PeakYIndex + traceCrosssectionRadius);
        startX = max(1, feature.PeakXIndex - traceCrosssectionRadius);
        stopX = min(width, feature.PeakXIndex + traceCrosssectionRadius);
        traceData = unitData(startY:stopY, startX:stopX, startZ:stopZ);
        zLength = stopZ - startZ + 1;
        trace(startZ_trace:stopZ_trace) = mean(reshape(traceData, [numel(traceData)/zLength, zLength]));
        startX = max(1, feature.PeakXIndex - thumbCrosssectionRadius);
        stopX = min(width, feature.PeakXIndex + thumbCrosssectionRadius);
        startY = max(1, feature.PeakYIndex - thumbCrosssectionRadius);
        stopY = min(height, feature.PeakYIndex + thumbCrosssectionRadius);
        thumbnail = unitData(startY:stopY, startX:stopX, feature.PeakZIndex);
        
        feature.WaveformStartIndex = startZ;
        feature.WaveformStopIndex = stopZ;
        feature.WaveformStartOffset = thumbZStartOffset;
        feature.WaveformStopOffset = thumbZStopOffset;
        feature.Thumbnail = thumbnail;
        feature.Waveform = trace;
        feature.RegionIndexes = indexes;
        feature.RegionXIndexes = xIndexes;
        feature.RegionYIndexes = yIndexes;
        feature.RegionZIndexes = zIndexes;
        
        for z = feature.LowerZBound:feature.UpperZBound
            frameIndexes = find(zIndexes == z);
            [~, highestIndex] = min(indexes(frameIndexes));
            topCol = xIndexes(frameIndexes(highestIndex));
            topRow = yIndexes(frameIndexes(highestIndex));
            contour = bwtraceboundary(data(:,:,z), [topRow, topCol], 'NE');
            feature.Outlines{z - feature.LowerZBound + 1} = contour;
        end
        
        feature.CalculateExtraProperties(obj);
        %target.AddFeature(feature);
        list.Add(feature);
    end
    target.AddFeatures(list);
    disp(['Detected ' num2str(validCount) ' sparklets']);
end




