function [list, message] = Detect(obj, target)

method = obj.DetectionMethod;
switch method
    case 'derivative'
        [list, message] = DetectDerivative(obj, target);
    otherwise
        list = siman.List;
        message = ['Unknown detection method: ' type];
end
end

function [list, message] = DetectDerivative(obj, target)
    list = siman.List;
    if isempty(target)
        message = 'No image target';
        return;
    end
    message = '';
    
    width = target.Width;
    
    cellDef = target.CellDefinition;
    lowerEdge = cellDef.LowerEdge;
    upperEdge = cellDef.UpperEdge;
    if isinf(upperEdge)
        upperEdge = width;
    end
    
    unitsDef = obj.GetUnitsDefinition();
    if isempty(unitsDef)
        message = ['Bad units definition name: ' obj.UnitsDefinitionName];
        return;
    end
    
    detectionDef = obj.DetectionDefinition;
    try
        data = detectionDef.ProcessSource(target);
    catch
        message = 'Error during creation of detection image';
        return;
    end
    trace = mean(data(:, lowerEdge:upperEdge), 2);
    
    filtWidth = obj.DetectionMedianFilterWidth;
    if filtWidth > 1
        trace = medfilt1(trace, filtWidth);
    end
    
    detectionTrace = diff(trace);
    switch obj.DerivativeThresholdType
        case {'image units', 'data units'}
            threshold = obj.DerivativeThreshold_DataUnits;
        case 'percentile'
            percentile = obj.DerivativeThreshold_Percentile;
            if percentile < 1
                percentile = percentile * 100;
            end
            if numel(trace) < 20
                values = detectionTrace;
            else
                values = detectionTrace(11:end-10);
            end
            values = sort(values);
            index = round(percentile/100 * numel(values));
            threshold = values(index);
            obj.DerivativeThreshold_Actual = threshold;
    end
    indexes = find(detectionTrace > threshold) + 1;
    if isempty(indexes)
        return;
    end
    
    startOffset = obj.WaveformStartOffset;
    stopOffset = obj.WaveformStopOffset;
    indexes(indexes <= -startOffset) = [];
    indexes(indexes > (length(trace)-stopOffset)) = [];
    
    data = unitsDef.ProcessSource(target);
    trace = mean(data(:, lowerEdge:upperEdge), 2);
    filtWidth = obj.UnitsMedianFilterWidth;
    if filtWidth > 1
        trace = medfilt1(trace, filtWidth);
    end
    
    peakIndexes = indexes;
    for i = 1:length(indexes)
        idx = indexes(i);
        [null,peakIndex] = max(trace(idx+startOffset:idx+stopOffset));
        peakIndexes(i) = peakIndex + idx + startOffset - 1;
    end
    
    peakIndexes(peakIndexes <= -startOffset) = [];
    peakIndexes(peakIndexes > (length(trace)-stopOffset)) = [];

    limit = obj.RefractoryLimit;
    for i = length(peakIndexes):-1:2
        if peakIndexes(i) < peakIndexes(i-1)+limit
            peakIndexes(i) = -1;
        end
    end
    peakIndexes(peakIndexes == -1) = [];
    
    % Create feature objects
    feature = siman.TimeProfile();
    feature.Name = 'Transient';
    feature.Profile = trace;
    target.AddFeature(feature);
        
    for i = 1:length(peakIndexes)
        feature = siman.Transient();
        feature.Name = ['Transient' num2str(i)];
        feature.Index = i;
        feature.PeakIndex = peakIndexes(i);
        feature.PeakAmplitude = trace(peakIndexes(i));
        feature.LowerEdge = lowerEdge;
        feature.UpperEdge = upperEdge;
        feature.WaveformStartOffset = startOffset;
        feature.WaveformStopOffset = stopOffset;
        feature.Waveform = trace(peakIndexes(i)+startOffset:peakIndexes(i)+stopOffset);
        feature.CalculateExtraProperties(obj);
        list.Add(feature);
    end
    target.AddFeatures(list);
end