classdef ObservableDictionary < matlab.mixin.Copyable
    % Must be a subclass of handle
    properties
        Collection = {};
        Keys = {};
        AllowDuplicates = false;
        DeleteElementsOnClear = false;
        DeleteElementsOnDelete = false;
    end
    properties (Dependent = true)
        Count
    end
    events
        DictionaryChange
    end
    methods
        function obj = ObservableDictionary()
        end
        
        function count = get.Count(obj)
            count = numel(obj.Collection);
        end
        
        function Add(obj, key, item)
            if nargin < 3
                siman.Debug.Throw('ObservableDictionary', 'Missing item or key to add');
            end
            if isempty(key) || ~ischar(key)
                siman.Debug.Throw('ObservableDictionary', 'Bad key to add');
            end
            if ~obj.AllowDuplicates && ismember(key, obj.Keys)
                siman.Debug.Throw('ObservableDictionary', 'Duplicate key to add');
            end
            obj.Collection{end + 1} = item;
            obj.Keys{end + 1} = key;
            obj.FireDictionaryChange('add', item); %TODO add key to change event
        end
        
        function RemoveKey(obj, key)
            index = obj.IndexOf(key);
            if isempty(index)
                return;
            end
            count = obj.Count;
            for i = index+1:count
                obj.Collection(i-1) = obj.Collection(i);
                obj.Keys(i-1) = obj.Keys(i);
            end
            obj.Collection = obj.Collection(1:end-1);
            obj.Keys = obj.Keys(1:end-1);
            obj.FireDictionaryChange('remove', item); %TODO add key to change event
        end
        
        function item = Item(obj, key)
            index = obj.IndexOf(key);
            if isempty(index)
                siman.Debug.Throw('ObservableDictionary', ['Key does not exist: ' key]);
            end
            item = obj.Collection{index};
        end
        
        function [item, key] = ElementAt(obj, index)
            if obj.Count < index
                siman.Debug.Throw('IndexOutOfBounds', 'Attempt to access collection out of bounds.');
            end
            item = obj.Collection{index};
            key = obj.Keys{index};
        end
                
        function Replace(obj, key, element)
            index = obj.IndexOf(key);
            if isempty(key)
                siman.Debug.ReportProblem(['Attempted to replace non-existing key: ' key]);
                return;
            end
            if obj.Count < index || index < 1
                siman.Debug.Throw('IndexOutOfBounds', 'Attempt to access collection out of bounds.');
            end
            obj.Collection{index} = element;
            obj.FireDictionaryChange('replace', index);
        end
        
        function index = IndexOf(obj, key)
            count = obj.Count;
            index = [];
            if isempty(key) || ~ischar(key)
                return;
            end
            for i = 1:count
                if strcmp(obj.Keys{i}, key)
                    index = i;
                    return;
                end
            end
        end
        
        function found = IsKey(obj, key)
            found = ~isempty(obj.IndexOf(key));
        end
        
        function Clear(obj)
            if obj.DeleteElementOnClear
                for i = 1:length(obj.Collection)
                    item = obj.Collection{i};
                    if ~isempty(item) && isobject(item)
                        delete(item);
                    end
                end
            end
            obj.Collection = {};
            obj.Keys = {};
            obj.FireDictionaryChange('remove all');
        end
        
        function delete(obj)
        end
        
        % Removed since subsref overloading also overloads method calling
        % (can't figure out how to get around this)
%         function result = subsref(obj, format)
%             %subsref - Overload to allow direct subscripting of collections
%             if ~strcmp(format(1).type, '()')
%                 error('Siman:BadCollectionSyntax', 'Bad indexing for collection.  Only () supported.')
%             end
%             
%             % obj(:).fieldname syntax (returns a cell array of values and errors if
%             % not all objects have that field)
%             if ischar(format(1).subs) && strcmp(format(1).subs, ':')
%                 if length(format) == 2
%                     error('Siman:BadCollectionSyntax', 'Bad indexing for collection.  Only (:).(field) supported.')
%                 end
%                 if ~strcmp(format(2).type, '.')
%                     error('Siman:BadCollectionSyntax', 'Bad indexing for collection.  Only (:).(field) supported.')
%                 end
%                 fieldname = format(2).subs{1};
%                 count = obj.Count;
%                 result = cell(count);
%                 for i = 1:count
%                     element = obj.ElementAt(i);
%                     if ~isprop(element, fieldname)
%                         error('Siman:InvalidCollectionElementProp', ['Collection element does not have property:' fieldname])
%                     end
%                     result{i} = element.(fieldname);
%                 end
%             end
%             result = obj.ElementAt(format(1).subs{1});
%             if length(format) > 1
%                 result = subsref(result, format(2:end));
%             end
%         end
        
        function FireDictionaryChange(obj, type, varargin)
            if nargin > 2
                notify(obj, 'DictionaryChange', siman.CollectionChangeEventData(type, varargin{:}));
            else
                notify(obj, 'DictionaryChange', siman.CollectionChangeEventData(type));
            end
        end
    end
    
    methods(Access = protected)
        % Override copyElement method:
        function cpObj = copyElement(obj)
            % Make a shallow copy of all properties
            cpObj = copyElement@matlab.mixin.Copyable(obj);
            % Make a deep copy of the collection
            for i = 1:length(cpObj.Collection)
                cpObj.Collection{i} = copy(cpObj.Collection{i});
            end
        end
    end
end
