classdef ProcessingDefPanel < siman.ToolPanel
    % DisplayWindow
    properties (SetObservable)
        LastOperationName
        LastDefinitionName
    end
    properties (Transient)
        OperationSubpanelID
        MainSubpanelID
        MessageAxesID
        
        ProcessingDef
        ProcessingDefListener
        
        NeedsOperationLayout = true;
        CurrentOperation
        CurrentOperationIndex
        OperationListener
    end
    
    methods
        % Constructor
        function obj = ProcessingDefPanel(parent)
            obj@siman.ToolPanel(parent);
        end
        
        function InitDefaults(obj)
            obj.Name = 'Image Processing';
            obj.Size = [225 250];
            obj.DockedLocation = 'south';
        end
        
        function InitPanel(obj)
            InitPanel@siman.ToolPanel(obj);
            
            %obj.MakeControls();
        end
        
        function OnAttachedPlotChanged(obj, type)
            obj.ResetCurrentDefinition();
        end
        
        function ResetCurrentDefinition(obj)
            if isempty(obj.CurrentAttachedPlot)
                if ~isempty(obj.ProcessingDef)
                    obj.SetProcessingDef([]);
                end
                return;
            end
            def = obj.CurrentAttachedPlot.CurrentProcessingDef;
            obj.SetProcessingDef(def);
        end
        
        function SetProcessingDef(obj, def)
            if isempty(def) && isempty(obj.ProcessingDef)
                return;
            end
            if length(def) ~= length(obj.ProcessingDef) || def ~= obj.ProcessingDef
                if ~isempty(obj.ProcessingDefListener)
                    delete(obj.ProcessingDefListener);
                    obj.ProcessingDefListener = [];
                end
                obj.ProcessingDef = def;
                if ~isempty(def)
                    obj.ProcessingDefListener = addlistener(def, 'DefinitionChange', @obj.OnDefinitionChanged);
                end
                obj.SetPanelDataSource(def);
                obj.InitProcessingDef();
                obj.QueueLayout();
            end
        end
        
        function OnDefinitionChanged(obj, ~, event)
            % Check for deletion of current operation
            index = obj.ProcessingDef.OperationsList.IndexOf(obj.CurrentOperation);
            if isempty(index)
                if (obj.ProcessingDef.OperationCount == 0)
                    op = [];
                else
                    op = obj.ProcessingDef.OperationsList.ElementAt(1);
                end
                obj.SetCurrentOperation(op);
            end
            if strcmp(event.Type, 'add')
                obj.SetCurrentOperation(event.Param);
            end
            obj.QueueRedraw();
        end
        
        function OnOperationChanged(obj, src, evt)
            obj.QueueRedraw();
        end
        
        function OnOperationLayoutChanged(obj, src, evt)
            obj.QueueOperationLayout();
        end
        
        function InitProcessingDef(obj)
            def = obj.ProcessingDef;
            obj.QueueOperationLayout();
            if isempty(def)
                op = [];
            else
                lastOpName = obj.LastOperationName;
                op = def.FindOperationByType(lastOpName);
                if isempty(op) && def.OperationCount > 0
                    op = def.GetOperation(1);
                    obj.LastOperationName = op.Type;
                end
            end
            obj.SetCurrentOperation(op);
        end
        
        function SetCurrentOperation(obj, op)
            if isempty(op) && isempty(obj.CurrentOperation)
                return;
            end
            if length(op) ~= length(obj.CurrentOperation) || op ~= obj.CurrentOperation
                if ~isempty(obj.OperationListener)
                    delete(obj.OperationListener);
                    obj.OperationListener = [];
                end
                obj.CurrentOperation = op;
                if ~isempty(op)
                    obj.OperationListener = addlistener(op, 'OperationChange', @obj.OnOperationChanged);
                    obj.OperationListener(2) = addlistener(op, 'OperationLayoutChange', @obj.OnOperationLayoutChanged);
                end
                obj.QueueOperationLayout();
                obj.QueueRedraw();
            end
        end
        
        function QueueOperationLayout(obj)
            obj.NeedsOperationLayout = true;
        end
        
        function ApplyChanges(obj)
            %ApplyChanges@siman.ToolPanel(obj);  cannot do this since operation layout needs to come before redraw
            if obj.NeedsLayout
                obj.ResetLayout();
                obj.NeedsLayout = false;
            end
            if obj.NeedsOperationLayout
                obj.ResetOperationControls();
                obj.NeedsOperationLayout = false;
            end
            if obj.NeedsRedraw
                obj.Redraw();
                obj.NeedsRedraw = false;
            end
        end
        
        function enabling = GetEnabling(obj, id, prop, context)
            switch prop
                case {'Name'}
                    enabling = ~isempty(obj.ProcessingDef) && obj.ProcessingDef.IsEditable;
                otherwise
                    enabling = GetEnabling@siman.ToolPanel(obj, id, prop, context);
            end
        end
        
        function UpdateControl(obj, id) % Handle manual refreshing
            tag = get(id, 'Tag');
            switch tag
                case 'DefinitionList'
                    obj.RefreshDefinitionList(id);
                    enabling = ~isempty(obj.ProcessingDef);
                case 'OperationList'
                    obj.RefreshOperationList(id);
                    enabling = ~isempty(obj.ProcessingDef) && obj.ProcessingDef.OperationCount > 0;
                case 'DefinitionListLabel'
                    enabling = ~isempty(obj.ProcessingDef);
                case 'OperationListLabel'
                    enabling = ~isempty(obj.ProcessingDef) && obj.ProcessingDef.OperationCount > 0;
                case 'ProcessingDef.SkipOperation'
                    enabling = ~isempty(obj.CurrentOperation);
                    if ~isempty(obj.CurrentOperation)
                        set(id, 'Value', obj.CurrentOperation.Skip);
                    end
                case 'ProcessingDef.StopOperation'
                    enabling = ~isempty(obj.CurrentOperation);
                    if ~isempty(obj.CurrentOperation)
                        set(id, 'Value', obj.CurrentOperation.Stop);
                    end
                otherwise
                    enabling = 'off';
                    siman.Debug.ReportProblem(['Unknown control to manual UpdateControl: ' tag]);
            end
            if ~ischar(enabling)
                if enabling
                    enabling = 'on';
                else
                    enabling = 'off';
                end
            end
            set(id, 'Enable', enabling);
        end
        
        function RefreshDefinitionList(obj, id)
            def = obj.ProcessingDef;
            if isempty(def)
                set(id, 'String', ' ');
                return
            end
            plot = obj.CurrentAttachedPlot;
            if isempty(plot)
                set(id, 'String', '');
                return
            end
             
            mgr = siman.ImageManager.Instance;
            setNames = mgr.GetDefinitionNameOptions();
            index = find(strcmp(obj.ProcessingDef.Name, setNames));
            set(id, 'String', setNames, 'Value', index);
        end
        
        function RefreshOperationList(obj, id)
            def = obj.ProcessingDef;
            if isempty(def) || obj.ProcessingDef.OperationCount == 0
                set(id, 'String', '', 'Value', 1);
                return
            end
            stopped = false;
            count = def.OperationCount;
            displayStrings = cell(1, count);
            index = 0;
            currentOp = obj.CurrentOperation;
            for i = 1:count
                op = def.GetOperation(i);
                str = op.GetDescription();
                if ~isempty(currentOp) && currentOp == op
                    index = i;
                end
                if ~op.Active
                    str = ['<skip> ' str];
                elseif op.Stop
                    str = ['<stop> ' str];
                end
                if stopped
                    str = ['------ ' str];
                end
                if ~op.IsValid
                    str = ['!! ' str];
                end
                if ~stopped
                    stopped = op.Stop;
                end
                displayStrings{i} = str;
            end
            if index == 0
                index = 1;
                siman.Debug.ReportProblem('Did not find current operation in current definition');
            end
            set(id, 'String', displayStrings, 'Value', index);
        end
        
        function ControlChanged(obj, control, data_obj, prop)
            switch prop
                case 'DefinitionList'
                    strings = get(control, 'String');
                    value = get(control, 'Value');
                    obj.ChangeDefinition(strings{value});
                case 'OperationList'
                    value = get(control, 'Value');
                    op = obj.ProcessingDef.GetOperation(value);
                    obj.SetCurrentOperation(op);
                otherwise
                    ControlChanged@siman.ToolPanel(obj, control, data_obj, prop);
            end
        end
        
        function ChangeDefinition(obj, new_name)
            plot = obj.CurrentAttachedPlot;
            plot.SetDefinition(new_name);
        end
                
        function delete(obj)
            delete(obj.ProcessingDefListener);
            obj.ProcessingDefListener = [];
            delete(obj.OperationListener);
            obj.OperationListener = [];
        end
    end
end