function result = PerformOperation(obj, input, def)

import siman.*;

dim = size(input);
width = dim(2);
height = dim(1);
if numel(dim) > 2
    frameCount = dim(3);
else
    frameCount = 1;
end

type = obj.Type;
switch lower(type)
    case 'subtract background'
        result = input - obj.BackgroundValue;

    case 'median filter'
        switch obj.MedianFilterType
            case 'equal sides'
                val = obj.MedianFilterWidth;
                result = input;
                for i = 1:frameCount
                    result(:,:,i) = medfilt2(input(:,:,i), [val val]);
                    if (def.CancelRequested)
                        return
                    end
                end
        end
        
    case 'calculate baseline'
        switch obj.BaselineFormat
            case 'linescan line'
                switch obj.BaselineCalcMethod
                    case 'minimum region'
                        traceMedianWidth = obj.BaselineMinRegionFilterWidth;
                        regionSize = obj.BaselineMinimumRegionWidth;
                        [baseline, sd] = min_region_baseline(input, regionSize, traceMedianWidth);
                    case 'manual region'
                        start = max(1, obj.BaselineManualRegionStart);
                        stop = min(height, obj.BaselineManualRegionStop);
                        baseline = mean(input(start:stop, :), 1);
                        sd = std(double(input(start:stop, :)), 0, 1);
                end
                medianWidth = obj.BaselineLinescanMedianFilterWidth;
                if medianWidth > 1
                    baseline = medfilt1(baseline, medianWidth);
                end
                result = {baseline, sd};
                
            case '2-d'
                if (def.ApplyMask)
                    mask = def.ImageMask;
                else
                    mask = ones(height, width);
                end
                if frameCount == 1
                    def.Baseline = result;
                    def.BaselineStandardDeviation = zeros(height, width);
                    return;
                end
                switch obj.BaselineCalcMethod
                    case 'minimum region'
                        traceMedianWidth = obj.BaselineMinRegionFilterWidth;
                        regionSize = obj.BaselineMinRegionFilterWidth;
                        [baseline, sd] = min_region_baseline(input, regionSize, traceMedianWidth, mask);
                    case 'manual region'
                        start = max(1, obj.BaselineManualRegionStart);
                        stop = min(frameCount, obj.BaselineManualRegionStop);
                        baseline = mean(input(:,:,start:stop), 3);
                        sd = std(double(input(:,:,start:stop)), 1, 3);
                    case 'sliding region'
                        % Current method ignores entire margin
                        width = max(1, obj.BaselineSlidingRegionWidth);
                        baseline = double(input);
                        sd = zeros(size(input));
                        offset = floor(width/2)-1;
                        stop = frameCount - width + 1;
                        for i = 1:stop
                            regionFrameStart = i;
                            regionFrameEnd = regionFrameStart+width-1;
                            baseline(:,:,i+offset) = mean(input(:,:,regionFrameStart:regionFrameEnd), 3);
                            sd(:,:,i+offset) = std(double(input(:,:,regionFrameStart:regionFrameEnd)), 1, 3);
                            if (def.CancelRequested)
                                return
                            end
                        end
                    case 'rolling ball'
                        % Current method ignores entire margin
                        radius = round(max(1, obj.BaselineRollingRadius));
                        baseline = double(input);
                        sd = zeros(size(input));
                        structEl = strel('ball', radius, radius);
                        for i = 1:frameCount
                            baseline(:,:,i) = imopen(input(:,:,i), structEl);
                            % No standard deviation for this method
                            if (def.CancelRequested)
                                return
                            end
                        end
                end
                medianKernel = obj.BaselineKernel;
                kernel = medianKernel.StructuringElement.getnhood;
                total = sum(kernel(:));
                if total > 1
                    dim = size(baseline);
                    if length(dim) < 3
                        dim(3) = 1;
                    end
                    for i = 1:dim(3)
                        baseline(:,:,i) = ordfilt2(baseline(:,:,i), ceil(total/2), kernel);
                        if (def.CancelRequested)
                            return
                        end
                    end
                end
                result = {baseline, sd};
                
            case 'constant'
        end
        switch obj.BaselineResultAction
            case 'calculate only'
            case 'baseline image'
                result{3} = baseline;
            case 'std dev image'
                result{3} = sd;
        end
        
    case 'baseline normalization'
        baseline = def.Baseline;
        dim = size(baseline);
        switch def.BaselineOperation.BaselineFormat
            case 'linescan line'
                switch obj.BaselineNormMethod
                    case 'divide'
                        result = double(input)./repmat(baseline, [height, 1, frameCount]);
                    case 'subtract'
                        result = double(input) - repmat(baseline, [height, 1, frameCount]);
                    case 'divide - 1'
                        result = double(input)./repmat(baseline, [height, 1, frameCount]) - 1;
                    case 'z score'
                        sd = def.BaselineStandardDeviation;
                        result = double(input) - repmat(baseline, [height, 1, frameCount]);
                        result = result./repmat(sd, [height, 1, frameCount]);
                end
                
            case '2-d'
                if length(dim) == 2
                    baseline = repmat(baseline, [1, 1, frameCount]);
                end
                switch obj.BaselineNormMethod
                    case 'divide'
                        result = double(input)./baseline;
                    case 'subtract'
                        result = double(input) - baseline;
                    case 'divide - 1'
                        result = double(input)./baseline - 1;
                    case 'z score'
                        sd = def.BaselineStandardDeviation;
                        if length(dim) == 2
                            sd = repmat(sd, [1, 1, frameCount]);
                        end
                        result = double(input) - baseline;
                        result = result./sd;
                end
                
            case 'constant'
        end
        
    case 'threshold'
        if strcmp(obj.ThresholdType, 'fixed value')
            value = obj.ThresholdValue;
            result = input > value;
            return
        end
        
        baseline = def.Baseline;
        switch def.BaselineOperation.BaselineFormat
            case 'linescan line'
                switch obj.ThresholdType
                    case 'baseline st dev'
                        value = obj.ThresholdValue;
                        sd = def.BaselineStandardDeviation;
                        threshold = repmat(baseline + value*sd, [height, 1]);
                        result = input >= threshold;
                        
                    case 'baseline percent'
                        value = obj.ThresholdValue;
                        threshold = repmat(baseline + (value/100)*baseline, [height, 1]);
                        result = input >= threshold;
                end
                
            case '2-d'
                switch obj.ThresholdType
                    case 'baseline st dev'
                        value = obj.ThresholdValue;
                        sd = def.BaselineStandardDeviation;
                        threshold = repmat(baseline + value*sd, [1, 1, frameCount]);
                        result = input >= threshold;
                        
                    case 'baseline percent'
                        value = obj.ThresholdValue;
                        threshold = repmat(baseline + (value/100)*baseline, [1, 1, frameCount]);
                        result = input >= threshold;
                end
                if (def.ApplyMask)
                    mask = def.ImageMask;
                    for i = 1:frameCount
                        result(:,:,i) = result(:,:,i) .* mask;
                        if (def.CancelRequested)
                            return
                        end
                    end
                end
        end
        
    case 'morph'
        kernel = obj.MorphKernel;
        switch obj.MorphType
            case 'dilate'
                result = input;
                for i = 1:frameCount;
                    result(:,:,i) = imdilate(input(:,:,i), kernel.StructuringElement);
                    if (def.CancelRequested)
                        return
                    end
                end
            case 'erode'
                result = input;
                for i = 1:frameCount;
                    result(:,:,i) = imerode(input(:,:,i), kernel.StructuringElement);
                    if (def.CancelRequested)
                        return
                    end
                end
            case 'open'
                result = input;
                for i = 1:frameCount;
                    result(:,:,i) = imopen(input(:,:,i), kernel.StructuringElement);
                    if (def.CancelRequested)
                        return
                    end
                end
            case 'close'
                result = input;
                for i = 1:frameCount;
                    result(:,:,i) = imclose(input(:,:,i), kernel.StructuringElement);
                    if (def.CancelRequested)
                        return
                    end
                end
            case 'dilate with threshold'
                result = input;
                if isempty(obj.MorphThresholdActual)
                    obj.CalculateMorphThreshold()
                end
                for i = 1:frameCount;
                    result(:,:,i) = dilate_threshold(input(:,:,i), kernel.StructuringElement, obj.MorphThresholdActual);
                    if (def.CancelRequested)
                        return
                    end
                end
            case 'erode with threshold'
                result = input;
                for i = 1:frameCount;
                    result(:,:,i) = erode_threshold(input(:,:,i), kernel.StructuringElement, obj.MorphThresholdActual);
                    if (def.CancelRequested)
                        return
                    end
                end
        end
        
    case 'swap dimensions'
        dimensions = obj.SwapDimensions;
        switch (dimensions)
            case 'y,z'
                result = permute(input, [3 2 1]);
        end
        
    case 'collapse dimension'
        dimension = obj.CollapseDimension;
        switch (dimension)
            case 'z'
                result = mean(input, 3);
        end
        
    case 'convert to ca'
        switch obj.ConvertCaMethod
            case 'resting'
                Kd = obj.ConvertCaKd;
                kDOverRestPlusOne = (Kd/obj.ConvertCaRestingConc) + 1.0;
                result = (Kd * input)./(kDOverRestPlusOne - input);
            case 'fmin-fmax'
                Kd = obj.ConvertCaKd;
                fMax = obj.ConvertCaFMax;
                Rf = obj.ConvertCaRf;
                
                result = input;
                result(result > fMax) = fMax - 1; % saturate to prevent weird formula behavior
                result = result/fMax;
                result = (result - 1/Rf)./(1 - result) * Kd;
        end
        
    otherwise
        siman.Debug.ReportProblem(['Unknown operation to perform: ' type]);
        result = input; % just to keep from crashing
end
obj.LastResult = result;