classdef ROI < siman.Notifier
    % Must be a subclass of handle
    properties
        Type = 'rectangle';
        MatlabROIHandle;
        ReferencePlot;
    end
    properties (GetObservable, SetObservable)
        IsSelected = false;
        CurrentPosition
    end
    properties (Transient)
        PositionCallbackID
    end
    properties (Dependent)
        Thumbnail
        Name
    end
    properties (Constant)
        TypeOptions = {'rectangle', 'ellipse', 'point', 'polygon', 'freehand'};
    end
    
    events
        RegionChange
    end
    
    methods
        function thumbnail = get.Thumbnail(obj)
            thumbnail = obj.GetThumbnail(obj);
        end
        function name = get.Name(obj)
            name = obj.GetName();
        end
    end
    
    methods
        function obj = ROI(type, plot)
            if nargin > 1
                obj.Type = type;
                obj.ReferencePlot = plot;
            end
        end
        
        function InitROIVisual(obj)
            ax = obj.ReferencePlot.AxesID;
            switch obj.Type
                case 'rectangle'
                    h = imrect(ax);
                case 'ellipse'
                    h = imellipse(ax);
                case 'point'
                    h = impoint(ax);
                case 'polygon'
                    h = impoly(ax);
                case 'freehand'
                    h = imfreehand(ax);
            end
            obj.PositionCallbackID = addNewPositionCallback(h,@(p)OnPositionChanged(obj, p));
            obj.MatlabROIHandle = h;
        end
        
        function OnPositionChanged(obj, pos)
            obj.CurrentPosition = pos;
            obj.FireRegionChange();
        end
        
        function mask = GetMask(obj)
            plot = obj.ReferencePlot;
            h = plot.ImageID;
            maskWithinCutout = obj.MatlabROIHandle.createMask(h);
            imageSize = plot.ImageSize;
            xRange = round(plot.ViewImageXRange);
            yRange = round(plot.ViewImageYRange);
            mask = false(imageSize(1), imageSize(2));
            mask(yRange(1):yRange(2), xRange(1):xRange(2)) = maskWithinCutout;
        end
        
        function data = GetExportData(obj)
            data = GetZTrace(obj);
        end
        
        function name = GetName(obj)
            name = obj.Type;
            pos = round(obj.CurrentPosition);
            if ~isempty(pos)
                name = [name ' ' num2str(pos(2)) ', ' num2str(pos(1))];
            end
        end
        
        function thumbnail = GetThumbnail(obj)
            thumbnail = []; %stub
        end
        
        function ztrace = GetZTrace(obj)
            plot = obj.ReferencePlot;
            data = plot.ProcessedData;
            if isempty(data)
                ztrace = [];
                return;
            end
            mask = obj.GetMask();
            dim = size(data);
            [x, y] = xyindexes(find(mask), dim);
            if length(dim) == 2
                frameCount = 1;
            else
                frameCount = dim(3);
            end
            %frameSize = dim(1) * dim(2);
            %data = reshape(data, frameSize, frameCount);
            pixelCount = length(x);
            traces = zeros(pixelCount, frameCount);
            for i = 1:pixelCount
                traces(i, :) = data(y(i), x(i), :);
            end
            ztrace = mean(traces);
        end
        
        function FireRegionChange(obj)
            notify(obj, 'RegionChange');
            %figure(2);
            %imagesc(obj.GetMask);
            %plot(obj.GetZTrace());
        end
        
        function text = GetSummary(obj)
            text = {};
            switch obj.Type
                case {'rect', 'ellipse', 'polygon', 'freehand'}
                    
                case {'point'}
            end
        end
        
        function delete(obj)
            if ~isempty(obj.MatlabROIHandle)
                delete(obj.MatlabROIHandle);
            end
        end
    end
    
    methods(Access = protected)
        % Override copyElement method:
        function cpObj = copyElement(obj)
            % Make a shallow copy of all properties
            cpObj = copyElement@matlab.mixin.Copyable(obj);
            % Make a deep copy of the collection
            for i = 1:length(cpObj.Collection)
                cpObj.Collection{i} = copy(cpObj.Collection{i});
            end
        end
    end
end
