classdef CellDefinition < siman.SimanObject
    % Must be a subclass of handle
    properties (Constant)
        OrientationOptions = {'linescan', '2-d'};
        TypeOptions = {'none', 'manual', 'threshold'};
        ThresholdTypeOptions = {'image units', 'percentile'};
    end
    properties (SetObservable, GetObservable)
        Orientation = 'linescan';
        Type = 'threshold';
    end
    properties (SetObservable, GetObservable)
        MinSize = 10;
        MaxSize = Inf;
        ManualLowerEdge = 1;
        ManualUpperEdge = Inf;
        AutomaticLowerEdge
        AutomaticUpperEdge
        ThresholdType = 'percentile';
        Threshold_DataUnits = 10;
        Threshold_Percentile = 90;
        Threshold_MedianFilter = 9;
        ConnectivityType = '8';
    end
    properties (Transient = true)
        DataSource          % Actual source of cell 
        NeedsUpdating = true;
        ErrorMessage
        IsValid
        Threshold_Actual
    end
    properties
        LowerEdge
        UpperEdge
        CellMask
        Outlines
    end
    properties (Dependent)
        NoCellDefined
    end
    
    events
        CellDefinitionChange
        CellMaskUpdated
        MethodLayoutChange
    end
    
    methods % Properties
        function val = get.NoCellDefined(obj)
            val = strcmp(obj.Type, 'none');
        end
        function set.NeedsUpdating(obj, value)
            obj.NeedsUpdating = value;
            if value
                notify(obj, 'CellDefinitionChange');
            end
        end
        function result = get.IsValid(obj)
            if obj.NeedsUpdating
                result = false;
                return
            end
            result = obj.IsValid;
        end
        function set.DataSource(obj, val)
            obj.DataSource = val;
            obj.QueueNeedsUpdating();
        end
    end
    
    methods
        % Constructor
        function obj = CellDefinition(source)
            obj@siman.SimanObject();
            
            obj.DataSource = source;
            obj.SelfListener = addlistener(obj, 'PropertyChanged', @obj.OnPropertyChanged);
        end
        
        function SetErrorState(obj, message)
            obj.ErrorMessage = message;
            obj.IsValid = isempty(message);
        end
        
        function OnPropertyChanged(obj, src, evnt)
            switch evnt.PropertyName
                otherwise
                    obj.QueueNeedsUpdating();
            end
            % Indicate need for layout change
            switch evnt.PropertyName
                case {'Orientation', 'Type'}
                    notify(obj, 'MethodLayoutChange');
                case 'ThresholdType'
                    notify(obj, 'MethodLayoutChange');
            end
        end
        
        function QueueNeedsUpdating(obj)
            obj.NeedsUpdating = true;
        end
        
        function ApplyChanges(obj)
            if obj.NeedsUpdating
                obj.SetErrorState('');
                obj.DetermineCellMask();
                obj.NeedsUpdating = false;
                notify(obj, 'CellMaskUpdated');
            end
        end
        
        function tf = HasPendingChanges(obj)
            tf = obj.NeedsUpdating;
        end
        
        % Handle enabling of controls tied to a local property
        function enabling = GetEnabling(obj, id, prop, context)
            enabling = true;
        end
        
        %function options = GetTypeList(obj)
        %    switch obj.Orientation
        %        case 'linescan'
        %            options = {'none', 'manual', 'automatic'};
        %        case '2-d'
        %            options = {'none', 'manual', 'threshold', 'automatic'};
        %    end
        %end
    end
end